/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "pluginmanager.h"
#include "Interfaces/plugininterface.h"

#include <QtCore>
#include <QtDebug>

class PluginManagerPrivate {
public:
    bool check(const QString &path);

public:
    QHash<QString, QVariant> names;
    QHash<QString, QVariant> versions;
    QHash<QString, QVariant> author;
    QHash<QString, QVariant> schema;
    QHash<QString, QVariantList> dependencies;

public:
    QHash<QString, QPluginLoader *> loaders;
};

bool PluginManagerPrivate::check(const QString &path)
{
    bool status = true;

    foreach (QVariant item, this->dependencies.value(path)) {

        QVariantMap mitem = item.toMap();
        QVariant na_mitem = mitem.value("name");
        QVariant ve_mitem = mitem.value("version");
        QString key = this->names.key(na_mitem);

        if (!this->names.values().contains(na_mitem)) {
            qWarning() << Q_FUNC_INFO << "  Missing dependency:" << na_mitem.toString() << "for plugin" << path;
            status = false;
            continue;
        }

        if (this->versions.value(key) != ve_mitem) {
            qWarning() << Q_FUNC_INFO << "    Version mismatch:" << na_mitem.toString() << "version"
                       << this->versions.value(this->names.key(na_mitem)).toString() << "but" << ve_mitem.toString()
                       << "required for plugin" << path;
            status = false;
            continue;
        }

        if (!check(key)) {
            qWarning() << Q_FUNC_INFO << "Corrupted dependency:" << na_mitem.toString() << "for plugin" << path;
            status = false;
            continue;
        }
    }

    return status;
}

PluginManager *PluginManager::instance(void)
{
    if (!s_instance) s_instance = new PluginManager;

    return s_instance;
}

void PluginManager::initialize(void)
{
    QDir path = QDir(qApp->applicationDirPath());
    path.cdUp();
    path.cd("plugins/bin");

    QStringList paths;
    paths << path.absolutePath() << "/usr/share/ckvsoft/qrk/plugins";
    path.setPath(qApp->applicationDirPath());
    path.cd("plugins");
    paths << path.absolutePath();
    path.setPath(qApp->applicationDirPath());
    path.cdUp();
    path.cd("plugins");
    paths << path.absolutePath();

    foreach (const QString &str, paths) {
        if (this->instance()->plugins().count() > 0) break;
        qInfo() << "Plugin search Path=" << str;
        path = QDir(str);
        foreach (QFileInfo info, path.entryInfoList(QDir::Files | QDir::NoDotAndDotDot)) {
            this->scan(info.absoluteFilePath().trimmed());
            this->load(info.absoluteFilePath().trimmed());
        }
    }
}

void PluginManager::uninitialize(void)
{
    foreach (const QString &path, d->loaders.keys())
        this->unload(path);
}

void PluginManager::scan(const QString &path)
{
    const QString schemaversion = "1";
    if (!QLibrary::isLibrary(path)) return;

    QPluginLoader *loader = new QPluginLoader(path);

    QVariant name = loader->metaData().value("MetaData").toObject().value("name").toVariant();
    QVariant schema = loader->metaData().value("MetaData").toObject().value("schema").toVariant();
    if (!schema.isValid() || schema.toString().compare(schemaversion) != 0) return;

    QList<QVariant> values = d->names.values();

    if (qobject_cast<PluginInterface *>(loader->instance())) {
        if (!values.contains(name)) {
            d->names.insert(path, loader->metaData().value("MetaData").toObject().value("name").toVariant());
            d->versions.insert(path, loader->metaData().value("MetaData").toObject().value("version").toVariant());
            d->author.insert(path, loader->metaData().value("MetaData").toObject().value("author").toVariant());
            d->schema.insert(path, loader->metaData().value("MetaData").toObject().value("schema").toVariant());
            d->dependencies.insert(
                path, loader->metaData().value("MetaData").toObject().value("dependencies").toArray().toVariantList());
        }
    } else {
        qWarning() << "can't load: " << path << " Error: " << loader->errorString();
    }

    delete loader;
}

void PluginManager::load(const QString &path)
{
    if (!QLibrary::isLibrary(path)) return;

    if (!d->check(path)) return;

    QPluginLoader *loader = new QPluginLoader(path);
    if (d->names.value(path) == loader->metaData().value("MetaData").toObject().value("name").toVariant()) {
        if (qobject_cast<PluginInterface *>(loader->instance())) {
            d->loaders.insert(path, loader);
            if (!qobject_cast<PluginInterface *>(loader->instance())->initialize())
                qWarning() << "can't initialize: " << path;
        } else {
            qWarning() << "can't load: " << path << " Error: " << loader->errorString();
            delete loader;
        }
    } else {
        delete loader;
    }
}

void PluginManager::unload(const QString &path)
{
    QPluginLoader *loader = d->loaders.value(path);

    if (!qobject_cast<PluginInterface *>(loader->instance())->deinitialize())
        qWarning() << "can't deinitialize: " << path;

    if (loader->unload()) {
        d->loaders.remove(path);
        delete loader;
    }
}

QStringList PluginManager::plugins(void)
{
    return d->loaders.keys();
}

QObject *PluginManager::getObjectByName(QString name)
{
    if (name.contains("*")) name = getHashValue(name);
    if (!d->names.key(QVariant(name)).isNull()) {
        if (d->loaders.value(d->names.key(QVariant(name)))) {
            return d->loaders.value(d->names.key(QVariant(name)))->instance();
        }
    }
    return Q_NULLPTR;
}

QString PluginManager::getNameByPath(QString path)
{
    return d->names.value(path).toString();
}

QString PluginManager::getHashValue(QString strVal)
{
    strVal.replace("*", "");
    QList<QVariant> values = d->names.values();
    foreach (QVariant value, values) {
        if (value.toString().contains(strVal)) return value.toString();
    }
    return strVal;
}

PluginManager::PluginManager(void)
    : d(new PluginManagerPrivate)
{
}

PluginManager::~PluginManager(void)
{
    delete d;

    d = Q_NULLPTR;
}

PluginManager *PluginManager::s_instance = Q_NULLPTR;
