#include "backupworker.h"
#include "abstractdatabase.h"
#include "rbac/crypto.h"

#include "JlCompress.h"
#include "backup.h"
#include "csqlquery.h"
#include "database.h"
#include "qrksettings.h"

#include <QApplication>
#include <QElapsedTimer>
#include <QSqlDatabase>
#include <QSqlError>
#include <QStandardPaths>

#include <QDebug>

BackupWorker::BackupWorker(
    const QStringList &fullpathFilelist, const QString &backupDir, const QString &tempname, const QString &confname)
    : m_fullpathFilelist(fullpathFilelist)
    , m_backupDir(backupDir)
    , m_tempname(tempname)
    , m_confname(confname)
{
}

BackupWorker::~BackupWorker()
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << "processBackup completed successfully.";
}

void BackupWorker::run()
{
    qInfo() << "Function Name: " << Q_FUNC_INFO << "copyFiles started.";
    bool copySuccess = copyDatabaseFiles(m_fullpathFilelist, m_backupDir, m_tempname);
    if (!copySuccess) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Datacopy failed!";
        emit finishedCopyFiles();
        return;
    }
    emit finishedCopyFiles();

    qInfo() << "Function Name: " << Q_FUNC_INFO << "processBackup started.";

    QElapsedTimer t;
    t.start();

    QString outfile
        = QString("%1/data_%2.zip").arg(m_backupDir).arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));

    QStringList fullpathFilelist;
    fullpathFilelist << QString(m_backupDir + "/qrktmp/" + m_tempname + "/%1-%2%3.db")
                            .arg(QDate::currentDate().year())
                            .arg(qApp->applicationName())
                            .arg(m_confname);

    QString filename;
    exportTables(m_backupDir + "/qrktmp/" + m_tempname, m_confname, filename);
    fullpathFilelist.append(filename);

    bool ok = JlCompress::compressFiles(outfile, fullpathFilelist);
    if (!ok) qWarning() << "Function Name: " << Q_FUNC_INFO << " JlCompress::compressFile:" << ok;

    removeOldestFiles();
    QFile::remove(filename);

    qDebug() << "Function Name: " << Q_FUNC_INFO << "Total Time elapsed: " << t.elapsed() << " ms";
    removeDir(m_backupDir + "/qrktmp/" + m_tempname);
}

bool BackupWorker::removeDir(const QString &dirName)
{
    bool result = true;
    QDir dir(dirName);

    if (dir.exists(dirName)) {
        Q_FOREACH (QFileInfo info,
            dir.entryInfoList(
                QDir::NoDotAndDotDot | QDir::System | QDir::Hidden | QDir::AllDirs | QDir::Files, QDir::DirsFirst)) {
            if (info.isDir()) {
                result = removeDir(info.absoluteFilePath());
            } else {
                result = QFile::remove(info.absoluteFilePath());
            }

            if (!result) {
                return result;
            }
        }
        result = dir.rmdir(dirName);
    }
    return result;
}

void BackupWorker::exportTables(const QString &backupDir, const QString &configName, QString &filename)
{
    QDate date = QDate::currentDate();
    QrkSettings settings;
    QFileInfo fi(settings.fileName());
    QString basename = fi.baseName();
    QString databasefile = QString("%1/%2-%3.db").arg(backupDir).arg(date.year()).arg(basename);

    if (QFile::exists(databasefile)) {
        QSqlDatabase dbc;
        if (dbc.isOpen()) dbc.close();

        bool isCipher = AbstractDataBase::isSqlCipher(databasefile);

        QString driver = (isCipher && QSqlDatabase::drivers().contains("QSQLCIPHER")) ? "QSQLCIPHER" : "QSQLITE";

        dbc = QSqlDatabase::addDatabase(driver, m_tempname);
        dbc.setDatabaseName(databasefile);

        if (!dbc.open()) {
            qWarning() << "BackupWorker: Could not open temp DB" << dbc.lastError().text();
            return;
        }

        if (driver == "QSQLCIPHER") {
            QString key = Crypto::encrypt("QSQLCIPHER", SecureByteArray("Globals"));
            if (!key.isEmpty()) {
                CSqlQuery keyQuery(dbc, Q_FUNC_INFO);
                keyQuery.exec(QString("PRAGMA key = '%1';").arg(key));
            }
        }

        CSqlQuery query(dbc, Q_FUNC_INFO);
        QStringList tables = dbc.tables();

        filename = QString("%1/%2%3_dump_%4.sql")
                       .arg(backupDir)
                       .arg(qApp->applicationName())
                       .arg(configName)
                       .arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));

        QFile file(filename);
        if (!file.open(QIODevice::WriteOnly | QIODevice::Text)) {
            qWarning() << "Unable to open file for writing:" << filename;
            dbc.close();
            return;
        }

        QTextStream streamer(&file);
        streamer.setCodec("UTF-8");

        bool isNotMySql = (dbc.driverName() != "QMYSQL");

        if (isNotMySql) {
            streamer << "BEGIN TRANSACTION;\n";
        } else {
            streamer << "SET FOREIGN_KEY_CHECKS=0;\nSTART TRANSACTION;\n";
        }

        foreach (const QString &table, tables) {
            if (table == "sqlite_sequence") continue;

            qApp->processEvents();
            streamer << getCreateTable(table) << "\n";

            // Spaltennamen ermitteln
            QStringList fields;
            if (isNotMySql) {
                query.exec(QString("PRAGMA TABLE_INFO(%1)").arg(table));
                while (query.next()) {
                    fields << query.value(1).toString();
                }
            } else {
                query.exec(QString("DESCRIBE %1;").arg(table));
                while (query.next()) {
                    fields << query.value(0).toString();
                }
            }

            if (fields.isEmpty()) continue;

            // Daten exportieren
            QString selectQuery = QString("SELECT %1 FROM %2").arg(fields.join(",")).arg(table);
            query.exec(selectQuery);

            while (query.next()) {
                qApp->processEvents();
                QStringList values;

                for (int i = 0; i < fields.size(); ++i) {
                    QVariant value = query.value(i);

                    if (value.isNull()) {
                        values << "NULL";
                    } else if (value.type() == QVariant::Int || value.type() == QVariant::Double
                        || value.type() == QVariant::LongLong) {
                        values << value.toString();
                    } else if (value.type() == QVariant::Bool) {
                        values << (value.toBool() ? "1" : "0");
                    } else if (value.type() == QVariant::Date) {
                        values << QString("'%1'").arg(value.toDate().toString("yyyy-MM-dd"));
                    } else if (value.type() == QVariant::DateTime) {
                        values << QString("'%1'").arg(value.toDateTime().toString("yyyy-MM-dd HH:mm:ss"));
                    } else if (value.type() == QVariant::Time) {
                        values << QString("'%1'").arg(value.toTime().toString("HH:mm:ss"));
                    } else {
                        QString strValue = value.toString().replace("'", "''");
                        values << QString("'%1'").arg(strValue);
                    }
                }
                streamer << QString("INSERT INTO %1 VALUES(%2);\n").arg(table).arg(values.join(", "));
            }
        }

        if (isNotMySql) {
            streamer << "COMMIT;";
        } else {
            streamer << "COMMIT;\nSET FOREIGN_KEY_CHECKS=1;";
        }

        file.close();
        dbc.close();
        QSqlDatabase::removeDatabase(m_tempname); // Wichtig: Connection-Cleanup!
    }
}

void BackupWorker::removeOldestFiles()
{
    QElapsedTimer t;
    t.start();
    QrkSettings settings;
    int keep = settings.value("keepMaxBackups", -1).toInt();
    if (keep == -1) return;

    QString backupDir = QDir(
        settings.value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation)).toString())
                            .absolutePath();

    QDir dir(backupDir);
    dir.setNameFilters(QStringList() << "*.zip");
    dir.setFilter(QDir::Files);
    dir.setSorting(QDir::Time | QDir::Reversed);
    QStringList list = dir.entryList();
    while (list.size() > keep) {
        qApp->processEvents();
        QString f = list.takeFirst();
        QFile::remove(dir.absoluteFilePath(f));
        qInfo() << "Function Name: " << Q_FUNC_INFO << " Remove old backup FileName: " << f;
    }
    qDebug() << "Function Name: " << Q_FUNC_INFO << "Time elapsed: " << t.elapsed() << " ms";
}

QString BackupWorker::getCreateTable(const QString &tablename)
{
    /*
    "SHOW CREATE TABLE `users`;"
    "SELECT sql FROM sqlite_master WHERE name='users'"

    "DROP TABLE [IF EXISTS] table_name"
    */
    QString drop = QString("\nDROP TABLE IF EXISTS `%1`;\n").arg(tablename);

    QSqlDatabase dbc = Database::database();

    CSqlQuery query(dbc, Q_FUNC_INFO);
    if (dbc.driverName() == "QSQLITE") {
        query.prepare("SELECT sql FROM sqlite_master WHERE name=:tablename;");
        query.bindValue(":tablename", tablename);
        if (query.exec())
            if (query.next()) return drop + query.value("sql").toString() + ";\n";
    } else {
        if (query.exec(QString("SHOW CREATE TABLE %1;").arg(tablename)))
            if (query.next()) return drop + query.value(1).toString() + ";\n";
    }

    qDebug() << "Function Name: " << Q_FUNC_INFO << " lasterror:" << query.lastError().text();
    qDebug() << "Function Name: " << Q_FUNC_INFO << " lastquery:" << Database::getLastExecutedQuery(query);

    return "";
}

bool BackupWorker::copyDatabaseFiles(
    const QStringList &filePaths, const QString &destinationDir, const QString &tempname)
{
    QElapsedTimer t;
    t.start();

    QDir destDir(destinationDir + "/qrktmp/" + tempname);

    // Prüfen, ob das Zielverzeichnis existiert, falls nicht, erstellen
    if (!destDir.exists() && !destDir.mkpath(".")) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Can not create Destinationdir: " << destinationDir;
        return false;
    }

    for (const QString &filePath : filePaths) {
        QFileInfo fileInfo(filePath);
        if (!fileInfo.exists() || !fileInfo.isFile()) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << "File not exists:" << filePath;
            continue;
        }

        QString destFilePath = destDir.filePath(fileInfo.fileName());

        if (!QFile::copy(filePath, destFilePath)) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << "Can not copy File: " << filePath << " to "
                       << destFilePath;
            continue;
        }
        qApp->processEvents();
    }
    qDebug() << "copyDatabaseFiles Time elapsed:" << t.elapsed() << " ms";

    return true;
}
