/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */
#include "primesigncardos_53.h"
#include <QDebug>

PrimeSignCARDOS_53::PrimeSignCARDOS_53(QString device_name, bool shared)
    : PrimeSignSmartCard(device_name, shared)
{
}

PrimeSignCARDOS_53::~PrimeSignCARDOS_53()
{
}

QString PrimeSignCARDOS_53::getCardType()
{
    return QObject::tr("PrimeSign CARDOS_53");
}

QString PrimeSignCARDOS_53::getExpireInfo()
{
    return parseExpiryDate("Dezember 2027", getCardType());
}

bool PrimeSignCARDOS_53::selectDF_QES()
{
    // if (m_DF_QES_Selected) return true;

    transmit(PRIMESIGN_SELECT_DF_QES, 7);
    m_DF_QES_Selected = true;
    return true;
}

QString PrimeSignCARDOS_53::signReceipt(QString data)
{
    QString jwsDataToBeSigned = RKSignatureModule::getDataToBeSigned(data);
    QString hashValue = RKSignatureModule::HashValue(jwsDataToBeSigned);

    QByteArray ba = 0;
    ba.append(hashValue.toUtf8());
    ba = QByteArray::fromHex(ba);
    unsigned char *hash = (unsigned char *)ba.data();

    QByteArray JWS_Signature;
    ASignResponse response = signHash(hash);

    for (uint i = 0; i < response.length; i++)
        JWS_Signature[i] = response.data[i]; // (const char)response.data[i];

    if (response.length > 0) {
        JWS_Signature = JWS_Signature.toBase64(QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
    } else
        JWS_Signature.append(base64Url_encode("Sicherheitseinrichtung ausgefallen"));

    return jwsDataToBeSigned + "." + JWS_Signature;
}

// Auslesen des Signaturzertifikates
QString PrimeSignCARDOS_53::getCertificate(bool base64)
{
    /*
     1. SELECT DF_QES
     2. SELECT EF_C_X509_CH_DS
     3. READ BINARY
    */

    selectDF_QES();

    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    QByteArray certificate;
    certificate = ReadFile();

    if (base64) return certificate.toBase64();

    return certificate;
}

/*
QString PrimeSignCARDOS_53::getCertificateSerial(bool hex)
{
    unsigned char data[256];

    selectDF_QES();
    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    DWORD i = response.length;
    if (i == 0) return QString::number(0);

    while (i--) {
        data[i] = response.data[i];
    }

    QByteArray ba = 0;
    ba.append((char *)data, 256);

    int len = static_cast<unsigned int>(static_cast<unsigned char>(ba.at(14)));
    long serial = 0;
    for (int i = 0; i < len; i++) {
        serial = (serial << 8) + static_cast<unsigned int>(static_cast<unsigned char>(ba.at(15 + i)));
    }

    QString serialHex = QString::number(serial, 16).toUpper();

    // put Certificate to Database
    if (!isCertificateInDB(serialHex)) putCertificate(serialHex, getCertificate(true));

    if (hex) {
        return serialHex;
    }

    return QString::number(serial);
}
*/

QString PrimeSignCARDOS_53::getCertificateSerial(bool hex)
{
    selectDF_QES();
    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    if (response.length < 15) return QStringLiteral("0"); // Antwort zu kurz

    QByteArray ba(reinterpret_cast<const char *>(response.data), qMin(response.length, DWORD(256)));

    int len = static_cast<unsigned char>(ba.at(14));
    if (ba.size() < 15 + len) return QStringLiteral("0"); // Seriennummer zu kurz

    QByteArray serialBytes = ba.mid(15, len);

    // Führendes 0x00 entfernen (optional)
    if (serialBytes.size() > 1 && (uchar)serialBytes[0] == 0x00) serialBytes.remove(0, 1);

    QString serialHex = serialBytes.toHex().toUpper();

    if (!isCertificateInDB(serialHex)) putCertificate(serialHex, getCertificate(true));

    if (hex) return serialHex;

    // Dezimalwert als String, auch für sehr große Werte
    QString serialDec = PrimeSignSmartCard::hexToDecimalString(serialBytes);
    return serialDec;
}

ASignResponse PrimeSignCARDOS_53::signHash(const unsigned char hash[32])
{

    selectDF_QES();

    bool blocked = false;
    unsigned char cmd[14] = PRIMESIGN_CMD_VERIFY;
    buildVerifyCommand(cmd);
    if (verifyPin(blocked)) {
        unsigned char hashcmd[32 + 6] = PRIMESIGN_OS53_CMD_PUT_HASH;
        memcpy(&hashcmd[5], hash, 32);
        return PrimeSignSmartCard::transmit(hashcmd, sizeof(hashcmd));
    }

    // PIN invalid → return empty or error response
    qWarning() << "PIN verification failed – cannot sign hash";

    ASignResponse resp;
    resp.code[0] = 0x69; // "Command not allowed"
    resp.code[1] = 0x82; // Security condition not satisfied
    return resp;
}

bool PrimeSignCARDOS_53::verifyPin(bool &blocked)
{
    unsigned char cmd[14] = PRIMESIGN_CMD_VERIFY;
    buildVerifyCommand(cmd);
    bool result = PrimeSignSmartCard::verifyPinCmd(blocked, cmd);
    qInfo() << "Function Name: " << Q_FUNC_INFO << " VerifyPIN: " << result << " blocked: " << blocked;
    return result;
}
