/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "backup.h"
#include "JlCompress.h"
#include "QtCore"
#include "backupworker.h"
#include "csqlquery.h"
#include "databasemanager.h"
#include "qrksettings.h"

#include <QApplication>
#include <QDebug>
#include <QElapsedTimer>
#include <QEventLoop>
#include <QObject>
#include <QProcess>
#include <QSqlDatabase>
#include <QSqlError>
#include <QStandardPaths>
#include <QTemporaryFile>
#include <QThreadPool>

void Backup::create()
{
    QrkSettings settings;
    QString dataDir
        = settings
              .value("sqliteDataDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/data")
              .toString();
    create(dataDir);
}

void Backup::create(QString dataDir)
{
    QrkSettings settings;
    QString backupDir
        = settings
              .value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/backup")
              .toString();
    QString confname = qApp->property("configuration").toString();
    if (!confname.isEmpty()) confname = "_" + confname;

    QDir directory(dataDir);

    QStringList infileList = directory.entryList(QStringList()
            << QString("%1-%2%3.db*").arg(QDate::currentDate().year()).arg(qApp->applicationName()).arg(confname),
        QDir::Files);
    QStringList fullpathFilelist;
    foreach (QString filename, infileList) {
        fullpathFilelist.append(dataDir + "/" + filename);
    }

    QString tempname;
    {
        QTemporaryFile temp("qrk-XXXXXX.tmp");
        if (!temp.open()) {
            qWarning() << "Function:" << Q_FUNC_INFO << " Unable to create temporary file!";
            return;
        }
        QFileInfo fi(temp);
        tempname = fi.baseName();
        temp.close();
    }

    QThreadPool *threadPool = QThreadPool::globalInstance();
    BackupWorker *worker = new BackupWorker(fullpathFilelist, backupDir, tempname, confname);
    worker->setAutoDelete(true); // Important for memory management

    // Create an event loop to wait for the task to finish copy
    QEventLoop loop;

    QObject::connect(worker, &BackupWorker::finishedCopyFiles, &loop, &QEventLoop::quit);

    // Submit the task to the thread pool
    threadPool->start(worker);

    loop.exec();
}

void Backup::restore(QString filename, bool restart, bool rename)
{
    if (restart) DatabaseManager::removeCurrentThread("CN");

    create();

    QrkSettings settings;
    QString dataDir
        = settings
              .value("sqliteDataDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/data")
              .toString();
    QString backupDir
        = settings
              .value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/backup")
              .toString();

    QString zipfile = QString("%1/%2").arg(backupDir).arg(filename);
    QStringList files = JlCompress::getFileList(zipfile);
    files = JlCompress::extractFiles(zipfile, files, dataDir);

    if (rename) {
        QString year = QString::number(QDate::fromString(filename.mid(5, 4), "yyyy").year());
        QString currentyear = QString::number(QDate::currentDate().year());
        if (year.compare(currentyear) != 0) {
            foreach (QString item, files) {
                QFile f(item);
                QString newname = item.replace(year, currentyear);
                f.remove(newname);
                f.rename(newname);
            }
        }
    }

    if (files.isEmpty()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " JlCompress::extractFiles: none, filename: " << filename
                   << " zipfile: " << zipfile;
    } else if (restart) {
        // Spawn a new instance of myApplication:
        QString app = QApplication::applicationFilePath();
        QStringList arguments = QApplication::arguments();
        arguments << "-r";
        QString wd = QDir::currentPath();
        QProcess::startDetached(app, arguments, wd);
        QApplication::exit();
    }
}

void Backup::pakLogFile()
{

    QrkSettings settings;
    QString confname = qApp->property("configuration").toString();
    if (!confname.isEmpty()) confname = "_" + confname;

    QString backupDir = QDir(
        settings.value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation)).toString())
                            .absolutePath();
    QString infile = QString("%1").arg(
        QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + QString("/qrk.log").arg(confname));
    QString outfile = QString("%1/qrk%2_log_%3.zip")
                          .arg(backupDir)
                          .arg(confname)
                          .arg(QDateTime::currentDateTime().toString("yyyyMMdd-hhmmss"));

    bool ok = JlCompress::compressFile(outfile, infile);
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " JlCompress::compressFile:" << ok;
        return;
    }

    QFile(infile).remove();
}

void Backup::cleanBackupTempDirectory()
{
    QrkSettings settings;
    QString backupDir
        = settings
              .value("backupDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/backup")
              .toString();
    qInfo() << "Function Name: " << Q_FUNC_INFO
            << " Remove Backup Tempdirectory: " << BackupWorker::removeDir(backupDir + "/qrktmp");
}
