/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "export.h"

#include "3rdparty/ckvsoft/csqlquery.h"
#include "singleton/Singleton.h"
#include "singleton/spreadsignal.h"

#include "RK/rk_signaturemodule.h"
#include "database.h"
#include "defines.h"
#include "qrksettings.h"
#include "utils/utils.h"

#include <QDateTime>
#include <QDebug>
#include <QDir>
#include <QFile>
#include <QJsonArray>
#include <QJsonDocument>
#include <QSqlDatabase>

#include <QSqlRecord>
#include <QString>

Export::Export(QObject *parent)
    : QObject(parent)
{
}

Export::~Export()
{
    Spread::Instance()->setProgressBarValue(-1);
}

QJsonDocument Export::depExport(int from, int to)
{
    QJsonObject root;
    QJsonArray group;
    QJsonObject object;

    object["Belege-kompakt"] = getReceipts(from, to);
    object["Signaturzertifikat"] = "";

    group.append(object);

    root["Belege-Gruppe"] = group;

    QJsonDocument doc(root);

    return doc;
}

QJsonDocument Export::mapExport()
{
    QJsonObject root;
    QJsonObject map;

    map = RKSignatureModule::getCertificateMap();

    root["base64AESKey"] = RKSignatureModule::getPrivateTurnoverKeyBase64();
    root["certificateOrPublicKeyMap"] = map;

    QJsonDocument doc(root);

    return doc;
}

QJsonArray Export::getReceipts(int from, int to)
{
    QJsonArray receipts;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT data FROM dep WHERE receiptNum BETWEEN :from AND :to "
                  "ORDER by id");
    query.bindValue(":from", from);
    query.bindValue(":to", to);

    query.exec();

    int i = 0;
    int count = query.record().count();

    while (query.next()) {
        i++;
        Spread::Instance()->setProgressBarValue(int((float(i) / float(count)) * 100));

        receipts.append(query.value(0).toString());
    }

    return receipts;
}

bool Export::depExport(QString filename)
{
    QFile outputFile(filename);

    /* Try and open a file for output */
    outputFile.open(QIODevice::WriteOnly | QIODevice::Text);

    /* Check it opened OK */
    if (!outputFile.isOpen()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error, unable to open" << filename << "for output";
        return false;
    }

    int beginID = 1;
    int endID = getLastMonthReceiptId();

    QJsonDocument dep = depExport(beginID, endID);
    QTextStream outStreamDEP(&outputFile);
    outStreamDEP << dep.toJson();
    /* Close the file */
    outputFile.close();

    if (endID == -1) return false;

    return true;
}

bool Export::depExport(QString outputDir, QString from, QString to)
{
    QString filenameDEPExport = QDir::toNativeSeparators("%1/dep-export.json").arg(outputDir);
    QString filenameMAPExport = QDir::toNativeSeparators("%1/cryptographicMaterialContainer.json").arg(outputDir);

    QFile outputFileDEP(filenameDEPExport);
    QFile outputFileMAP(filenameMAPExport);

    /* Try and open a file for output */
    outputFileDEP.open(QIODevice::WriteOnly | QIODevice::Text);
    outputFileMAP.open(QIODevice::WriteOnly | QIODevice::Text);

    /* Check it opened OK */
    if (!outputFileDEP.isOpen()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error, unable to open" << outputFileDEP.fileName()
                   << "for output";
        return false;
    }

    /* Check it opened OK */
    if (!outputFileMAP.isOpen()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error, unable to open" << outputFileDEP.fileName()
                   << "for output";
        return false;
    }

    /* Point a QTextStream object at the file */
    QTextStream outStreamDEP(&outputFileDEP);
    QTextStream outStreamMAP(&outputFileMAP);

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT MIN(receiptNum) as begin, MAX(receiptNum) as end FROM "
                  "receipts WHERE timestamp BETWEEN :fromDate AND :toDate");
    query.bindValue(":fromDate", from);
    query.bindValue(":toDate", to);
    query.exec();

    if (query.next()) {
        QJsonDocument dep = depExport(query.value("begin").toInt(), query.value("end").toInt());
        QJsonDocument map = mapExport();
        outStreamDEP << dep.toJson();
        outStreamMAP << map.toJson();
    }

    /* Close the file */
    outputFileDEP.close();
    outputFileMAP.close();

    return true;
}

int Export::getLastMonthReceiptId()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT MAX(receiptNum) as maxID FROM receipts WHERE payedBy=:payedBy");
    query.bindValue(":payedBy", PAYED_BY_REPORT_EOM);
    query.exec();
    if (query.next()) {
        int maxID = query.value("maxID").toInt();
        query.prepare("SELECT payedBy FROM receipts WHERE receiptNum=:id");
        query.bindValue(":id", maxID + 1);
        query.exec();
        if (query.next()) {
            int type = query.value("payedBy").toInt();
            if (type == PAYED_BY_MONTH_RECEIPT || type == PAYED_BY_YEAR_RECEIPT) return maxID + 1;
        }
    }
    return -1;
}

bool Export::createBackup()
{
    QrkSettings settings;
    QString directoryname = settings.value("externalDepDirectory", "").toString();

    if (Utils::isDirectoryWritable(directoryname)) {
        QString filename
            = QDir::toNativeSeparators("%1/DEP-7-%2_backup_%3.json")
                  .arg(directoryname)
                  .arg(Database::getCashRegisterId())
                  .arg(QDateTime::currentDateTime().toString(Qt::ISODate).replace(':', "").replace('-', ""));
        if (depExport(filename)) return true;
    }

    return false;
}

bool Export::createBackup(int &counter)
{
    QVariant vValue;
    QString strValue;
    int id = Database::select_globals("DepCounter", vValue, strValue);
    if (id > 0) {
        counter = vValue.toInt();
    } else {
        Database::insert2globals("DepCounter", 3, QString());
        counter = 0;
    }

    bool ret = createBackup();
    if (ret) {
        Database::insert2globals("DepCounter", (counter == 0) ? 2 : counter - 1, QString());
    }

    return ret;
}
