/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */
#include "waiterlock.h"

#include <QSerialPort>
#include <QTime>

WaiterLock::WaiterLock(QObject *parent)
    : QObject(parent)
{
    m_waiterlockids.insert(ADDIMAT, "OUT\r\n");
}

WaiterLock::~WaiterLock()
{
    m_mutex.lock();
    m_quit = true;
    m_mutex.unlock();
}

void WaiterLock::setData(WaiterLockType type, const QString &portName, int waitTimeout)
{
    const QMutexLocker locker(&m_mutex);
    m_waiterlocktype = type;
    m_portName = portName;
    m_waitTimeout = waitTimeout;
}

void WaiterLock::run()
{
    bool currentPortNameChanged = false;

    m_mutex.lock();
    QString currentPortName;
    if (currentPortName != m_portName) {
        currentPortName = m_portName;
        currentPortNameChanged = true;
    }

    int currentWaitTimeout = m_waitTimeout;
    QString currentRespone = m_response;
    m_mutex.unlock();
    QSerialPort serial;
    serial.setFlowControl(QSerialPort::HardwareControl);

    while (!m_quit) {
        if (currentPortNameChanged) {
            serial.close();
            serial.setPortName(currentPortName);

            if (!serial.open(QIODevice::ReadWrite)) {
                emit error(tr("Can't open %1, error code %2").arg(m_portName).arg(serial.error()));
                return;
            }
        }

        if (serial.waitForReadyRead(currentWaitTimeout)) {
            // read request
            m_requestData += serial.readAll();

            if (m_requestData.endsWith(m_waiterlockids.value(m_waiterlocktype))) {
                const QString request = QString::fromUtf8(m_requestData);
                m_requestData.clear();
                if (request.compare(m_lastrequest) != 0) {
                    m_lastrequest = request;
                    emit this->request(m_lastrequest.replace(m_waiterlockids.value(m_waiterlocktype), ""));
                }
            }
        } else {
            emit timeout(tr("Wait read request timeout %1").arg(QTime::currentTime().toString()));
            m_mutex.lock();
            if (currentPortName != m_portName) {
                currentPortName = m_portName;
                currentPortNameChanged = true;
            } else {
                currentPortNameChanged = false;
            }
            currentWaitTimeout = m_waitTimeout;
            currentRespone = m_response;
            m_mutex.unlock();
        }
    }
}

QString WaiterLock::getTypeValue(WaiterLockType type)
{
    return m_waiterlockids.value(type);
}

QString WaiterLock::getKeyValue()
{
    return m_lastrequest;
}
