/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "sortfiltersqlquerymodel.h"

#include <QSqlDriver>
#include <QSqlError>
#include <QSqlField>
#include <QSqlRecord>

#include <QDebug>

SortFilterSqlQueryModel::SortFilterSqlQueryModel(QObject *parent)
    : QSqlQueryModel(parent)
{
}

void SortFilterSqlQueryModel::setQuery(const QString &query, const QSqlDatabase &db)
{
    m_queryClause = query;
    m_dbc = db;

    m_filterString.clear();
    // filterColumn.clear();
    m_filterFlags = Qt::MatchStartsWith;
    m_sortKeyColumn = -1;
    m_sortOrder = Qt::DescendingOrder;
}

void SortFilterSqlQueryModel::select()
{

    if (m_queryClause.isEmpty() || (!m_dbc.isValid())) return;

    QString query = m_queryClause;

    if (!m_filterString.isEmpty() && !m_filterColumn.isEmpty()) {
        QString whereClause;
        QString esFilterString = m_filterString;
        QString esFilterColumn = m_filterColumn;

        if (m_filterFlags == Qt::MatchExactly) {
            whereClause = "WHERE CAST(%1 as text) = %2";
        } else if (m_filterFlags == Qt::MatchStartsWith) {
            whereClause = "WHERE CAST(%1 as text) LIKE %2";
            esFilterString.append("%");
        } else if (m_filterFlags == Qt::MatchContains) {
            whereClause = "WHERE CAST(%1 as text) LIKE %2";
            esFilterString.append("%");
            esFilterString.prepend("%");
        } else if (m_filterFlags == Qt::MatchEndsWith) {
            whereClause = "WHERE CAST(%1 as text) LIKE %2";
            esFilterString.prepend("%");
        } else {
            return;
        } // unhandled filterflag

        QSqlDriver *driver = m_dbc.driver();
        esFilterColumn = driver->escapeIdentifier(m_filterColumn, QSqlDriver::FieldName);
        QSqlField field;
        field.setType(QVariant::String);
        field.setValue(esFilterString);
        esFilterString = driver->formatValue(field);

        whereClause = whereClause.arg(esFilterColumn).arg(esFilterString);
        query.append(" " + whereClause);
    }

    if (m_sortKeyColumn >= 0) {
        QString orderClause;
        // orderClause = "ORDER BY " + QString::number(m_sortKeyColumn+1) + " " +
        // ((m_sortOrder == Qt::AscendingOrder) ? "ASC" : "DESC");
        orderClause = "ORDER BY LENGTH(" + QString::number(m_sortKeyColumn + 1) + "),"
            + QString::number(m_sortKeyColumn + 1) + " " + ((m_sortOrder == Qt::AscendingOrder) ? "ASC" : "DESC");
        query.append(" " + orderClause);
    }

    QSqlQueryModel::setQuery(query, m_dbc);

    qDebug() << "Function Name: " << Q_FUNC_INFO << "query: " << query;
    if (QSqlQueryModel::lastError().isValid()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "lasterror: " << QSqlQueryModel::lastError().text();
    }
}

void SortFilterSqlQueryModel::setSort(int column, Qt::SortOrder order)
{
    m_sortKeyColumn = column;
    m_sortOrder = order;
    SortFilterSqlQueryModel::setFilterColumn(this->record().fieldName(column));
    emit sortChanged();
}

void SortFilterSqlQueryModel::sort(int column, Qt::SortOrder order)
{
    if ((m_sortKeyColumn != column) || (m_sortOrder != order)) {
        setSort(column, order);
        select();
    }
}

void SortFilterSqlQueryModel::setFilterColumn(const QString &column)
{
    m_filterColumn = column;
}

void SortFilterSqlQueryModel::setFilter(const QString &filter)
{
    m_filterString = filter;
}

void SortFilterSqlQueryModel::setFilterFlags(const Qt::MatchFlag flags)
{
    m_filterFlags = flags;
}

void SortFilterSqlQueryModel::filter(const QString &filter)
{
    if (m_filterString != filter) {
        setFilter(filter);
        select();
    }
}

QString SortFilterSqlQueryModel::getFilterColumnName()
{
    return this->headerData(m_sortKeyColumn, Qt::Horizontal).toString();
}

int SortFilterSqlQueryModel::getFilterColumnId()
{
    return m_sortKeyColumn;
}
