/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "couponinoutdialog.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "coupondialog.h"
#include "coupongenerator.h"
#include "couponprinter.h"
#include "defines.h"
#include "givendialog.h"
#include "qrkpaymentdialog.h"
#include "qrkprogress.h"
#include "qrksettings.h"
#include "qrktimedmessagebox.h"
#include "receiptitemmodel.h"
#include "reports.h"
// #include "3rdparty/ckvsoft/rbac/crypto.h"
// #include "3rdparty/profeatures/profeatures.h"
#include "database.h"
// #include "qrkjournal.h"
#include "qrksettings.h"
#include "ui_couponinoutdialog.h"

#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlRelationalTableModel>

#include <QJsonArray>
#include <QJsonObject>

#include <QDebug>

CouponInOutDialog::CouponInOutDialog(QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::CouponInOutDialog)
{
    ui->setupUi(this);
    ui->dateTimeEdit->setDateTime(QDateTime::currentDateTime());

    ui->typeComboBox->addItem("🗂 " + tr("Mehrzweck Gutschein"), COUPON_TYPE::TYPE_MULTI);
    ui->typeComboBox->addItem("📄 " + tr("Einzweck Gutschein"), COUPON_TYPE::TYPE_SINGLE);

    ui->grossLabel->setText(tr("Brutto Betrag in %1").arg(Database::getCurrency()));

    ui->infoTextLabel->setText("");

    ui->taxlabel->setHidden(ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI);
    ui->taxcomboBox->setHidden(ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI);

    ui->customerEdit->setHidden(true);
    ui->customerLabel->setHidden(true);

    QRegExp rx("^?(?:0|[1-9][0-9]*),?[0-9][0-9]$");
    QValidator *doubleVal = new QRegExpValidator(rx, this);
    ui->grossEdit->setValidator(doubleVal);

    connect(ui->saveAndprintButton, &QPushButton::clicked, this, &CouponInOutDialog::createReceiptAndPrint);
    connect(ui->savePushButton, &QPushButton::clicked, this,
        static_cast<bool (CouponInOutDialog::*)()>(&CouponInOutDialog::createReceipt));
    connect(ui->cancelPushButton, &QrkPushButton::clicked, this, &CouponInOutDialog::close);
    connect(ui->grossEdit, &QLineEdit::textChanged, this, &CouponInOutDialog::grossEditChanged);
    connect(ui->typeComboBox, &QComboBox::currentTextChanged, this, &CouponInOutDialog::grossEditChanged);

    connect(ui->typeComboBox, &QComboBox::currentTextChanged, this, [this]() {
        ui->taxlabel->setHidden(ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI);
        ui->taxcomboBox->setHidden(ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI);
    });


    QSqlDatabase dbc = Database::database();

    m_taxModel = new QSqlRelationalTableModel(this, dbc);
    QString q = QString("SELECT id, tax FROM taxtypes WHERE taxlocation='%1'").arg(Database::getTaxLocation());
    m_taxModel->setQuery(q, dbc);
    ui->taxcomboBox->setModel(m_taxModel);
    ui->taxcomboBox->setModelColumn(1); // show tax
    ui->taxcomboBox->setCurrentIndex(0);

    CSqlQuery query(Database::database("COUPON"), Q_FUNC_INFO);
    query.prepare("SELECT timestamp FROM coupon ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) ui->dateTimeEdit->setMinimumDateTime(query.value("timestamp").toDateTime());
    readSettings();
}

CouponInOutDialog::~CouponInOutDialog()
{
    writeSettings();
    delete ui;
}

void CouponInOutDialog::setMinimumDateTime(QString datetimeIsoString)
{
    ui->dateTimeEdit->setMinimumDateTime(QDateTime::fromString(datetimeIsoString, Qt::ISODate));
}

void CouponInOutDialog::grossEditChanged()
{
    if (QLocale().toDouble(ui->grossEdit->text()) > 0.00) {
        ui->savePushButton->setDisabled(false);
        ui->saveAndprintButton->setDisabled(false);
    } else {
        ui->savePushButton->setDisabled(true);
        ui->saveAndprintButton->setDisabled(true);
    }
    ui->infoTextLabel->setText("");
}


bool CouponInOutDialog::createReceipt()
{
    return createReceipt(false);
}
bool CouponInOutDialog::createReceiptAndPrint()
{
    return createReceipt(true);
}

bool CouponInOutDialog::createReceipt(bool printing)
{

    /*
     * {"receipt":[
     *  {"customertext": "Customer Text",
     *   "payedBy": "0",
     *   "items":[
     *     { "count": "3", "name": "Kupplung", "gross": "122,70", "tax": "20" },
     *     { "count": "1", "name": "Bremsbeläge", "gross": "32,30", "tax": "10" },
     *     { "count": "2", "name": "Benzinschlauch", "gross": "17,80",    QSqlDatabase dbc = Database::database();

    m_taxModel = new QSqlRelationalTableModel(this, dbc);
    QString q = QString("SELECT id, tax FROM taxtypes WHERE taxlocation='%1'").arg(Database::getTaxLocation());
    m_taxModel->setQuery(q, dbc);
    ui->taxComboBox->setModel(m_taxModel);
    ui->taxComboBox->setModelColumn(1); // show tax
    ui->taxComboBox->setCurrentIndex(0);
 "tax": "20"
     * }, { "count": "1", "name": "Ölfilter", "gross": "104,50", "tax": "13" }
     *    ]
     *   }
     * ]}
     */


    ReceiptItemModel receiptitem;

    QBCMath sum(ui->grossEdit->text().toDouble());
    sum.round(2);

    QRKPaymentDialog question(sum, false, this);
    int code = question.exec();
    if (code == QDialog::Rejected) return false;

    QJsonObject mainitem;
    QJsonArray items;

    QDateTime invoiceTime = QDateTime::currentDateTime();

    int payedBy = 0;

    if (code == QRKPaymentDialog::CASHRECEIPT_TICKET)
        payedBy = PAYED_BY_CASH;
    else if (code == QRKPaymentDialog::CREDITCARD_TICKET)
        payedBy = PAYED_BY_CREDITCARD;
    else if (code == QRKPaymentDialog::DEBITCARD_TICKET)
        payedBy = PAYED_BY_DEBITCARD;
    else if (code == QRKPaymentDialog::PRIVATE_TICKET)
        payedBy = PAYED_BY_PRIVATE;
    else if (code == QRKPaymentDialog::EMPLOYEE_TICKET)
        payedBy = PAYED_BY_EMPLOYEE;
    else if (code == QRKPaymentDialog::ADVERTISING_TICKET)
        payedBy = PAYED_BY_ADVERTISING;


    CouponGenerator generator;
    QString couponcode = generator.generateUniqueCouponCode(10);

    mainitem["payedBy"] = QString::number(payedBy);
    mainitem["visible"] = 0;
    mainitem["group"] = 1;

    mainitem["customertext"] = ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI
        ? QString("Mehrzweck Gutschein %1").arg(couponcode)
        : QString("Einzweck Gutschein %1").arg(couponcode);

    QJsonObject item;
    QBCMath tax(ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI
            ? 0.00
            : ui->taxcomboBox->currentText().toDouble());
    tax.round(2);
    item["count"] = "1";
    item["name"] = ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI
        ? QString("Mehrzweck Gutschein %1").arg(couponcode)
        : QString("Einzweck Gutschein %1").arg(couponcode);
    //        item["discount"] = (payedBy < 0) ? "100.00" : "0.00";
    item["gross"] = (payedBy < PAYED_BY_CASH) ? "0.00" : sum.toString();
    item["tax"] = tax.toString();
    items.append(item);
    mainitem["items"] = items;

    receiptitem.newOrder();

    QrkSettings settings;
    bool useGivenDialog = settings.value("useGivenDialog", false).toBool();
    bool receiptPrintDialog = settings.value("useReceiptPrintedDialog", false).toBool();
    if (receiptitem.setReceiptImportMode(mainitem)) {
        if (useGivenDialog && payedBy == PAYED_BY_CASH) {
            double s = sum.toDouble();
            GivenDialog given(s, this);
            if (given.exec() == 0) {
                return false;
            }
            receiptitem.setGiven(given.getGiven());
        }

        QRKProgress waitBar;
        waitBar.setText(tr("Beleg wird erstellt."));
        waitBar.setWaitMode();
        waitBar.show();

        qApp->processEvents();

        QDateTime dateTime;
        dateTime = QDateTime::currentDateTime();

        Reports rep;
        bool ret = rep.checkEOAny(dateTime);
        if (!ret) {
            return false;
        }

        //    emit sendDatagram("topay",
        //    ui->sumLabel->text().replace(Database::getCurrency() ,""));

        QSqlDatabase dbc = Database::database();
        dbc.transaction();

        int currentReceipt = receiptitem.createReceipts();
        bool sql_ok = true;
        if (currentReceipt > 0) {
            if (receiptitem.createOrder()) {
                receiptitem.setReceiptTime(dateTime);
                if (receiptitem.finishReceipts(payedBy)) {
                    sql_ok = true;
                } else {
                    sql_ok = false;
                }
            } else {
                sql_ok = false;
            }
        }

        if (sql_ok && currentReceipt > 0) {
            dbc.commit();
            if (receiptPrintDialog) {
                QrkTimedMessageBox messageBox(10, QMessageBox::Information, tr("Drucker"),
                    tr("Beleg %1 wurde gedruckt. Nächster Vorgang wird gestartet.").arg(currentReceipt),
                    QMessageBox::Yes | QMessageBox::Default);

                messageBox.setDefaultButton(QMessageBox::Yes);
                messageBox.setButtonText(QMessageBox::Yes, QObject::tr("OK"));
                messageBox.exec();
            }
            if (printing) {
                int type = ui->typeComboBox->currentData().toInt();
                QBCMath tax = ui->taxcomboBox->currentText();
                QBCMath gross(ui->grossEdit->text());
                QString info = QString("%1 - %2 %3")
                                   .arg(type == 0 ? "Mehrzweck" : QString("Einzweck (%1%)").arg(tax.toLocale()))
                                   .arg(gross.toLocale())
                                   .arg(Database::getCurrency());

                CouponPrinter printer(couponcode, info);
                printer.exec();
            }
            save(couponcode, currentReceipt);

        } else {
            sql_ok = dbc.rollback();
            QMessageBox::warning(this, tr("Fehler"),
                tr("Datenbank und/oder Signatur Fehler!\nAktueller BON kann nicht "
                   "erstellt werden. (Rollback: %1).\nÜberprüfen Sie ob genügend "
                   "Speicherplatz für die Datenbank vorhanden ist. Weitere Hilfe gibt "
                   "es im Forum. https:://www.ckvsoft.at")
                    .arg(sql_ok ? tr("durchgeführt") : tr("fehlgeschlagen")));
            qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << dbc.lastError().text();
        }

        //    emit sendDatagram("finished", "");

        return sql_ok;
    }
    return false;
}


void CouponInOutDialog::save(QString &code, int currentReceipt)
{
    QSqlDatabase dbc = Database::database("COUPONINOUT");
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(
        "INSERT INTO coupons (timestamp, code, type, customer, gross, credit, tax, storno, reference, checksum, description, userid) "
        "VALUES (:timestamp, :code, :type, :customer, :gross, :credit, :tax, :storno, :reference, :checksum, :description, :userid);");


    // Berechne den neuen Prüfsummenwert
    int checksumvalue = (CouponDialog::getLastCheckSum().toInt());
    checksumvalue += int(QLocale().toDouble(ui->grossEdit->text()) * 100);
    query.bindValue(":checksum",
        Crypto::encrypt(SecureByteArray(QString::number(checksumvalue).toLatin1()), SecureByteArray("Coupons")));
    query.bindValue(":timestamp", QDateTime::currentDateTime().toString(Qt::ISODate));
    query.bindValue(":code", code);
    query.bindValue(":type", int(ui->typeComboBox->currentData().toInt()));
    query.bindValue(":customer", 0);
    query.bindValue(":gross", int(QLocale().toDouble(ui->grossEdit->text()) * 100));
    query.bindValue(":credit", int(QLocale().toDouble(ui->grossEdit->text()) * 100));
    query.bindValue(":tax",
        ui->typeComboBox->currentData().toInt() == COUPON_TYPE::TYPE_MULTI
            ? 0
            : ui->taxcomboBox->currentText().toDouble() * 100);
    query.bindValue(":storno", 0);
    query.bindValue(":reference", currentReceipt);
    query.bindValue(":description", ui->descriptionTextEdit->toPlainText());
    query.bindValue(":userid", RBAC::Instance()->getUserId());

    query.exec();
    /*
    QrkJournal journal;
    journal.journalInsertLine("Gutschein",
        tr("Manueller %1 - %2")
            .arg((ui->flowComboBox->currentData().toInt() == FLOW_IN) ? tr("Verkauf") : tr("Storno"))
            .arg((ui->flowComboBox->currentData().toInt() == FLOW_IN)
                    ? int(QLocale().toDouble(ui->grossEdit->text()) * 100)
                    : int(QLocale().toDouble(ui->grossEdit->text()) * 100) * -1));
    journal.journalInsertLine("Kassabuch", tr("Kassastand - %1").arg(ProFeatures::getCheckoutContents()));
    journal.journalInsertLine("Kassabuch",
        tr("checksum: %1")
            .arg(Crypto::encrypt(
                SecureByteArray(QString::number(checksumvalue).toLatin1()), SecureByteArray("CashBook"))));
*/

    QDialog::accept();
}


void CouponInOutDialog::readSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    restoreGeometry(settings.value("InOutDialogWindowGeometry").toByteArray());
    settings.endGroup();
}

void CouponInOutDialog::writeSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    settings.save2Settings("InOutDialogWindowGeometry", saveGeometry());
    settings.endGroup();
}
