/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "couponredemption.h"
#include "ui_couponredemption.h"

#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "coupondialog.h"
#include "database.h"
#include "receiptitemmodel.h"

#include <QCompleter>
#include <QRegExpValidator>
#include <QSqlDatabase>
#include <QSqlError>
#include <QTimer>

#include <QDebug>

CouponRedemption::CouponRedemption(ReceiptItemModel *model, QWidget *parent)
    : QWidget(parent)
    , ui(new Ui::CouponRedemption)
    , m_model(model)
{
    ui->setupUi(this);

    connect(ui->redeemEdit, &QLineEdit::textChanged, this, &CouponRedemption::redeemChanged);
    connect(ui->codeEdit, &QLineEdit::textChanged, this, &CouponRedemption::couponcode);
    connect(ui->thirdPartyCheckBox, &QCheckBox::stateChanged, this, &CouponRedemption::checkBoxStateChanged);

    ui->thirdPartyCheckBox->setChecked(false);

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT code FROM coupons");
    query.exec();

    QStringList completerList;

    while (query.next()) {
        QString value = query.value("code").toString();
        completerList << value;
    }

    QCompleter *editorCompleter = new QCompleter(completerList);
    editorCompleter->setCaseSensitivity(Qt::CaseInsensitive);
    editorCompleter->setFilterMode(Qt::MatchContains);
    ui->codeEdit->setCompleter(editorCompleter);

    connect(ui->cancelPushButton, &QrkPushButton::clicked, this, &CouponRedemption::cancel);
    connect(ui->redeemPushButton, &QrkPushButton::clicked, this, &CouponRedemption::accepted);

    QRegExp rx("^?(?:0|[1-9][0-9]*),?[0-9][0-9]$");
    QValidator *doubleVal = new QRegExpValidator(rx, this);

    m_summap = groupAndSum(REGISTER_COL::REGISTER_COL_TAX, REGISTER_COL::REGISTER_COL_TOTAL);

    ui->redeemEdit->setValidator(doubleVal);
    ui->redeemPushButton->setEnabled(false);
    // ui->codeEdit->setFocus();
    QTimer::singleShot(0, [=]() {
        ui->codeEdit->setFocus();
    });
}

CouponRedemption::~CouponRedemption()
{
    delete ui;
}

void CouponRedemption::checkBoxStateChanged(int state)
{
    ui->availableLabel->setHidden(state);
    ui->availableLabel2->setHidden(state);
    ui->remainingLabel->setHidden(state);
    ui->remainingLabel2->setHidden(state);

    m_toPay = ui->toPayLabel->text().toDouble() * 100;

    if (state == Qt::Checked) {
        ui->codeEdit->clear();
        ui->redeemEdit->setText(ui->toPayLabel->text());
        ui->redeemEdit->setEnabled(true);
        ui->typeLabel->setText(tr("Mehrzweck Gutschein"));
        emit valueChanged("code", "3rdparty");
    } else {
        ui->redeemEdit->setText("0");
        ui->redeemEdit->setEnabled(true);
        ui->typeLabel->setText("");
    }
}

QMap<QString, double> CouponRedemption::groupAndSum(REGISTER_COL taxColumn, REGISTER_COL sumColumn)
{
    QMap<QString, double> groupedSums;

    for (int row = 0; row < m_model->rowCount(); ++row) {
        QStandardItem *taxItem = m_model->item(row, taxColumn);
        QStandardItem *sumItem = m_model->item(row, sumColumn);

        if (taxItem && sumItem) {
            QBCMath taxKey(taxItem->text());
            taxKey.round(2);
            QBCMath sumValue(sumItem->text());
            sumValue.round(2);

            groupedSums[taxKey.toString()] += sumValue.toDouble();
        }
    }

    return groupedSums;
}

void CouponRedemption::couponcode(QString code)
{

    int completition = ui->codeEdit->completer()->currentRow();

    qDebug() << "completition: " << completition;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT * FROM coupons WHERE code is :code");
    query.bindValue(":code", code);
    query.exec();
    if (query.next()) {
        m_credit = query.value("credit").toInt();
        m_available = m_credit;
        bool isSingle = (query.value("type").toInt() == COUPON_TYPE::TYPE_SINGLE);
        qDebug() << "Function Name: " << Q_FUNC_INFO << "isSingle: " << isSingle << " - "
                 << query.value("type").toInt();

        if (m_available == 0) ui->infoLabel->setText(tr("Guthaben ist aufgebraucht"));

        // m_available = query.value("gross").toInt();
        m_toPay = ui->toPayLabel->text().toDouble() * 100;
        if (isSingle) {
            QBCMath tax(QString::number(query.value("tax").toInt() / 100));
            tax.round(2);
            QBCMath singlesum(m_summap.value(tax.toString(), 0));
            singlesum.round(2);
            qDebug() << "Function Name: " << Q_FUNC_INFO << "tax: " << tax.toString()
                     << " sum: " << singlesum.toLocale();

            if (singlesum > 0) {
                ui->typeLabel->setText(tr("Einzweck Gutschein (MwSt: %1%)").arg((tax.toLocale())));
                m_toPay = (singlesum * 100).getIntPart().toInt();
                emit valueChanged("tax", tax.toString());
            } else {
                ui->typeLabel->setText(tr("Einzweck Gutschein (MwSt: %1%)").arg((tax.toLocale())));
                if (m_available > 0) ui->infoLabel->setText(tr("Für keinen aktuellen Artikel anwendbar"));
                m_toPay = 0;
            }
        } else
            ui->typeLabel->setText(tr("Mehrzweck Gutschein"));

        qDebug() << "Function Name: " << Q_FUNC_INFO << "toPay: " << m_toPay;

        if (m_credit < m_toPay)
            redeemChanged(QString::number(m_credit / 100.00));
        else
            redeemChanged(QString::number((m_toPay) / 100.00, 'f', 2));

        if (isSingle)
            ui->redeemEdit->setEnabled(false);
        else
            ui->redeemEdit->setFocus();

        emit valueChanged("single", QString::number(isSingle));
        emit valueChanged("code", ui->codeEdit->text());
    }
}

void CouponRedemption::redeemChanged(QString value)
{

    qDebug() << "Function Name: " << Q_FUNC_INFO << "redeem changed: " << value;
    value = value.replace(',', '.');

    blockSignals(true);
    QBCMath redeem(value);
    redeem *= 100;
    QBCMath credit(m_credit);
    QBCMath available(m_available);
    QBCMath toPay(m_toPay);

    ui->redeemEdit->setText(value);
    credit -= redeem;

    QBCMath text1((credit / 100));
    text1.round(2);
    QBCMath text2((available / 100));
    text2.round(2);
    ui->remainingLabel->setText(text1.toLocale());
    ui->availableLabel->setText(text2.toLocale());

    QBCMath open((toPay / 100) - value);
    open.round(2);
    qDebug() << "Function Name: " << Q_FUNC_INFO << "open: " << open.toString()
             << " toPay: " << (toPay / 100).toString() << " value: " << value;

    ui->openLabel->setText(open.toLocale());

    if ((redeem > 0 && credit >= 0) || ui->thirdPartyCheckBox->isChecked())
        ui->redeemPushButton->setEnabled(true);
    else
        ui->redeemPushButton->setEnabled(false);

    blockSignals(false);
    emit valueChanged("redeem", (redeem / 100).toString());
}

bool CouponRedemption::update(QMap<QString, QVariant> &arguments)
{

    qDebug() << "Function Name: " << Q_FUNC_INFO << "arguments: " << arguments;
    int topay = arguments.value("topay", -1).toInt();
    QString code = arguments.value("code").toString();
    int receiptNumber = arguments.value("receiptNumber", 0).toInt();

    arguments.insert("testvalue", "test");

    if (code.compare("3rdparty") == 0) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "3rdparty coupon.";
        return true;
    }

    if (receiptNumber == 0 || code.isEmpty() || topay < 0) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "Not all required parameters provided.";
        return false;
    }

    QSqlDatabase dbc = Database::database("COUPONINOUT");

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("UPDATE coupons SET credit = credit - :topay WHERE code = :code;");
    query.bindValue(":topay", topay);
    query.bindValue(":code", code);

    if (!query.exec()) {
        return false;
    }

    query.prepare("SELECT id, credit FROM coupons WHERE code = :code;");
    query.bindValue(":code", code);

    if (!query.exec()) {
        return false;
    }

    int couponId = -1;
    bool deleted = false;
    if (query.next()) {
        couponId = query.value("id").toInt();
        deleted = query.value("credit").toInt() == 0;
    } else {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "Coupon with code" << code << "not found.";
        return false;
    }

    if (deleted) {
        query.prepare("UPDATE coupons SET deleted = :deleted WHERE code = :code;");
        query.bindValue(":deleted", deleted);
        query.bindValue(":code", code);

        if (!query.exec()) {
            return false;
        }
    }

    query.prepare("INSERT INTO coupon_redemptions (coupon_id, redemption_date, redemption_amount) "
                  "VALUES (:coupon_id, :redemption_date, :redemption_amount);");

    query.bindValue(":coupon_id", couponId);
    query.bindValue(":redemption_date", QDateTime::currentDateTime().toString(Qt::ISODate));
    query.bindValue(":redemption_amount", topay);

    if (!query.exec()) {
        return false;
    }

    return true;
}


void CouponRedemption::setValue(const QString &key, const QString &value)
{
    if (key == "sum") {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "key: " << key << "value: " << value;
        ui->toPayLabel->setText(value);
        ui->openLabel->setText(value);
    }
}
