/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "barcodessettings.h"
#include "qrkpushbutton.h"
#include "qrksettings.h"

#include <QCheckBox>
#include <QFontDatabase>
#include <QFormLayout>
#include <QGroupBox>
#include <QLabel>
#include <QPainter>
#include <QPrintDialog>
#include <QPrinter>
#include <QTextDocument>
#include <QVBoxLayout>

BarcodesSettings::BarcodesSettings(QWidget *parent)
    : QWidget(parent)
{

    int id = QFontDatabase::addApplicationFont(":font/free3of9.ttf");
    QFontDatabase::applicationFontFamilies(id).at(0);


    m_mainGroup = new QGroupBox(this);
    m_mainGroup->setFlat(true);
    QVBoxLayout *mainLayout = new QVBoxLayout(this);

    QHBoxLayout *horizontalLayout = new QHBoxLayout(m_mainGroup);
    QGridLayout *mainGridLayout = new QGridLayout();

    // EAN13
    m_ean13Group = new QGroupBox(m_mainGroup);
    m_ean13Group->setTitle(tr("EAN-13 / variable Gewichts- und Preiseinheiten"));
    m_ean13checkbox_T21 = new QCheckBox(m_ean13Group);
    m_ean13checkbox_T28 = new QCheckBox(m_ean13Group);

    QFormLayout *eanFormLayout = new QFormLayout(m_ean13Group);
    eanFormLayout->setAlignment(Qt::AlignLeft);
    // m_ean13checkbox_T21->setHidden(true);
    eanFormLayout->addRow(tr("Type 21/22 gruppieren:"), m_ean13checkbox_T21);
    eanFormLayout->addRow(tr("Type 28/29 gruppieren:"), m_ean13checkbox_T28);
    mainGridLayout->addWidget(m_ean13Group, 0, 0, 1, 2);

    // Amount
    m_amountGroup = new QGroupBox(m_mainGroup);
    m_amountGroup->setTitle(tr("Werte"));
    m_barcode_amount_0 = new QLineEdit(m_amountGroup);
    m_barcode_amount_1 = new QLineEdit(m_amountGroup);
    m_barcode_amount_2 = new QLineEdit(m_amountGroup);
    m_barcode_amount_3 = new QLineEdit(m_amountGroup);
    m_barcode_amount_4 = new QLineEdit(m_amountGroup);
    m_barcode_amount_5 = new QLineEdit(m_amountGroup);
    m_barcode_amount_6 = new QLineEdit(m_amountGroup);
    m_barcode_amount_7 = new QLineEdit(m_amountGroup);
    m_barcode_amount_8 = new QLineEdit(m_amountGroup);
    m_barcode_amount_9 = new QLineEdit(m_amountGroup);

    QFormLayout *amountFormLayout = new QFormLayout(m_amountGroup);
    QSpacerItem *verticalSpacer2 = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);

    amountFormLayout->setAlignment(Qt::AlignLeft);
    amountFormLayout->addRow(tr("Wert   0:"), m_barcode_amount_0);
    amountFormLayout->addRow(tr("Wert   1:"), m_barcode_amount_1);
    amountFormLayout->addRow(tr("Wert   2:"), m_barcode_amount_2);
    amountFormLayout->addRow(tr("Wert   3:"), m_barcode_amount_3);
    amountFormLayout->addRow(tr("Wert   4:"), m_barcode_amount_4);
    amountFormLayout->addRow(tr("Wert   5:"), m_barcode_amount_5);
    amountFormLayout->addRow(tr("Wert   6:"), m_barcode_amount_6);
    amountFormLayout->addRow(tr("Wert   7:"), m_barcode_amount_7);
    amountFormLayout->addRow(tr("Wert   8:"), m_barcode_amount_8);
    amountFormLayout->addRow(tr("Wert   9:"), m_barcode_amount_9);
    amountFormLayout->addItem(verticalSpacer2);

    mainGridLayout->addWidget(m_amountGroup, 1, 1, 1, 1);

    // Action
    m_actionGroup = new QGroupBox(m_mainGroup);
    m_actionGroup->setTitle(tr("Aktionen - "));
    m_barcode_finishReceipt = new QLineEdit(m_actionGroup);
    m_barcode_removeLastPosition = new QLineEdit(m_actionGroup);
    m_barcode_endOfDay = new QLineEdit(m_actionGroup);
    m_barcode_discount = new QLineEdit(m_actionGroup);
    m_barcode_editPrice = new QLineEdit(m_actionGroup);

    m_barcode_printLastReceiptAgain = new QLineEdit(m_actionGroup);
    m_barcode_cancelLastReceipt = new QLineEdit(m_actionGroup);

    m_barcode_amount_00 = new QLineEdit(m_actionGroup);
    m_barcode_amount_000 = new QLineEdit(m_actionGroup);
    m_barcode_amount_250 = new QLineEdit(m_actionGroup);
    m_barcode_amount_500 = new QLineEdit(m_actionGroup);

    QFormLayout *actionFormLayout = new QFormLayout(m_actionGroup);
    QSpacerItem *verticalSpacer = new QSpacerItem(40, 40, QSizePolicy::Expanding, QSizePolicy::Expanding);

    actionFormLayout->addRow(tr("BON Abschluss:"), m_barcode_finishReceipt);
    actionFormLayout->addRow(tr("Letzten Artikel entfernen:"), m_barcode_removeLastPosition);
    actionFormLayout->addRow(tr("Tagesabschluss:"), m_barcode_endOfDay);
    actionFormLayout->addRow(tr("Rabatt ändern:"), m_barcode_discount);
    actionFormLayout->addRow(tr("Preis ändern:"), m_barcode_editPrice);
    actionFormLayout->addRow(tr("Kopie des letzten BON Drucken:"), m_barcode_printLastReceiptAgain);
    actionFormLayout->addRow(tr("Storno des letzten BONs:"), m_barcode_cancelLastReceipt);
    actionFormLayout->addRow(tr("Wert  00:"), m_barcode_amount_00);
    actionFormLayout->addRow(tr("Wert 000:"), m_barcode_amount_000);
    actionFormLayout->addRow(tr("Wert 250:"), m_barcode_amount_250);
    actionFormLayout->addRow(tr("Wert 500:"), m_barcode_amount_500);
    actionFormLayout->addItem(verticalSpacer);

    mainGridLayout->addWidget(m_actionGroup, 1, 0, 1, 1);
    mainGridLayout->setRowStretch(1, 1);

    horizontalLayout->addLayout(mainGridLayout);
    mainLayout->addWidget(m_mainGroup);

    m_mainGroup->setCheckable(true);
    m_mainGroup->setChecked(false);
    m_mainGroup->setTitle(tr("Barcode Plugin verwenden"));

    QrkPushButton *cancelButton = new QrkPushButton;
    cancelButton->setMinimumHeight(60);
    cancelButton->setMinimumWidth(0);
    QIcon cancelicon = QIcon(":src/icons/cancel.png");
    QSize size = QSize(32, 32);
    cancelButton->setIcon(cancelicon);
    cancelButton->setIconSize(size);
    cancelButton->setText(tr("Beenden"));

    QrkPushButton *saveButton = new QrkPushButton;
    saveButton->setMinimumHeight(60);
    saveButton->setMinimumWidth(0);

    QIcon saveicon = QIcon(":src/icons/save.png");
    saveButton->setIcon(saveicon);
    saveButton->setIconSize(size);
    saveButton->setText(tr("Speichern"));

    QrkPushButton *printBarcodesButton = new QrkPushButton;
    printBarcodesButton->setMinimumHeight(60);
    printBarcodesButton->setMinimumWidth(0);

    QIcon printicon = QIcon(":src/icons/ok-print.png");
    printBarcodesButton->setIcon(printicon);
    printBarcodesButton->setIconSize(size);
    printBarcodesButton->setText(tr("Speichern und Drucken"));

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    QSpacerItem *spacer = new QSpacerItem(40, 40, QSizePolicy::Expanding, QSizePolicy::Expanding);
    buttonLayout->addItem(spacer);
    buttonLayout->addWidget(printBarcodesButton);
    buttonLayout->addWidget(saveButton);
    buttonLayout->addWidget(cancelButton);

    mainLayout->addWidget(m_mainGroup);
    mainLayout->addLayout(buttonLayout);
    mainLayout->setStretch(0, 1);
    mainLayout->addSpacing(12);

    connect(printBarcodesButton, &QPushButton::clicked, this, &BarcodesSettings::printBarcodes);
    connect(saveButton, &QPushButton::clicked, this, &BarcodesSettings::save);
    connect(cancelButton, &QPushButton::clicked, this, &BarcodesSettings::cancelClicked);

    setLayout(mainLayout);

    init();
}

void BarcodesSettings::init()
{
    QrkSettings settings;
    settings.beginGroup("BarCodesPlugin");

    m_mainGroup->setChecked(settings.value("barcode_enabled", false).toBool());
    m_ean13checkbox_T21->setChecked(settings.value("Ean13Type21_22_group", true).toBool());
    m_ean13checkbox_T28->setChecked(settings.value("Ean13Type28_29_group", true).toBool());

    m_barcode_finishReceipt->setText(settings.value("barcodeFinishReceipt", "100009000001").toString());
    m_barcode_removeLastPosition->setText(settings.value("barcodeRemoveLastPosition", "100009000002").toString());
    m_barcode_endOfDay->setText(settings.value("barcodeEndOfDay", "100009000003").toString());
    m_barcode_discount->setText(settings.value("barcodeDiscount", "100009000007").toString());
    m_barcode_editPrice->setText(settings.value("barcodeEditPrice", "100009000010").toString());
    m_barcode_printLastReceiptAgain->setText(settings.value("barcodePrintLastReceiptAgain", "100009000005").toString());
    m_barcode_cancelLastReceipt->setText(settings.value("barcodeCancelReceipt", "100009000006").toString());
    m_barcode_amount_0->setText(settings.value("barcodeAmount_0", "100008000000").toString());
    m_barcode_amount_1->setText(settings.value("barcodeAmount_1", "100008000001").toString());
    m_barcode_amount_2->setText(settings.value("barcodeAmount_2", "100008000002").toString());
    m_barcode_amount_3->setText(settings.value("barcodeAmount_3", "100008000003").toString());
    m_barcode_amount_4->setText(settings.value("barcodeAmount_4", "100008000004").toString());
    m_barcode_amount_5->setText(settings.value("barcodeAmount_5", "100008000005").toString());
    m_barcode_amount_6->setText(settings.value("barcodeAmount_6", "100008000006").toString());
    m_barcode_amount_7->setText(settings.value("barcodeAmount_7", "100008000007").toString());
    m_barcode_amount_8->setText(settings.value("barcodeAmount_8", "100008000008").toString());
    m_barcode_amount_9->setText(settings.value("barcodeAmount_9", "100008000009").toString());
    m_barcode_amount_00->setText(settings.value("barcodeAmount_00", "100008000020").toString());
    m_barcode_amount_000->setText(settings.value("barcodeAmount_000", "100008000030").toString());
    m_barcode_amount_250->setText(settings.value("barcodeAmount_250", "100008000250").toString());
    m_barcode_amount_500->setText(settings.value("barcodeAmount_500", "100008000500").toString());

    settings.endGroup();
}

void BarcodesSettings::save()
{
    QrkSettings settings;
    settings.beginGroup("BarCodesPlugin");

    settings.save2Settings("barcode_enabled", m_mainGroup->isChecked());
    settings.save2Settings("barcodeFinishReceipt", m_barcode_finishReceipt->text());
    settings.save2Settings("barcodeRemoveLastPosition", m_barcode_removeLastPosition->text());
    settings.save2Settings("barcodeEndOfDay", m_barcode_endOfDay->text());
    settings.save2Settings("barcodeDiscount", m_barcode_discount->text());
    settings.save2Settings("barcodeEditPrice", m_barcode_editPrice->text());
    settings.save2Settings("barcodePrintLastReceiptAgain", m_barcode_printLastReceiptAgain->text());
    settings.save2Settings("barcodeCancelReceipt", m_barcode_cancelLastReceipt->text());
    settings.save2Settings("barcodeAmount_0", m_barcode_amount_0->text());
    settings.save2Settings("barcodeAmount_1", m_barcode_amount_1->text());
    settings.save2Settings("barcodeAmount_2", m_barcode_amount_2->text());
    settings.save2Settings("barcodeAmount_3", m_barcode_amount_3->text());
    settings.save2Settings("barcodeAmount_4", m_barcode_amount_4->text());
    settings.save2Settings("barcodeAmount_5", m_barcode_amount_5->text());
    settings.save2Settings("barcodeAmount_6", m_barcode_amount_6->text());
    settings.save2Settings("barcodeAmount_7", m_barcode_amount_7->text());
    settings.save2Settings("barcodeAmount_8", m_barcode_amount_8->text());
    settings.save2Settings("barcodeAmount_9", m_barcode_amount_9->text());
    settings.save2Settings("barcodeAmount_00", m_barcode_amount_00->text());
    settings.save2Settings("barcodeAmount_000", m_barcode_amount_000->text());
    settings.save2Settings("barcodeAmount_250", m_barcode_amount_250->text());
    settings.save2Settings("barcodeAmount_500", m_barcode_amount_500->text());

    settings.save2Settings("Ean13Type21_22_group", m_ean13checkbox_T21->isChecked());
    settings.save2Settings("Ean13Type28_29_group", m_ean13checkbox_T28->isChecked());

    settings.endGroup();
}

void BarcodesSettings::printBarcodes()
{
    QPrinter printer;

    QPrintDialog *dialog = new QPrintDialog(&printer, this);
    dialog->setWindowTitle("Print Document");
    if (dialog->exec() != QDialog::Accepted) return;

    QTextDocument doc;

    QString text = "<table style=\"width: 100%;\" cellspacing=\"5\" "
                   "cellpadding=\"5\"><tbody><tr>"
                   "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_finishReceipt->text() + "*</td><td>&nbsp;</td>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_removeLastPosition->text() + "*</td>"
        + "</tr><tr>" + "<td>" + tr("BON Abschluss") + "</td><td>&nbsp;</td>" + "<td>" + tr("Letzten Artikel entfernen")
        + "</td>" + "</tr><tr>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_endOfDay->text()
        + "*</td><td>&nbsp;</td>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_discount->text() + "*</td>" + "</tr><tr>" + "<td>" + tr("Tagesabschluss") + "</td><td>&nbsp;</td>"
        + "<td>" + tr("Rabatt ändern") + "</td>" + "</tr><tr>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_editPrice->text()
        + "*</td><td>&nbsp;</td>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_printLastReceiptAgain->text() + "*</td>" + "</tr><tr>" + "<td>" + tr("Preis ändern")
        + "</td><td>&nbsp;</td>" + "<td>" + tr("Kopie des letzten BON Drucken") + "</td>" + "</tr><tr>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_cancelLastReceipt->text()
        + "*</td><td>&nbsp;</td><td>&nbsp;</td>" + "</tr><tr>" + "<td>" + tr("Storno des letzten BONs:")
        + "</td><td>&nbsp;</td><td>&nbsp;</td>" + "</tr></tbody></table>";

    doc.setHtml(text);
    doc.adjustSize();
    doc.print(&printer);

    text = "<table style=\"width: 100%;\" cellspacing=\"5\" "
           "cellpadding=\"5\"><tbody><tr>"
           "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_0->text() + "*</td><td>&nbsp;</td>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_1->text() + "*</td>" + "</tr><tr>"
        + "<td>" + tr("Wert 0") + "</td><td>&nbsp;</td>" + "<td>" + tr("Wert 1") + "</td>" + "</tr><tr>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_2->text()
        + "*</td><td>&nbsp;</td>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_3->text() + "*</td>" + "</tr><tr>" + "<td>" + tr("Wert 2") + "</td><td>&nbsp;</td>" + "<td>"
        + tr("Wert 3") + "</td>" + "</tr><tr>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_4->text() + "*</td><td>&nbsp;</td>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_5->text() + "*</td>" + "</tr><tr>"
        + "<td>" + tr("Wert 4") + "</td><td>&nbsp;</td>" + "<td>" + tr("Wert 5") + "</td>" + "</tr><tr>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_6->text()
        + "*</td><td>&nbsp;</td>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_7->text() + "*</td>" + "</tr><tr>" + "<td>" + tr("Wert 6") + "</td><td>&nbsp;</td>" + "<td>"
        + tr("Wert 7") + "</td>" + "</tr><tr>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_8->text() + "*</td><td>&nbsp;</td>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_9->text() + "*</td>" + "</tr><tr>"
        + "<td>" + tr("Wert 8") + "</td><td>&nbsp;</td>" + "<td>" + tr("Wert 9") + "</td>" + "</tr><tr>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_00->text()
        + "*</td><td>&nbsp;</td>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_000->text() + "*</td>" + "</tr><tr>" + "<td>" + tr("Wert 00") + "</td><td>&nbsp;</td>"
        + "<td>" + tr("Wert 000") + "</td>" + "</tr><tr>" + "<td style=\"font-family:free 3 of 9;font-size:50px\">*"
        + m_barcode_amount_250->text() + "*</td><td>&nbsp;</td>"
        + "<td style=\"font-family:free 3 of 9;font-size:50px\">*" + m_barcode_amount_500->text() + "*</td>"
        + "</tr><tr>" + "<td>" + tr("Wert 250") + "</td><td>&nbsp;</td>" + "<td>" + tr("Wert 500") + "</td>"
        + "</tr></tbody></table>";

    doc.setHtml(text);
    doc.adjustSize();
    doc.print(&printer);

    emit save();
}
