/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "productchart.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "database.h"
#include "qrkpushbutton.h"
#include "qrksettings.h"
#include "utils/utils.h"

#include <QComboBox>
#include <QDateEdit>
#include <QDir>
#include <QFile>
#include <QFileDialog>
#include <QHeaderView>
#include <QLabel>
#include <QLocale>
#include <QMetaType>
#include <QSplitter>
#include <QSqlDatabase>
#include <QSqlError>
#include <QStandardItemModel>
#include <QTableView>
#include <QVBoxLayout>

#include <QDebug>

ProductChart::ProductChart(QWidget *parent)
    : QDialog(parent)
{

    setupModel();
    setupViews();

    loadData();

    setWindowTitle(tr("Chart"));
    //    resize(750, 400);

    readSettings();
}

ProductChart::~ProductChart()
{
    writeSettings();
}

void ProductChart::setupModel()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);
    q.prepare("SELECT COUNT(id) FROM products WHERE products.groupid > 1");
    int rows = 0;
    if (q.next()) rows = q.value(0).toInt();

    model = new QStandardItemModel(rows, 3, this);
    model->setHeaderData(0, Qt::Horizontal, tr("Produktname"));
    model->setHeaderData(1, Qt::Horizontal, tr("verkauft"));
    model->setHeaderData(2, Qt::Horizontal, "%");
}

void ProductChart::setupViews()
{

    QVBoxLayout *vlayout = new QVBoxLayout;
    m_splitter = new QSplitter;
    QTableView *table = new QTableView;

    m_chart = new ChartWidget;
    m_chart->setType(ChartWidget::Pie);
    m_chart->setLegendType(ChartWidget::None);

    m_splitter->addWidget(table);
    m_splitter->addWidget(m_chart);
    m_splitter->setSizes(QList<int>({ 150, 100 }));

    table->setModel(model);

    QHBoxLayout *comboLayout = new QHBoxLayout;

    m_topflop = new QComboBox;
    m_topflop->addItem(tr("Top"), "DESC");
    m_topflop->addItem(tr("Flop"), "ASC");

    m_combo = new QComboBox;
    m_combo->addItem("10", 10);
    m_combo->addItem("20", 20);
    m_combo->addItem("50", 50);
    m_combo->addItem(tr("alle"), -1);
    m_combo->setCurrentText("10");
    QLabel *comboLabel = new QLabel;
    comboLabel->setText(tr("Anzeigen der "));


    m_toDate = new QDateEdit;
    m_toDate->setDateTime(QDateTime::currentDateTime());
    m_toDate->setCalendarPopup(true);
    m_toDate->setMaximumDate(QDate::currentDate());

    m_fromDate = new QDateEdit;
    m_fromDate->setDate(Database::getFirstReceiptDate());
    m_fromDate->setCalendarPopup(true);
    m_fromDate->setMinimumDate(Database::getFirstReceiptDate());
    m_fromDate->setMaximumDate(m_toDate->date());
    m_toDate->setMinimumDate(m_fromDate->date());

    QLabel *fromDateLabel = new QLabel;
    fromDateLabel->setText(tr("vom"));
    QLabel *toDateLabel = new QLabel;
    toDateLabel->setText(tr("bis zum"));

    QLabel *categoriesLabel = new QLabel;
    categoriesLabel->setText(tr("Kategorie"));
    m_categories = new QComboBox;
    m_categories->addItem(tr("Alle"), 0);
    addCategories();

    QLabel *groupsLabel = new QLabel;
    groupsLabel->setText(tr("Warengruppe"));
    m_groups = new QComboBox;
    m_groups->addItem(tr("Alle"), 0);
    addGroups();

    QSpacerItem *comboSpacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    comboLayout->addWidget(comboLabel);
    comboLayout->addWidget(m_topflop);
    comboLayout->addWidget(m_combo);
    comboLayout->addWidget(fromDateLabel);
    comboLayout->addWidget(m_fromDate);
    comboLayout->addWidget(toDateLabel);
    comboLayout->addWidget(m_toDate);
    comboLayout->addWidget(categoriesLabel);
    comboLayout->addWidget(m_categories);
    comboLayout->addWidget(groupsLabel);
    comboLayout->addWidget(m_groups);
    comboLayout->addItem(comboSpacer);

    QrkPushButton *pushButton = new QrkPushButton;
    pushButton->setMinimumHeight(60);
    pushButton->setMinimumWidth(0);

    QIcon icon = QIcon(":src/icons/ok.png");
    QSize size = QSize(24, 24);
    pushButton->setIcon(icon);
    pushButton->setIconSize(size);
    pushButton->setText(tr("OK"));

    QrkPushButton *exportPushButton = new QrkPushButton;
    exportPushButton->setMinimumHeight(60);
    exportPushButton->setMinimumWidth(0);

    QIcon exporticon = QIcon(":src/icons/save.png");
    exportPushButton->setIcon(exporticon);
    exportPushButton->setIconSize(size);
    exportPushButton->setText(tr("Export"));

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    QSpacerItem *spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    buttonLayout->addItem(spacer);
    buttonLayout->addWidget(exportPushButton);
    buttonLayout->addWidget(pushButton);

    QItemSelectionModel *selectionModel = new QItemSelectionModel(model);
    table->setSelectionModel(selectionModel);
    table->setSortingEnabled(true);
    table->sortByColumn(1, Qt::DescendingOrder);
    QHeaderView *headerView = table->horizontalHeader();
    headerView->setSectionResizeMode(0, QHeaderView::Stretch);

    QFrame *lineH1 = new QFrame;
    lineH1->setFrameShape(QFrame::HLine);
    QFrame *lineH2 = new QFrame;
    lineH2->setFrameShape(QFrame::HLine);

    vlayout->addLayout(comboLayout);
    vlayout->addWidget(lineH1);
    vlayout->addWidget(m_splitter);
    vlayout->addWidget(lineH2);
    vlayout->addLayout(buttonLayout);
    vlayout->setStretch(2, 1);
    setLayout(vlayout);

    connect(pushButton, &QPushButton::clicked, this, &ProductChart::close);
    connect(exportPushButton, &QPushButton::clicked, this, &ProductChart::export_csv);
    connect(m_combo, &QComboBox::currentTextChanged, this, &ProductChart::comboBoxChanged);
    connect(m_categories, &QComboBox::currentTextChanged, this, &ProductChart::categoriesBoxChanged);
    connect(m_groups, &QComboBox::currentTextChanged, this, &ProductChart::groupBoxChanged);
    connect(m_topflop, &QComboBox::currentTextChanged, this, &ProductChart::comboBoxChanged);
    connect(m_fromDate, &QDateEdit::dateChanged, this, &ProductChart::datetimeChanged);
    connect(m_toDate, &QDateEdit::dateChanged, this, &ProductChart::datetimeChanged);
}

void ProductChart::loadData(SORTORDER order)
{

    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);

    //    q.prepare("SELECT sum(sold) as sold, sum(sold * gross) as gross FROM products WHERE products.groupid > 1");
    q.prepare(QString("select sum(orders.count) as sold, sum(orders.count) * orders.gross as gross from receipts join "
                      "orders ON receipts.receiptNum = orders.receiptId join products ON orders.product = products.id "
                      "%1 AND receipts.timestamp BETWEEN :fromdate AND :todate")
            .arg(m_groupSelect));
    q.bindValue(":fromdate", m_fromDate->dateTime().toString(Qt::ISODate));
    q.bindValue(":todate", m_toDate->dateTime().toString(Qt::ISODate));

    q.exec();

    QBCMath whole = 0;
    while (q.next()) {
        if (order == GROSS)
            whole += q.value("gross").toDouble();
        else
            whole += q.value("sold").toDouble();
    }

    whole.round(2);

    if (whole.toDouble() == 0.0) {
        model->removeRows(0, model->rowCount(QModelIndex()), QModelIndex());
        return;
    }

    QBCMath current(0.0);

    if (order == SOLD) {
        // q.prepare(QString("SELECT name, sold FROM products  WHERE products.groupid > 1 ORDER BY sold %1 LIMIT
        // :limit").arg(m_topflop->currentData().toString()));
        q.prepare(QString("select products.name, sum(orders.count) as sold, sum(orders.count) * orders.gross as "
                          "sumgross from receipts join orders ON receipts.receiptNum = orders.receiptId join products "
                          "ON orders.product = products.id %1 AND receipts.timestamp BETWEEN :fromdate AND :todate "
                          "group by product ORDER BY sold %2 LIMIT :limit")
                .arg(m_groupSelect)
                .arg(m_topflop->currentData().toString()));
    } else {
        //        q.prepare("SELECT name, sold * gross as sumgross FROM products  WHERE products.groupid > 1 ORDER BY
        //        sumgross DESC LIMIT :limit");
        q.prepare(QString("select products.name, sum(orders.count) as sold, sum(orders.count) * orders.gross as "
                          "sumgross from receipts join orders ON receipts.receiptNum = orders.receiptId join products "
                          "ON orders.product = products.id %1 AND receipts.timestamp BETWEEN :fromdate AND :todate "
                          "group by product ORDER BY sumgross LIMIT :limit")
                .arg(m_groupSelect));
    }

    q.bindValue(":limit", m_maxview);
    q.bindValue(":fromdate", m_fromDate->dateTime().toString(Qt::ISODate));
    q.bindValue(":todate", m_toDate->dateTime().toString(Qt::ISODate));

    q.exec();

    model->removeRows(0, model->rowCount(QModelIndex()), QModelIndex());

    int size = QColor::colorNames().size();
    int row = 0;
    int color = 0;
    while (q.next()) {
        model->insertRows(row, 1, QModelIndex());

        QString name = q.value("name").toString();
        QBCMath part;
        if (order == SOLD) {
            part = q.value("sold").toDouble();
        } else {
            part = q.value("sumgross").toDouble();
        }

        part.round(2);

        current += part;
        QBCMath percentage((part / whole) * 100);
        percentage.round(2);

        model->setData(model->index(row, 0, QModelIndex()), name);
        model->setData(model->index(row, 1, QModelIndex()), part.toString());
        model->setData(model->index(row, 2, QModelIndex()), percentage.toString().rightJustified(5, '0') + " %");
        model->setData(model->index(row, 1, QModelIndex()), Qt::AlignRight, Qt::TextAlignmentRole);
        model->setData(model->index(row, 2, QModelIndex()), Qt::AlignRight, Qt::TextAlignmentRole);

        model->setData(model->index(row, 0, QModelIndex()), QColor(QColor::colorNames().at(color)), Qt::DecorationRole);

        m_chart->addItem(name, QColor::colorNames().at(color), percentage.toDouble());
        row++;
        color++;
        if (color + 2 > size) color = 0;
    }

    if (whole > 0) {
        QBCMath percentage(((whole - current) / whole) * 100);
        QBCMath part(whole - current);
        part.round(2);
        percentage.round(2);
        QString name = tr("Rest");
        m_chart->addItem(name, QColor::colorNames().at(color + 1), percentage.toDouble());
        if (percentage.toDouble() > 0.00) {
            model->insertRows(row, 1, QModelIndex());
            model->setData(model->index(row, 0, QModelIndex()), name);
            model->setData(model->index(row, 1, QModelIndex()), part.toString());
            model->setData(model->index(row, 2, QModelIndex()), percentage.toString().rightJustified(5, '0') + " %");
            model->setData(model->index(row, 1, QModelIndex()), Qt::AlignRight, Qt::TextAlignmentRole);
            model->setData(model->index(row, 2, QModelIndex()), Qt::AlignRight, Qt::TextAlignmentRole);

            model->setData(
                model->index(row, 0, QModelIndex()), QColor(QColor::colorNames().at(color + 1)), Qt::DecorationRole);
        }
    }
}

void ProductChart::datetimeChanged(const QDate &)
{
    m_fromDate->setMaximumDate(m_toDate->date());
    m_toDate->setMinimumDate(m_fromDate->date());

    QString text;
    comboBoxChanged(text);
}

void ProductChart::comboBoxChanged(QString)
{
    m_maxview = m_combo->currentData().toInt();
    m_chart->removeItems();
    loadData();
    m_chart->repaint();
}

void ProductChart::categoriesBoxChanged(QString)
{
    m_categoriesId = m_categories->currentData().toInt();
    addGroups();
    m_groups->setCurrentIndex(0);
}

void ProductChart::groupBoxChanged(QString)
{
    m_groupId = m_groups->currentData().toInt();
    if (m_groupId == 0)
        m_groupSelect = QString("where products.groupid IN (%1)").arg(getGroupIds().join(","));
    else
        m_groupSelect = QString("where products.groupid = %1").arg(m_groupId);

    m_chart->removeItems();
    loadData();
    m_chart->repaint();
}

void ProductChart::addCategories()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);
    q.exec("SELECT id, name from categories");
    while (q.next())
        m_categories->addItem(q.value("name").toString(), q.value("id").toInt());
}

void ProductChart::addGroups()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);
    QString categoriesWhere
        = QString("groups.categoryId %1").arg(m_categoriesId == 0 ? "> 0" : QString("= %1").arg(m_categoriesId));

    m_groups->clear();
    m_groups->addItem(tr("Alle %1").arg(getCategoryNameById(m_categoriesId)), 0);
    q.exec(QString("SELECT id, name from groups WHERE %1 AND id > 1").arg(categoriesWhere));
    while (q.next())
        m_groups->addItem(q.value("name").toString(), q.value("id").toInt());
}

QStringList ProductChart::getGroupIds()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);
    if (m_categoriesId == 0)
        q.exec("SELECT id FROM groups WHERE id > 1");
    else
        q.exec(QString("SELECT id FROM groups WHERE categoryId = %1 AND id > 1").arg(m_categoriesId));

    QStringList list;
    while (q.next()) {
        list.append(q.value("id").toString());
    }

    return list;
}

QString ProductChart::getCategoryNameById(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);
    q.exec(QString("SELECT name FROM categories WHERE id = %1").arg(id));
    if (q.next()) return q.value("name").toString();

    return {};
}

void ProductChart::readSettings()
{
    QrkSettings settings;
    settings.beginGroup("Chart");
    restoreGeometry(settings.value("WindowGeometry").toByteArray());
    m_splitter->restoreGeometry(settings.value("splitterGeometry").toByteArray());
    m_splitter->restoreState(settings.value("splitterState").toByteArray());

    settings.endGroup();
}

void ProductChart::writeSettings()
{
    QrkSettings settings;
    settings.beginGroup("Chart");
    settings.save2Settings("WindowGeometry", saveGeometry());
    settings.save2Settings("splitterGeometry", m_splitter->saveGeometry(), false);
    settings.save2Settings("splitterState", m_splitter->saveState(), false);

    settings.endGroup();
}

void ProductChart::export_csv()
{

    QrkSettings settings;
    QString lastUsedDirectory = settings.value("Chat/lastUsedDirectory", QDir::currentPath()).toString();

    m_filename = QFileDialog::getSaveFileName(Q_NULLPTR, tr("Datei speichern"), lastUsedDirectory, "Chart (*.csv)",
        Q_NULLPTR, QFileDialog::DontUseNativeDialog);

    if (m_filename.isNull()) return;

    settings.save2Settings("Chat/lastUsedDirectory", m_filename);

    QFile f(m_filename);
    if (f.open(QIODevice::WriteOnly)) {
        QTextStream ts(&f);
        QStringList strList;

        ts << tr("\"%1/%2 %3 - %4\"\n")
                  .arg(m_categories->currentText())
                  .arg(m_groups->currentText())
                  .arg(m_fromDate->date().toString())
                  .arg(m_toDate->date().toString());

        ts << tr("Artikelname;Anzahl;Anteil\n");

        for (int i = 0; i < model->rowCount(); i++) {
            strList.clear();

            for (int j = 0; j < model->columnCount(); j++) {
                if (Utils::isNumber(model->data(model->index(i, j))))
                    strList << QBCMath(model->data(model->index(i, j)).toString()).toLocale();
                else
                    strList << "\"" + model->data(model->index(i, j)).toString().replace("\"", "\"\"") + "\"";
            }

            ts << strList.join(";") + "\n";
        }
        f.close();
    }
}
