/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "categoryedit.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "database.h"
#include "utils/utils.h"

#include <ui_categoryedit.h>

#include <QDebug>
#include <QJsonArray>
#include <QJsonObject>
#include <QSqlError>

//--------------------------------------------------------------------------------

CategoryEdit::CategoryEdit(QWidget *parent, int id)
    : QDialog(parent)
    , ui(new Ui::CategoryEdit)
    , m_id(id)
{
    ui->setupUi(this);

    const QStringList colorNames = QColor::colorNames();
    int index = 0;
    ui->colorComboBox->addItem(tr("Standard Farbe"));
    const QModelIndex idx = ui->colorComboBox->model()->index(index++, 0);
    ui->colorComboBox->model()->setData(idx, "", Qt::BackgroundColorRole);

    foreach (const QString &colorName, colorNames) {
        const QColor color(colorName);
        QString fg = Utils::color_best_contrast(color.name());
        const QColor fg_color(fg);
        ui->colorComboBox->addItem(colorName, color);
        const QModelIndex idx = ui->colorComboBox->model()->index(index++, 0);
        ui->colorComboBox->model()->setData(idx, color, Qt::BackgroundColorRole);
        ui->colorComboBox->model()->setData(idx, fg_color, Qt::ForegroundRole);
    }

    QJsonArray printers = Database::getPrinters();
    ui->printerComboBox->addItem(tr("kein Drucker"), 0);
    foreach (const QJsonValue &value, printers) {
        QJsonObject obj = value.toObject();
        ui->printerComboBox->addItem(obj["name"].toString(), obj["id"].toInt());
    }

    ui->okButton->setDisabled(true);

    if (m_id != -1) {
        QSqlDatabase dbc = Database::database();

        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.exec(QString("SELECT name, color, printerid, visible FROM categories WHERE id=%1").arg(id));
        query.next();

        ui->name->setText(query.value(0).toString());
        ui->visibleCheckBox->setChecked(query.value("visible").toBool());
        ui->okButton->setDisabled(ui->name->text().isEmpty());

        int i;
        for (i = 0; i <= ui->colorComboBox->count(); i++) {
            QString color = ui->colorComboBox->model()->index(i, 0).data(Qt::BackgroundColorRole).toString();
            if (query.value(2).toString() == color) break;
        }

        if (i > ui->colorComboBox->count()) i = 0;

        QPalette palette(ui->colorComboBox->palette());
        QString colorValue = query.value(2).toString();
        if (!colorValue.isEmpty()) {
            QColor color(query.value(2).toString());
            palette.setColor(QPalette::Active, QPalette::Button, color);
            palette.setColor(QPalette::Highlight, color);
            //    palette.setColor(QPalette::ButtonText, Qt::white);
            ui->colorComboBox->setPalette(palette);
        }
        ui->colorComboBox->setCurrentIndex(i);

        int currentPrinterId = query.value("printerid").toInt();
        for (i = 0; i <= ui->printerComboBox->count(); i++) {
            int itemData = ui->printerComboBox->itemData(i).toInt();
            if (currentPrinterId == itemData) break;
        }

        if (i > ui->printerComboBox->count()) i = 0;

        ui->printerComboBox->setCurrentIndex(i);
    }

    QRegExp re("^[^ ].+$");
    QRegExpValidator *v = new QRegExpValidator(re, this);
    ui->name->setValidator(v);

    connect(ui->colorComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), this,
        &CategoryEdit::colorComboChanged);
    connect(ui->okButton, &QPushButton::clicked, this, &CategoryEdit::accept);
    connect(ui->cancelButton, &QPushButton::clicked, this, &CategoryEdit::reject);
    connect(ui->name, &QLineEdit::textChanged, this, [this](const QString &newText) {
        ui->okButton->setDisabled(newText.isEmpty());
    });
}

CategoryEdit::~CategoryEdit()
{
    delete ui;
}

void CategoryEdit::colorComboChanged(int idx)
{

    QString colorValue = ui->colorComboBox->itemData(idx, Qt::BackgroundColorRole)
                             .toString(); // ->itemText(idx); //
                                          // ->model()->index(ui->colorComboBox->currentIndex(),
                                          // 0).data(Qt::BackgroundColorRole).toString();
    QPalette palette(ui->colorComboBox->palette());
    QColor color(colorValue);
    palette.setColor(QPalette::Active, QPalette::Button, color);
    palette.setColor(QPalette::Highlight, color);
    ui->colorComboBox->setPalette(palette);
}

//--------------------------------------------------------------------------------

void CategoryEdit::accept()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QString color = ui->colorComboBox->model()
                        ->index(ui->colorComboBox->currentIndex(), 0)
                        .data(Qt::BackgroundColorRole)
                        .toString();

    if (m_id == -1) {
        query.prepare(QString("INSERT INTO categories (name, color, printerid, visible) "
                              "VALUES(:name, :color, :printerid, :visible)"));
    } else {
        query.prepare(QString("UPDATE categories SET name=:name, color=:color, "
                              "printerid=:printerid, visible=:visible WHERE id=:id"));
        query.bindValue(":id", m_id);
    }

    int printerid = ui->printerComboBox->currentData().toInt();

    query.bindValue(":name", ui->name->text());
    query.bindValue(":color", color);
    query.bindValue(":visible", ui->visibleCheckBox->isChecked());

    if (printerid > 0)
        query.bindValue(":printerid", printerid);
    else
        query.bindValue(":printerid", QVariant(QVariant::Int));

    query.exec();

    QDialog::accept();
}

//--------------------------------------------------------------------------------
