/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "numerickeypad.h"
#include "preferences/settings.h"
#include "pushbutton.h"
#include "qbcmath/bcmath.h"

#include <QApplication>
#include <QDesktopWidget>
#include <QDoubleValidator>
#include <QGridLayout>
#include <QLineEdit>

NumericKeypad::NumericKeypad(bool full, QWidget *parent)
    : QWidget(parent)
{

    Settings settings;
    setHidden(true);
    QGridLayout *layout = new QGridLayout(this);
    layout->setMargin(0);

    QVBoxLayout *vboxLayout = new QVBoxLayout;
    m_lineEdit = new QLineEdit();
    m_lineEdit->setAlignment(Qt::AlignRight);

    m_digits = settings.value("decimalDigits", 2).toInt();
    if (!settings.value("useDecimalQuantity", false).toBool()) m_digits = 0;

    QSize buttonSize = settings.value("numpadButtonSize", QSize(30, 30)).toSize();
    bool smalldisplay = false;
    if (QApplication::desktop()->height() <= 768) {
        buttonSize.setHeight(30);
        buttonSize.setWidth(30);
        smalldisplay = true;
    }

    QString style = "QPushButton {"
                    "color: white;"
                    "border: 1px solid #199909;"
                    "border-radius: 6px;"
                    "background-color: rgba(0, 0, 232, 0.5);"
                    "background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 0.67, "
                    "stop: 0 rgba(22, 86, 232, 0.5), stop: 1 rgba(22, 86, 232, 1));"
                    "}"
                    "QPushButton:pressed {"
                    "color: white;"
                    "border: 1px solid #333333;"
                    "background-color: #222222;"
                    "background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 0.67, "
                    "stop: 0 #444444, stop: 1 #222222);"
                    "}";

    /*
    style = "background-color: qradialgradient(spread:pad, cx:0.499807, cy:0.489,
    radius:0.5, fx:0.499, fy:0.488909, stop:0.0795455 rgba(0, 147, 185, 255),
    stop:1 rgba(30, 30, 30, 255));" "border: 5px solid black;" "border-radius:
    6px;";
    */

    QString styleRed = "QPushButton {"
                       "color: white;"
                       "border: 1px solid #199909;"
                       "border-radius: 6px;"
                       "background-color: rgba(222, 0, 0, 1);"
                       "background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 0.67, "
                       "stop: 0 rgba(111, 0, 0, 0.5), stop: 1 rgba(255, 0, 0, 1));"
                       "}"
                       "QPushButton:pressed {"
                       "color: white;"
                       "border: 1px solid #333333;"
                       "background-color: #222222;"
                       "background-color: qlineargradient(x1: 0, y1: 0, x2: 0, y2: 0.67, "
                       "stop: 0 #444444, stop: 1 #222222);"
                       "}";

    for (int i = 1; i <= 9; ++i) {
        PushButton *singleButton = new PushButton();
        singleButton->setStyleSheet(style);
        singleButton->setFixedSize(buttonSize);
        singleButton->setText(QString::number(i));
        layout->addWidget(singleButton, (12 - i) / 3, (i - 1) % 3);
        QString text = QString().setNum(i);
        connect(singleButton, &QPushButton::clicked, this, [this, text]() {
            buttonClicked(text);
        });
    }

    PushButton *buttonZero = new PushButton("0");
    buttonZero->setStyleSheet(style);
    buttonZero->setFixedSize(buttonSize);

    PushButton *buttonDot = new PushButton(QLocale().decimalPoint());
    buttonDot->setStyleSheet(style);
    buttonDot->setFixedSize(buttonSize);

    PushButton *buttonClear = new PushButton(tr("C"));
    buttonClear->setStyleSheet(styleRed);
    //    buttonClear->setFixedSize(buttonSize);
    buttonClear->setMaximumHeight(16777215);
    buttonClear->setFixedWidth(buttonSize.width());
    buttonClear->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);

    PushButton *buttonPlusMinus = new PushButton(tr("+/-"));
    buttonPlusMinus->setStyleSheet(style);
    buttonPlusMinus->setFixedSize(buttonSize);

    if (full) {
        m_buttonCount = new PushButton(tr("Anzahl"));
        m_buttonCount->setStyleSheet("Text-align:left");
        m_buttonCount->setIcon(QIcon(":/ckvsoft/resources/icons/count.png"));
        m_buttonCount->setIconSize(QSize(32, 32));
        m_buttonCount->setFixedHeight(buttonSize.height());
        if (smalldisplay) m_buttonCount->setMinimumWidth(70);

        m_buttonSinglePrice = new PushButton(tr("E-Preis"));
        m_buttonSinglePrice->setStyleSheet("Text-align:left");
        m_buttonSinglePrice->setIcon(QIcon(":/ckvsoft/resources/icons/singleprice.png"));
        m_buttonSinglePrice->setIconSize(QSize(32, 32));
        m_buttonSinglePrice->setFixedHeight(buttonSize.height());
        if (smalldisplay) m_buttonSinglePrice->setMinimumWidth(70);

        m_buttonDiscount = new PushButton(tr("Rabatt"));
        m_buttonDiscount->setStyleSheet("Text-align:left");
        m_buttonDiscount->setIcon(QIcon(":/ckvsoft/resources/icons/discount.png"));
        m_buttonDiscount->setIconSize(QSize(32, 32));
        m_buttonDiscount->setFixedHeight(buttonSize.height());
        if (smalldisplay) m_buttonDiscount->setMinimumWidth(70);

        m_buttonPrice = new PushButton(tr("Preis"));
        m_buttonPrice->setStyleSheet("Text-align:left");
        m_buttonPrice->setIcon(QIcon(":/ckvsoft/resources/icons/sum.png"));
        m_buttonPrice->setIconSize(QSize(32, 32));
        m_buttonPrice->setFixedHeight(buttonSize.height());
        if (smalldisplay) m_buttonPrice->setMinimumWidth(70);

        vboxLayout->addWidget(m_buttonCount);
        vboxLayout->addWidget(m_buttonDiscount);
        vboxLayout->addWidget(m_buttonPrice);
        vboxLayout->addWidget(m_buttonSinglePrice);

        /*
        layout->addWidget( m_buttonCount, 1, 4, 1, 2);
        layout->addWidget( m_buttonDiscount, 2, 4, 1, 2);
        layout->addWidget( m_buttonPrice, 3, 4, 1, 2);
        layout->addWidget( m_buttonSinglePrice, 4, 4, 1, 2);
    */
        layout->addLayout(vboxLayout, 1, 4, 4, 1, Qt::AlignBottom);

        connect(m_buttonCount, &QPushButton::clicked, this, &NumericKeypad::setCount);
        connect(m_buttonSinglePrice, &QPushButton::clicked, this, &NumericKeypad::setSinglePrice);
        connect(m_buttonDiscount, &QPushButton::clicked, this, &NumericKeypad::setDiscount);
        connect(m_buttonPrice, &QPushButton::clicked, this, &NumericKeypad::setPrice);
    } else {
        PushButton *buttonBackspace = new PushButton();
        buttonBackspace->setIcon(QIcon(":/ckvsoft/resources/icons/backspace.png"));
        buttonBackspace->setIconSize(QSize(32, 32));
        buttonBackspace->setFixedSize(buttonSize);
        buttonBackspace->setStyleSheet(style);

        //       layout->addWidget( m_lineEdit, 0, 0, 1, 4 );
        //       layout->addWidget( buttonBackspace, 1, 3, 1, 1 );
        connect(buttonBackspace, &QPushButton::clicked, this, &NumericKeypad::backspace);
    }

    layout->addWidget(buttonPlusMinus, 4, 0);
    layout->addWidget(buttonZero, 4, 1);
    layout->addWidget(buttonDot, 4, 2);
    layout->addWidget(buttonClear, 3, 3, 2, 1);

    connect(buttonPlusMinus, &QPushButton::clicked, this, [this]() {
        buttonClicked("-");
    });
    connect(buttonZero, &QPushButton::clicked, this, [this]() {
        buttonClicked("0");
    });
    connect(buttonDot, &QPushButton::clicked, this, [this]() {
        buttonClicked(QLocale().decimalPoint());
    });
    connect(buttonClear, &QPushButton::clicked, m_lineEdit, &QLineEdit::clear);
    connect(m_lineEdit, &QLineEdit::textChanged, this, &NumericKeypad::setText);
}

const QString NumericKeypad::text() const
{
    return m_text;
}

void NumericKeypad::discountButtonSetEnabled(bool enabled)
{
    m_buttonDiscount->setEnabled(enabled);
}

void NumericKeypad::setDiscountButtonHidden(bool hidden)
{
    if (hidden)
        m_buttonDiscount->hide();
    else
        m_buttonDiscount->show();
}

void NumericKeypad::setCountButtonHidden(bool hidden)
{
    if (hidden)
        m_buttonCount->hide();
    else
        m_buttonCount->show();
}

void NumericKeypad::setSinglePriceButtonHidden(bool hidden)
{
    if (hidden)
        m_buttonSinglePrice->hide();
    else
        m_buttonSinglePrice->show();
}

void NumericKeypad::setPriceButtonHidden(bool hidden)
{
    if (hidden)
        m_buttonPrice->hide();
    else
        m_buttonPrice->show();
}

void NumericKeypad::buttonClicked(const QString &newText)
{
    if (newText == QLocale().decimalPoint() && m_text.indexOf(QLocale().decimalPoint()) > 0)
        return;
    else if (newText == QLocale().decimalPoint() && m_text.isEmpty()) {
        setText("0" + newText);
        return;
    }

    if (newText == "-" && m_text.indexOf('-') >= 0) {
        QString text = m_text;
        setText(text.remove("-"));
        return;
    } else if (newText == "-") {
        setText(newText + m_text);
        return;
    }

    setText(m_text + newText);
}

void NumericKeypad::setEditFocus()
{
    m_lineEdit->setFocus();
    clear();
}

void NumericKeypad::setText(const QString &newText)
{
    if (newText == m_text) return;

    m_text = newText;
    m_lineEdit->setText(m_text);

    emit textChanged(m_text);
}

void NumericKeypad::clear()
{
    m_lineEdit->setText("");
}

void NumericKeypad::backspace(bool)
{
    QString text = m_text;
    text.chop(1);
    setText(text);
}

void NumericKeypad::setCount(bool)
{
    if (m_lineEdit->text().isEmpty()) return;

    QBCMath text(QLocale().toDouble(m_lineEdit->text()));

    if (m_digits == 0) {
        emit valueButtonPressed(QString::number(text.toInt()), NUMPAD_VALUE_BUTTON_COUNT);
    } else {
        text.round(m_digits);
        emit valueButtonPressed(text.toString(), NUMPAD_VALUE_BUTTON_COUNT);
    }

    m_lineEdit->setText("");
}

void NumericKeypad::setDiscount(bool)
{
    if (m_lineEdit->text().isEmpty()) return;

    QBCMath text(QLocale().toDouble(m_lineEdit->text()));
    text.round(2);

    emit valueButtonPressed(text.toString(), NUMPAD_VALUE_BUTTON_DISCOUNT);
    m_lineEdit->setText("");
}

void NumericKeypad::setSinglePrice(bool)
{
    if (m_lineEdit->text().isEmpty()) return;

    QBCMath text(QLocale().toDouble(m_lineEdit->text()));
    text.round(2);

    emit valueButtonPressed(text.toString(), NUMPAD_VALUE_BUTTON_SINGLE);
    m_lineEdit->setText("");
}

void NumericKeypad::setPrice(bool)
{
    if (m_lineEdit->text().isEmpty()) return;

    QBCMath text(QLocale().toDouble(m_lineEdit->text()));
    text.round(2);

    emit valueButtonPressed(text.toString(), NUMPAD_VALUE_BUTTON_TOTAL);
    m_lineEdit->setText("");
}
