/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "primesignsmartcard.h"
#include <QDebug>

/*
        RK Suite defined in Detailspezifikation/ABS 2
        suite for a closed system (closed systems are identified by the ZDA-ID
   AT0) R1_AT0("1", "AT0", "ES256", "SHA-256", 8), suite for an open system (in
   this case with the virtual ZDA identified by AT100) R1_AT100("1", "AT100",
   "ES256", "SHA-256", 8);
*/

PrimeSignSmartCard::PrimeSignSmartCard(QString device_name, bool shared)
    : RKSignatureSmartCard(device_name, shared)
{
}

PrimeSignSmartCard::~PrimeSignSmartCard()
{
}

/**
 * @brief PrimeSignSmartCard::getMessage
 * @param code
 * @return
 */
QString PrimeSignSmartCard::getMessage(const unsigned char code[2])
{
    unsigned short error_code = (code[0] << 8) | code[1]; // korrektes Zusammenfügen
    QString hexCode = QString("%1").arg(error_code, 4, 16, QChar('0')).toUpper();

    QString germanMessage;
    QString englishMessage;

    if (error_code == 0x9000) {
        germanMessage = "Erfolg";
        englishMessage = "Success / OK";
    } else if (error_code == 0x6700) {
        germanMessage = "Falsche Länge";
        englishMessage = "Wrong length";
    } else if (error_code == 0x6982) {
        germanMessage = "Sicherheitsstatus nicht erfüllt";
        englishMessage = "Security status not satisfied";
    } else if (error_code == 0x6983) {
        germanMessage = "Authentifizierungsmethode blockiert (z.B. PIN-Fail)";
        englishMessage = "Authentication method blocked (e.g. PIN blocked)";
    } else if (error_code == 0x6985) {
        germanMessage = "Bedingungen nicht erfüllt";
        englishMessage = "Conditions of use not satisfied";
    } else if (error_code == 0x6A80) {
        germanMessage = "Ungültige Daten";
        englishMessage = "Incorrect data";
    } else if (error_code == 0x6A82) {
        germanMessage = "Datei nicht gefunden";
        englishMessage = "File not found";
    } else if (error_code == 0x6A83) {
        germanMessage = "Rekord nicht gefunden";
        englishMessage = "Record not found";
    } else if (error_code == 0x6A84) {
        germanMessage = "Nicht genug Speicherplatz";
        englishMessage = "Not enough memory space";
    } else if (error_code == 0x6D00) {
        germanMessage = "Ungültiger Befehl";
        englishMessage = "Instruction code not supported or invalid";
    } else if (error_code == 0x6E00) {
        germanMessage = "Ungültige Klasse";
        englishMessage = "Class not supported";
    } else if (error_code == 0x6F00) {
        germanMessage = "Unbekannter Fehler";
        englishMessage = "No precise diagnosis / General error";
    } else {
        germanMessage = "Unbekannter Fehlercode";
        englishMessage = "Unknown error code";
    }

    return QString("Fehlercode: 0x%1 - %2 / %3").arg(hexCode).arg(germanMessage).arg(englishMessage);
}

/**
 * @brief PrimeSignSmartCard::transmit
 * @param txBuffer
 * @param txLength
 * @return
 */
ASignResponse PrimeSignSmartCard::transmit(const unsigned char *txBuffer, DWORD txLength)
{
    ASignResponse result;
    memset(&result, 0, sizeof(ASignResponse));

    if (RKSignatureSmartCard::transmit(txBuffer, txLength, result.data, &result.length)) {
        result.code[0] = result.data[result.length - 2];
        result.code[1] = result.data[result.length - 1];
        result.length -= 2;
        if (result.code[0] != 0x90 || result.code[1] != '\0') {
            qCritical() << "Function Name: " << Q_FUNC_INFO << getMessage(result.code);
        }
    }

    return result;
}

/**
 * @brief PrimeSignSmartCard::selectApplication
 * @return
 */
bool PrimeSignSmartCard::selectApplication()
{
    if (!isCardPresent()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: "
                   << "There is no Smart card in the reader";
        return false;
    }
    if (!connect()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: "
                   << "There is a Smart card in the reader but the connection failed";
        return false;
    }
    // There is a Smart card in the reader and connected successful
    return true;
}

QByteArray PrimeSignSmartCard::ReadFile()
{
    QByteArray ba = 0;

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    DWORD offset = response.length;
    for (DWORD i = 0; i < response.length; i++) {
        ba.append(response.data[i]);
    }
    while (256 == response.length) {
        cmd[2] = (offset >> 8) & 0xFF;
        cmd[3] = offset & 0xFF;
        response = transmit(cmd, 7);
        for (DWORD i = 0; i < response.length; i++) {
            ba.append(response.data[i]);
        }
        offset += response.length;
    }

    return ba;
}
