/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "a_signacos_04.h"
#include "a_signacos_ID.h"
#include "a_signacos_ID_V41.h"
#include "a_signcardos_53.h"
#include "a_signonline.h"
#include "primesigncardos_53.h"
#include "primesignonline.h"

#include "rk_signaturemodulefactory.h"
#include "rk_smartcardinfo.h"

#include "pindialog.h"
#include "preferences/settings.h"

#include <QDebug>

/**
 * @brief RKSignatureModuleFactory::createInstance
 * @param reader
 * @return
 */
RKSignatureModule *RKSignatureModuleFactory::createInstance(QString reader, bool demomode)
{

    Settings settings;
    bool onlinetestmode = false;
    bool sharedmode = settings.value("scard_sharedmode", false).toBool();

    if (reader.isNull() || reader.isEmpty()) reader = settings.value("currentCardReader", "").toString();

    if (reader.isEmpty()) {
        if (demomode) {
            onlinetestmode = true;
#if QT_VERSION >= QT_VERSION_CHECK(5, 10, 0)
            if (QRandomGenerator::global()->bounded(2) == 0) {
                reader = "u123456789@123456789@https://hs-abnahme.a-trust.at/asignrkonline/v2";
                qDebug("Demo-Modus: A-Trust Testserver");
            } else {
                reader = "user123@KNJOJ-BLZDB-2QR2L-XPUUY-JR5Z6@https://rs-fc8349ca.ps.prime-sign.com";
                qDebug("Demo-Modus: PrimeSign Testserver");
            }
#else
            reader = "u123456789@123456789@https://hs-abnahme.a-trust.at/asignrkonline/v2";
            qDebug("Demo-Modus: A-Trust Testserver");
#endif
        } else {
            reader = settings.value("atrust_connection", "").toString();
            if (reader.isEmpty()) reader = settings.value("primesign_connection", "").toString();

            qDebug() << "Reader aus Settings:" << reader;
        }
    }

    RKSignatureModule *signatureModule = Q_NULLPTR;

    if (reader.isEmpty()) return new RKSmartCardInfo(reader, sharedmode);

    if (reader.split("@").size() == 3) {
        QStringList parts = reader.split("@");
        QString user = parts[0];
        QString secret = parts[1];
        QString url = parts[2];

        QUrl parsedUrl(url);
        QString host = parsedUrl.host();

        if (host.contains("a-trust.at", Qt::CaseInsensitive)) {
            (onlinetestmode) ? qDebug("A-Trust Online (Testserver)") : qDebug("A-Trust Online (Liveserver)");
            return new ASignOnline(reader);
        } else if (host.contains("prime-sign.com", Qt::CaseInsensitive)) {
            (onlinetestmode) ? qDebug("PrimeSign Online (Testserver)") : qDebug("PrimeSign Online (Liveserver)");
            return new PrimeSignOnline(reader);
        }
    }

    QString ATR;
    for (int i = 0; i < 3; i++) {
        ATR = getATR(reader, sharedmode);
        if (!ATR.isEmpty()) break;

        qDebug() << "Function Name: " << Q_FUNC_INFO << " Wait for ATR";
        QThread::msleep(200);
    }

    if (ATR == "3BBF11008131FE45455041000000000000000000000000F1" || ATR == "3BBF13008131FE45455041"
        || ATR == "3BBF11008131FE45455041000000000000000000000000F1") {
        // cardInstance = new SmartCard_ACOS(reader);
        signatureModule = new ASignACOS_04(reader, sharedmode);
        qDebug("Card: ACOS04");
    } else if (ATR == "3BDF18008131FE588031905241016405C903AC73B7B1D444") {
        // cardInstance = new SmartCard_CardOS_5_3(reader);
        signatureModule = new ASignCARDOS_53(reader, sharedmode);
        qDebug("Card: CARDOS 5.3");
    } else if (ATR.startsWith("3BDF96FF910131FE4680319052410264050200AC73D622C017")
        || ATR.startsWith("3BDF18FF910131FE4680319052410264050200AC73D622C099")
        || ATR.startsWith("3BDF97008131FE4680319052410364050201AC73D622C0F8")
        || ATR.startsWith("3BDF97008131FE468031905241026405C903AC73D622C030")
        || ATR.startsWith("3BDF96FF918131FE468031905241026405C903AC73D622C05F")
        || ATR.startsWith("3BDF18FF918131FE468031905241026405C903AC73D622C0D1")) {
        signatureModule = new ASignACOS_ID(reader, sharedmode);
        qDebug("Card: ACOS ID");
    } else if (ATR.startsWith("3BDF97008131FE4680319052410464050401AC73D622C0F9")) {
        signatureModule = new ASignACOS_IDV41(reader, sharedmode);
        qDebug("Card: ACOS ID V41");
    } else if (ATR.startsWith("3BDE18008131FE588062C903595072696D655369676ED1")) {
        signatureModule = new PrimeSignCARDOS_53(reader, sharedmode);
        signatureModule->selectApplication();
        QString serial = signatureModule->getCertificateSerial(true);
        QByteArray pin = signatureModule->getPinForSerial(serial);
        // Prüfen, ob Karte gesperrt ist
        if (pin == "blocked") {
            qWarning() << "Card: Pin is blocked";
            delete signatureModule;
            signatureModule = new RKSmartCardInfo(reader, sharedmode);
            return signatureModule;
        }

        if (pin.isEmpty()) {
            PinDialog dlg;
            dlg.setCardSerial(serial);
            if (dlg.exec() == QDialog::Accepted && dlg.isSuccess()) {
                pin = signatureModule->getPinForSerial(serial);
            } else {
                // Abbruch, keine Karte erzeugen
                signatureModule = new RKSmartCardInfo(reader, sharedmode);
            }
        }

        signatureModule->setPin(pin);
        bool blocked = false;
        if (!signatureModule->verifyPin(blocked)) {
            qWarning() << "Ungültige PIN – Karte wird nicht geladen";

            // Ungültige PIN aus der Datenbank löschen
            signatureModule->setPin(QByteArray()); // sicherstellen, dass kein weiterer PIN verwendet wird
            if (blocked)
                signatureModule->storePinForSerial(serial, QByteArray("blocked"));
            else
                signatureModule->storePinForSerial(serial, QByteArray());

            // Fallback auf Info-Modul
            delete signatureModule;
            signatureModule = new RKSmartCardInfo(reader, sharedmode);
        } else {
            qDebug() << "Card: CARDOS 5.3 PrimeSign mit gültiger PIN: " << pin;
        }
    } else {
        signatureModule = new RKSmartCardInfo(reader, sharedmode);
        qWarning() << "Keine oder falsche Karte! ATR:" << ATR;
    }

    signatureModule->selectApplication();
    return signatureModule;
}

/**
 * @brief RKSignatureModuleFactory::getATR
 * @param reader
 * @return
 */
QString RKSignatureModuleFactory::getATR(QString reader, bool sharedmode)
{
    RKSmartCardInfo sc(reader, sharedmode);
    QString ATR = sc.getATR();
    return ATR;
}
