/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "backupworker.h"
#include "qrkprogress.h"
#if defined(_WIN32) || defined(__APPLE__)
#    include "3rdparty/fervor-autoupdate/fvupdater.h"
#endif

#include "3rdparty/ckvsoft/RK/rk_signaturemodulefactory.h"
#include "3rdparty/ckvsoft/databasemanager.h"
#include "3rdparty/ckvsoft/preferences/registrationtab.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/rbac/aclmanager.h"
#include "3rdparty/ckvsoft/rbac/aclwizard.h"
#include "3rdparty/ckvsoft/singleton/spreadsignal.h"
#include "aboutdlg.h"
#include "backup.h"
#include "database.h"
#include "defines.h"
#include "export/exportdep.h"
#include "export/exportjournal.h"
#include "export/exportproducts.h"
#include "foninfo.h"
#include "import/csvimportwizard.h"
#include "manager/managerdialog.h"
#include "pluginmanager/Interfaces/independentinterface.h"
#include "pluginmanager/pluginmanager.h"
#include "pluginmanager/pluginview.h"
#include "preferences/settingsdialog.h"
#include "qrk.h"
#include "qrkdocument.h"
#include "qrkhome.h"
#include "qrkprinter.h"
#include "qrkregister.h"
#include "qrksettings.h"
#include "qrktimedmessagebox.h"
#include "qrkuserlogin.h"
#include "qrkwaiterlockacs.h"
#include "reports.h"
#include "utils/demomode.h"
#include "utils/utils.h"
#include "utils/versionchecker.h"
#include "utils/waiterlock.h"

#if defined(QRK_GASTRO)
#    include "qrkgastro.h"
#endif

#include <ui_qrk.h>

#include <QCloseEvent>
#include <QDebug>
#include <QDesktopServices>
#include <QDir>
#include <QLCDNumber>
#include <QLabel>
#include <QMessageBox>
#include <QProcess>
#include <QProgressBar>
#include <QScreen>
#include <QStackedWidget>
#include <QThread>
#include <QTimer>
#include <QTreeView>

//-----------------------------------------------------------------------

QRK::QRK(bool importmode, QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::MainWindow)
    , m_importmode(importmode)

{

    connect(Spread::Instance(), &SpreadSignal::updateProgressBar, this, &QRK::setStatusBarProgressBar);
    connect(Spread::Instance(), &SpreadSignal::updateProgressBarWait, this, &QRK::setStatusBarProgressBarWait);
    connect(Spread::Instance(), &SpreadSignal::updateSafetyDevice, this, &QRK::setSafetyDevice);

    connect(RBAC::Instance(), static_cast<void (Acl::*)()>(&Acl::userChanged), this, &QRK::init);
    connect(RBAC::Instance(), static_cast<void (Acl::*)()>(&Acl::userChanged), this, &QRK::userChanged);

    ui->setupUi(this);

#if defined(_WIN32) || defined(__APPLE__)
    connect(ui->actionQRK_Software_Update, &QAction::triggered, [=]() {
        FvUpdater::sharedUpdater()->CheckForUpdatesNotSilent();
    });

    ui->actionQRK_Software_Update->setVisible(true);
#endif

    m_dep = new QLabel(this);
    m_dep->setMinimumSize(m_dep->sizeHint());
    m_depPX = new QLabel(this);
    m_depPX->setMinimumSize(m_depPX->sizeHint());
    m_safetyDevicePX = new QLabel(this);
    m_safetyDevicePX->setMinimumSize(m_safetyDevicePX->sizeHint());

    m_currentRegisterYearLabel = new QLabel(this);
    m_currentRegisterYearLabel->setMinimumSize(m_currentRegisterYearLabel->sizeHint());
    m_currentRegisterYearLabel->setToolTip(tr("Kassenjahr"));
    m_cashRegisterIdLabel = new QLabel(this);
    m_cashRegisterIdLabel->setMinimumSize(m_cashRegisterIdLabel->sizeHint());
    m_cashRegisterIdLabel->setToolTip(tr("Kassenidentifikationsnummer"));

    m_progressBar = new QProgressBar(this);
    m_dateLcd = new QLCDNumber(this);
    m_dateLcd->setDigitCount(20);
    m_dateLcd->setMode(QLCDNumber::Dec);
    m_dateLcd->setSegmentStyle(QLCDNumber::Flat);
    m_dateLcd->setFrameStyle(QFrame::NoFrame);

    QLabel *configName = new QLabel(QrkSettings::getConfigName(), this);
    configName->setForegroundRole(QPalette::ButtonText);
    statusBar()->addWidget(configName, 0);
    statusBar()->addPermanentWidget(m_dep, 0);
    statusBar()->addPermanentWidget(m_depPX, 0);
    statusBar()->addPermanentWidget(m_safetyDevicePX, 0);
    statusBar()->addPermanentWidget(m_cashRegisterIdLabel, 0);
    statusBar()->addPermanentWidget(m_currentRegisterYearLabel, 0);
    statusBar()->addPermanentWidget(m_progressBar, 0);
    statusBar()->addPermanentWidget(m_dateLcd, 0);

    m_checkDateTime = QDateTime::currentDateTime();
    m_timer = new QTimer(this);
    connect(m_timer, &QTimer::timeout, this, &QRK::timerDone);

    m_currentRegisterYear = QDateTime::currentDateTime().toString("yyyy").toInt();
    // setWindowFlags(Qt::CustomizeWindowHint | Qt::WindowTitleHint);

#ifdef QRK_PROFEATURES
    if (RegistrationTab::isActive(qApp->property("appBaseName").toString())) {
        RegistrationTab reg(qApp->property("appBaseName").toString(), true, this);
        int days;
        if (reg.isValid(days)) {
            PluginManager::instance()->initialize();
            initPlugins();
        }
    } else {
        QAction *action = ui->menuPlugins->addAction(tr("Dieser Menüpunkt ist nur in der GASTRO und PRO Version "
                                                        "verfügbar."));
        connect(action, &QAction::triggered, [this]() {
            RegistrationTab registration(qApp->property("appBaseName").toString(), false, this);
            QJsonObject data;
            data["cid"] = Database::getCashRegisterId();
            data["master"]
                = Database::getShopName() + "\n" + Database::getShopMasterData().trimmed().replace("\r\n", "\n");
            registration.setAlternateWebJsonData(data);
            registration.setShowActivateMessageBox(false);
            registration.callWeb();
        });
#else
    ui->menuPlugins->menuAction()->setVisible(false);
#endif
#ifdef QRK_PROFEATURES
    }
#endif

    // Stacked Widget
    m_stackedWidget = new QStackedWidget(this);
    this->setCentralWidget(m_stackedWidget);

    qDebug() << "Function Name: " << Q_FUNC_INFO << "timer start: " << m_timer;
    m_timer->start(1000);

    iniStack();

    connect(ui->export_CSV, &QAction::triggered, this, &QRK::export_CSV);
    connect(ui->import_CSV, &QAction::triggered, this, &QRK::import_CSV);
    connect(ui->export_JSON, &QAction::triggered, this, &QRK::export_JSON);
    connect(ui->exportProducts_CSV, &QAction::triggered, this, &QRK::exportProducts_CSV);
    connect(ui->actionDEPexternalBackup, &QAction::triggered, this, &QRK::backupDEP);
    connect(ui->actionDatenbank_sichern, &QAction::triggered, this, &QRK::backup);
    connect(ui->actionQRK_Beenden, &QAction::triggered, this, &QRK::exitSlot);
    connect(ui->actionEinstellungen, &QAction::triggered, m_qrk_home, &QRKHome::settingsSlot);
    connect(ui->actionVollbild, &QAction::triggered, this, &QRK::fullScreenSlot);

    connect(ui->actionAbout_QRK, &QAction::triggered, this, &QRK::actionAbout_QRK);
    connect(ui->actionAbout_QT, &QAction::triggered, qApp, &QApplication::aboutQt);
    connect(ui->actionQRK_Forum, &QAction::triggered, this, &QRK::actionQRK_Forum);
    connect(ui->actionDEMO_Daten_zur_cksetzen, &QAction::triggered, this, &QRK::actionResetDemoData);
    connect(ui->actionDEMOMODUS_Verlassen, &QAction::triggered, this, &QRK::actionLeaveDemoMode);
    connect(ui->actionInfos_zur_Registrierung_bei_FON, &QAction::triggered, this, &QRK::infoFON);
    connect(ui->actionResuscitationCashRegister, &QAction::triggered, this, &QRK::actionResuscitationCashRegister);
    connect(ui->actionAclManager, &QAction::triggered, this, &QRK::actionAclManager);

    connect(ui->actionArtikel_Manager, &QAction::triggered, this, &QRK::onManagerButton_clicked);
    connect(ui->actionBenutzer_Manager, &QAction::triggered, this, &QRK::actionAclManager);

    connect(this, &QRK::setImportMode, m_qrk_home, &QRKHome::setImportMode);
    connect(m_qrk_home, &QRKHome::endOfDay, this, &QRK::endOfDaySlot);
    connect(m_qrk_home, &QRKHome::endOfMonth, this, &QRK::endOfMonthSlot);
    connect(m_qrk_home, &QRKHome::fullScreenButton_clicked, this, &QRK::fullScreenSlot);
    connect(m_qrk_home, &QRKHome::exitButton_clicked, this, &QRK::exitSlot);
    connect(m_qrk_home, &QRKHome::logOnOff, this, &QRK::logOnOff);

    connect(m_qrk_home, &QRKHome::documentButton_clicked, this, &QRK::onDocumentButton_clicked);
    connect(m_qrk_home, &QRKHome::productmanagerButton_clicked, this, &QRK::onManagerButton_clicked);
    connect(m_qrk_home, &QRKHome::usermanagerButton_clicked, this, &QRK::actionAclManager);
    connect(m_qrk_home, &QRKHome::refreshMain, this, &QRK::init);
    connect(m_qrk_home, &QRKHome::registerButton_clicked, this, &QRK::onRegisterButton_clicked);

#if defined(QRK_GASTRO)
    connect(m_qrk_gastro, &QRKGastro::cancelGastroButton_clicked, this, &QRK::onCancelRegisterButton_clicked);
#else
    connect(m_qrk_register, &QRKRegister::cancelRegisterButton_clicked, this, &QRK::onCancelRegisterButton_clicked);
    connect(m_qrk_register, &QRKRegister::finishedReceipt, this, &QRK::finishedReceipt);
    connect(m_qrk_register, &QRKRegister::fullScreen, this, &QRK::setFullScreenMode);
    connect(m_qrk_register, &QRKRegister::sendDatagram, this, &QRK::sendDatagram);
#endif

    connect(m_qrk_document, &QRKDocument::cancelDocumentButton, this, &QRK::onCancelDocumentButton_clicked);
    connect(m_qrk_document, &QRKDocument::documentButton_clicked, this, &QRK::onDocumentButton_clicked);

    connect(ui->actionUser_1_test, &QAction::triggered, this, &QRK::logOnOff);

    QString title = QString("%1 V%2.%3 - Qt Registrier Kasse - %4")
                        .arg(qApp->applicationName())
                        .arg(QRK_VERSION_MAJOR)
                        .arg(QRK_VERSION_MINOR)
                        .arg(Database::getShopName());
    setWindowTitle(title);

    init();
    restore();

    m_checkerThread = new QThread;
    m_versionChecker = new VersionChecker;
    m_versionChecker->moveToThread(m_checkerThread);
    connect(m_versionChecker, &VersionChecker::Version, this, &QRK::newApplicationVersionAvailable);
    connect(m_checkerThread, &QThread::started, m_versionChecker, &VersionChecker::run);
    m_checkerThread->start();

    startACS();
}

//--------------------------------------------------------------------------------

QRK::~QRK()
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << "timer stop: " << m_timer;
    m_timer->stop();
    m_versionChecker->deleteLater();
    if (m_waiterLock) {
        m_waiterLock->deleteLater();
    }
    //    DatabaseManager::removeCurrentThread("CN");
    DatabaseManager::clear();

    delete ui;
    PluginManager::instance()->uninitialize();
    QRKPrinter::clearAllGlobalResources();

    qInfo() << "Function Name: " << Q_FUNC_INFO << "stopping QRK";
}

void QRK::startACS()
{
    QrkSettings settings;
    if (settings.value("Acs/waiterlock_enabled").toBool()) {
        WaiterLock::WaiterLockType type;
        switch (settings.value("Acs/waiterlock_type").toInt()) {
            case 0:
                type = WaiterLock::ADDIMAT;
                break;
            case 1:
                type = WaiterLock::DALLAS;
                break;
            default:
                type = WaiterLock::ADDIMAT;
        }
        if (!m_waiterLock) {
            m_acsThread = new QThread;
            m_waiterLock = new WaiterLock;
            m_waiterLock->setData(type, settings.value("Acs/waiterlock_port").toString(), 10);
            m_waiterLock->moveToThread(m_acsThread);
            connect(m_waiterLock, &WaiterLock::request, this, &QRK::waiterLockRequest);
            connect(m_acsThread, &QThread::started, m_waiterLock, &WaiterLock::run);
            m_acsThread->start();
        } else {
            m_waiterLock->setData(type, settings.value("Acs/waiterlock_port").toString(), 10);
        }
    } else {
        if (m_waiterLock) {
            m_waiterLock->deleteLater();
            m_waiterLock = Q_NULLPTR;
        }
    }
}

//--------------------------------------------------------------------------------

void QRK::iniStack()
{
    m_qrk_home = new QRKHome(m_importmode, m_stackedWidget);
    m_stackedWidget->addWidget(m_qrk_home);

#if defined(QRK_GASTRO)
    m_qrk_gastro = new QRKGastro(m_stackedWidget);
    m_stackedWidget->addWidget(m_qrk_gastro);
#else
    m_qrk_register = new QRKRegister(m_stackedWidget);
    m_stackedWidget->addWidget(m_qrk_register);
#endif
    m_qrk_document = new QRKDocument(m_stackedWidget);
    m_stackedWidget->addWidget(m_qrk_document);

    m_stackedWidget->setCurrentWidget(m_qrk_home);
}

//--------------------------------------------------------------------------------

void QRK::initPlugins()
{

    QStringList plugins = PluginManager::instance()->plugins();
    qDebug() << "Function Name: " << Q_FUNC_INFO << "Plugins: " << plugins;
    QListIterator<QString> i(plugins);
    bool hasPlugins = false;
    QList<QAction *> actions;
    while (i.hasNext()) {
        QString path = i.next();
        QString name = PluginManager::instance()->getNameByPath(path);
        qInfo() << "Function Name: " << Q_FUNC_INFO << "Load plugins: " << name << " (" << path << ")";
        IndependentInterface *plugin
            = qobject_cast<IndependentInterface *>(PluginManager::instance()->getObjectByName(name));
        if (plugin) {
            QAction *action = new QAction(plugin->getPluginName(), this);
            action->setObjectName(name);
            actions << action;
            connect(action, &QAction::triggered, this, &QRK::runPlugin);
            /*
                        QAction *action = ui->menuPlugins->addAction(plugin->getPluginName(), this, SLOT(runPlugin()));
                        action->setObjectName(name);
                        addAction(action);
            */
            // delete plugin;
            hasPlugins = true;
        }
    }
    if (hasPlugins) {
        std::sort(actions.begin(), actions.end(), [](QAction *a, QAction *b) {
            return a->objectName() > b->objectName();
        });

        ui->menuPlugins->addActions(actions);
        ui->menuPlugins->addSeparator();
    }

    ui->menuPlugins->addAction(tr("Plugins ..."), this, SLOT(viewPlugins()));
    // ui->menuPlugins->addAction(tr("Plugins ..."), this, &QRK::viewPlugins);
}

void QRK::runPlugin()
{
    QAction *action = qobject_cast<QAction *>(QObject::sender());
    QString name = action->objectName();
    if (!RBAC::Instance()->hasPermission("plugin_run_" + name, true)) return;
    IndependentInterface *plugin
        = qobject_cast<IndependentInterface *>(PluginManager::instance()->getObjectByName(name));
    if (plugin) {
        QMap<QString, QVariant> arguments;
        plugin->process(arguments);
        // delete plugin;
        // plugin = Q_NULLPTR;
        emit init();
    }
}

void QRK::viewPlugins()
{
    PluginView view(this);
    view.setCloseButtonVisible(true);
    view.exec();
    emit init();
}

void QRK::timerDone()
{
    QDateTime t = QDateTime::currentDateTime();
    m_dateLcd->display(t.toString("dd-MM-yyyy  hh:mm:ss"));
    if (m_checkDateTime.secsTo(t) < 0) DateTimeCheck();

    m_checkDateTime = t;
}

//--------------------------------------------------------------------------------

void QRK::DateTimeCheck()
{
    // DateTime check
    if (Database::getLastJournalEntryDate().secsTo(QDateTime::currentDateTime()) < 0) {
        QMessageBox messageBox(QMessageBox::Critical, QObject::tr("Eventueller Datum/Uhrzeit Fehler"),
            QObject::tr("ACHTUNG! Die Uhrzeit des Computers ist eventuell "
                        "falsch.\n\nLetzter Datenbankeintrag: %1\nDatum/Uhrzeit: %2")
                .arg(Database::getLastJournalEntryDate().toString())
                .arg(QDateTime::currentDateTime().toString()),
            QMessageBox::Yes | QMessageBox::No, this);
        messageBox.setButtonText(QMessageBox::Yes, QObject::tr("QRK beenden?"));
        messageBox.setButtonText(QMessageBox::No, QObject::tr("Weiter machen"));
        messageBox.setWindowFlags(messageBox.windowFlags() | Qt::WindowStaysOnTopHint);

        if (messageBox.exec() == QMessageBox::Yes) {
            QrkSettings settings;
            settings.removeSettings("QRK_RUNNING", false);
            if (!isFullScreen()) {
                settings.save2Settings("mainWindowGeometry", saveGeometry(), false);
                settings.save2Settings("mainWindowState", saveState(), false);
            }
            PluginManager::instance()->uninitialize();
            // QApplication::exit();
            qApp->quit();
        }
    }
}

void QRK::setStatusBarProgressBar(int value, bool add)
{
    if (value < 0) {
        if (m_progressBar->value() > -1) m_progressBar->setValue(100);

        QTimer::singleShot(2000, m_progressBar, SLOT(reset()));
        if (QApplication::overrideCursor()) QApplication::restoreOverrideCursor();

    } else {
        if (!QApplication::overrideCursor()) QApplication::setOverrideCursor(Qt::BusyCursor);

        if (add) value = m_progressBar->value() + value;
        m_progressBar->setValue(value);
    }
}

void QRK::setStatusBarProgressBarWait(bool on_off)
{
    m_progressBar->setMinimum(0);
    if (on_off) {
        m_progressBar->setMaximum(0);
        m_progressBar->setValue(0);

        if (!QApplication::overrideCursor()) QApplication::setOverrideCursor(Qt::BusyCursor);
    } else {
        m_progressBar->setMaximum(100);
        m_progressBar->reset();

        if (QApplication::overrideCursor()) QApplication::restoreOverrideCursor();
    }
}

void QRK::setSafetyDevice(bool active)
{
    if (active != m_previousSafetyDeviceState) {
        m_previousSafetyDeviceState = active;
        init();
    }
}

//--------------------------------------------------------------------------------

void QRK::init()
{
    startACS();
    m_currentRegisterYearLabel->setText(QObject::tr(" KJ: %1 ").arg(getCurrentRegisterYear()));
    m_cashRegisterIdLabel->setText(QObject::tr(" KID: %1 ").arg(Database::getCashRegisterId()));

    ui->menuDEMOMODUS->setEnabled(DemoMode::isDemoMode());
    ui->menuDEMOMODUS->menuAction()->setVisible(DemoMode::isDemoMode());
    ui->export_JSON->setVisible(RKSignatureModule::isDEPactive());
    ui->actionInfos_zur_Registrierung_bei_FON->setVisible(RKSignatureModule::isDEPactive() && !DemoMode::isDemoMode());

    setShopName();
    m_cashRegisterId = Database::getCashRegisterId();

    emit m_qrk_home->init();

    /**
     * @brief DEPaktive
     */
    m_dep->setText(tr("DEP-7"));
    if (Database::getTaxLocation() == "AT") {
        m_qrk_home->setExternalDepLabels(true);
        m_dep->setVisible(true);
        m_depPX->setVisible(true);
        m_safetyDevicePX->setVisible(true);
        emit updateRKStatus();
        qApp->processEvents();
    } else {
        m_qrk_home->setExternalDepLabels(false);
        m_dep->setVisible(false);
        m_depPX->setVisible(false);
        m_safetyDevicePX->setVisible(false);
    }
}

void QRK::updateRKStatus()
{
    QString DEPtoolTip = tr("DEP-7 Inaktiv");
    QPixmap pm1(32, 32);
    QPixmap pm2(32, 32);

    if (RKSignatureModule::isDEPactive()) {
        RKSignatureModule *signaturinfo = RKSignatureModuleFactory::createInstance("", DemoMode::isDemoMode());

        if (signaturinfo->selectApplication()) {
            m_previousSafetyDeviceState = true;
            QString serial = signaturinfo->getCertificateSerial(true);
            QString cardType = signaturinfo->getCardType();
            QString expiryText = signaturinfo->getExpireInfo();
            if (!expiryText.isEmpty()) {
                QMessageBox mb(tr("Karte abgelaufen"), expiryText, QMessageBox::Warning, QMessageBox::Yes,
                    QMessageBox::NoButton, QMessageBox::NoButton, this);
                mb.setButtonText(QMessageBox::Yes, tr("Ok"));
                mb.exec();
            }
            DEPtoolTip = tr("DEP-7 (RKSV Daten Erfassungs Protokoll) aktiv, "
                            "Kartentype: %1 Seriennummer: %2")
                             .arg(cardType)
                             .arg(serial);
            pm2.fill(Qt::green);
            m_safetyDevicePX->setToolTip(tr("SignaturErstellungsEinheit aktiv, "
                                            "Kartentype: %1 Seriennummer: %2")
                    .arg(cardType)
                    .arg(serial));
#if not defined(QRK_GASTRO)
            m_qrk_register->safetyDevice(true);
#endif
            m_qrk_home->safetyDevice(true);
        } else {
            m_previousSafetyDeviceState = false;
            DEPtoolTip = tr("DEP-7 (RKSV Daten Erfassungs Protokoll) aktiv, "
                            "SignaturErstellungsEinheit ausgefallen");
            pm2.fill(Qt::red);
            m_safetyDevicePX->setToolTip(tr("SignaturErstellungsEinheit ausgefallen"));
#if not defined(QRK_GASTRO)
            m_qrk_register->safetyDevice(false);
#endif
            m_qrk_home->safetyDevice(false);
        }

        pm1.fill(Qt::green);
        m_depPX->setPixmap(pm1);
        m_depPX->setToolTip(tr("DEP-7 (RKSV Daten Erfassungs Protokoll) aktiv"));
        m_safetyDevicePX->setPixmap(pm2);

        delete signaturinfo;
    } else {
        pm1.fill(Qt::red);
        m_depPX->setPixmap(pm1);
        m_safetyDevicePX->setVisible(false);
    }
    m_dep->setToolTip(DEPtoolTip);
}

void QRK::restore()
{
    QrkSettings settings;
    restoreGeometry(settings.value("mainWindowGeometry").toByteArray());
    restoreState(settings.value("mainWindowState").toByteArray());
}

void QRK::logOnOff(bool)
{
    if (RBAC::Instance()->Login()) {
        if (m_insideLogin) return;

        if (RBAC::Instance()->getUserId() != 0) {
            m_insideLogin = true;
            RBAC::Instance()->setuserId(-1);
            m_login = new QrkUserLogin(this);
            connect(m_login, &QrkUserLogin::accepted, m_login, &QrkUserLogin::hide, Qt::DirectConnection);
            connect(this, &QRK::waiterLockAccept, m_login, &QrkUserLogin::WaiterLock);

            if (m_login->exec() == QDialog::Rejected) {
                QTimer::singleShot(1, this, SLOT(exitSlot()));
                m_forceexit = true;
            }
            m_insideLogin = false;
        }
    } else {
        RBAC::Instance()->setuserId(0);
    }
}

void QRK::setResuscitationCashRegister(bool visible)
{
    ui->menuNEUE_KASSE_ERSTELLEN->setEnabled(visible);
    ui->menuNEUE_KASSE_ERSTELLEN->menuAction()->setVisible(visible);
}

//--------------------------------------------------------------------------------

void QRK::setShopName()
{
    m_shopName = Database::getShopName();
    m_cashRegisterIdLabel->setText(QObject::tr(" KID: %1 ").arg(Database::getCashRegisterId()));
}

//--------------------------------------------------------------------------------

void QRK::actionAclManager()
{
    if (!RBAC::Instance()->hasPermission("admin_access", true)) return;
    AclWizard::createFirstRoleAndUser();
    if (RBAC::Instance()->getAllUsers().isEmpty()) return;

    int days;
    AclManager am;
    QrkWaiterLockACS acs;
    if (m_waiterLock) disconnect(m_waiterLock, &WaiterLock::request, Q_NULLPTR, Q_NULLPTR);

    if (RegistrationTab(qApp->property("appBaseName").toString(), false).isValid(days)) {
        acs.setWaiterLock(m_waiterLock);
        am.setACS(&acs);
    }
    am.exec();
    if (m_waiterLock) connect(m_waiterLock, &WaiterLock::request, this, &QRK::waiterLockRequest);
    emit init();
}

void QRK::actionAbout_QRK()
{
    AboutDlg dlg;
    dlg.exec();
}

void QRK::actionQRK_Forum()
{
    QString link = "https://www.ckvsoft.at/forum";
    QDesktopServices::openUrl(QUrl(link));
}

//--------------------------------------------------------------------------------
void QRK::import_CSV()
{
    if (!RBAC::Instance()->hasPermission("import_csv", true)) return;

    CsvImportWizard importWizard(this);
    importWizard.exec();
}

void QRK::export_CSV()
{
    ExportJournal xport;
    xport.Export();
}

void QRK::export_JSON()
{
    ExportDEP xport;
    xport.dExport();
}

void QRK::exportProducts_CSV()
{
    ExportProducts xport;
    xport.Export();
}

void QRK::infoFON()
{
    setStatusBarProgressBarWait(true);
    ui->menubar->setEnabled(false);
    FONInfo finfo;
    setStatusBarProgressBarWait(false);
    ui->menubar->setEnabled(true);
    finfo.exec();
}

//--------------------------------------------------------------------------------

void QRK::endOfDaySlot()
{
    if (RBAC::Instance()->hasPermission("tasks_create_eod", true)) {
        if (ignoreOpenTickets()) {
            Reports rep;
            rep.endOfDay();
        }
    }
}

//--------------------------------------------------------------------------------

void QRK::endOfMonthSlot()
{
    if (RBAC::Instance()->hasPermission("tasks_create_eom", true)) {
        if (ignoreOpenTickets()) {
            Reports rep;
            rep.endOfMonth();
        }
    }
}

bool QRK::ignoreOpenTickets()
{
#ifdef QRK_GASTRO
    QrkSettings settings;
    bool ignore = settings.value("ignore_open_tickets", false).toBool();
    if (!QRKGastro::openTickets() || ignore) return true;

    bool mustDo = Reports().mustDoEOAny(QDateTime::currentDateTime());
    if (QTime::currentTime() > QTime(8, 0, 0) && mustDo) return true;

    QMessageBox mb(tr("Offene Bonierung"),
        tr("Es gibt noch offenen Bonierungen. Bitte diese vor dem Abschluss abrechnen."), QMessageBox::Warning,
        QMessageBox::Yes, QMessageBox::NoButton, QMessageBox::NoButton, this);
    mb.setButtonText(QMessageBox::Yes, tr("Ok"));
    QStringList list = QRKGastro::openTicketsList();
    mb.setDetailedText(list.join("\n"));
    mb.exec();

    return false;
#else
    return true;
#endif
}

void QRK::onCancelDocumentButton_clicked()
{
    m_stackedWidget->setCurrentWidget(m_qrk_home);
    m_qrk_home->init();
}

//--------------------------------------------------------------------------------

#if defined(QRK_GASTRO)
void QRK::onRegisterButton_clicked()
{
    if (!RBAC::Instance()->hasPermission("gastro_register_access")) {
        QMessageBox::warning(Q_NULLPTR, tr("Information!"),
            tr("Leider haben Sie keine Berechtigung.\nFehlende Berechtigung '%1'")
                .arg(RBAC::Instance()->getPermNameFromID(RBAC::Instance()->getPermIDfromKey("gastro_register_"
                                                                                            "access"))));
        return;
    }

    if (m_stackedWidget->currentWidget() != m_qrk_gastro) {
        if (m_qrk_gastro->init()) m_stackedWidget->setCurrentWidget(m_qrk_gastro);
    }
}

void QRK::onCancelRegisterButton_clicked()
{
    m_stackedWidget->setCurrentWidget(m_qrk_home);
    m_qrk_home->init();
}

#else
void QRK::onRegisterButton_clicked()
{

    if (!RBAC::Instance()->hasPermission("register_access")) {
        QMessageBox::warning(Q_NULLPTR, tr("Information!"),
            tr("Leider haben Sie keine Berechtigung.\nFehlende Berechtigung '%1'")
                .arg(RBAC::Instance()->getPermNameFromID(RBAC::Instance()->getPermIDfromKey("register_access"))));
        return;
    }

    m_qrk_register->init();
    m_qrk_register->newOrder();
    m_stackedWidget->setCurrentWidget(m_qrk_register);
}

void QRK::onCancelRegisterButton_clicked()
{
    m_qrk_register->clearModel();
    m_stackedWidget->setCurrentWidget(m_qrk_home);
    m_qrk_home->init();
}

void QRK::finishedReceipt()
{
    // m_qrk_register->clearModel();
    m_qrk_home->init();
    //    m_qrk_register->init();
    m_qrk_register->newOrder();
}
#endif

void QRK::onManagerButton_clicked()
{
    if (!RBAC::Instance()->hasPermission("manager_access")) {
        QMessageBox::warning(Q_NULLPTR, tr("Information!"),
            tr("Leider haben Sie keine Berechtigung.\nFehlende Berechtigung '%1'")
                .arg(RBAC::Instance()->getPermNameFromID(RBAC::Instance()->getPermIDfromKey("manager_access"))));
        return;
    }

    ManagerDialog manager;
    manager.exec();
}

//--------------------------------------------------------------------------------

void QRK::onDocumentButton_clicked()
{
    if (!RBAC::Instance()->hasPermission("documents_access")) {
        QMessageBox::warning(Q_NULLPTR, tr("Information!"),
            tr("Leider haben Sie keine Berechtigung.\nFehlende Berechtigung '%1'")
                .arg(RBAC::Instance()->getPermNameFromID(RBAC::Instance()->getPermIDfromKey("documents_access"))));
        return;
    }

    m_stackedWidget->setCurrentWidget(m_qrk_document);
    m_qrk_document->documentList(m_qrk_home->isImportMode());
}

//--------------------------------------------------------------------------------

void QRK::fullScreenSlot()
{
    QrkSettings settings;
    if (isFullScreen()) {
        ui->actionVollbild->setChecked(false);
        setWindowState(windowState() ^ Qt::WindowFullScreen);
        restoreGeometry(settings.value("mainWindowGeometry").toByteArray());
        restoreState(settings.value("mainWindowState").toByteArray());
    } else {
        settings.save2Settings("mainWindowGeometry", saveGeometry(), false);
        settings.save2Settings("mainWindowState", saveState(), false);
        showFullScreen();
    }
}

void QRK::setFullScreenMode(bool set)
{
    QrkSettings settings;
    if (set) {
        settings.save2Settings("numKeyPadGeometry", saveGeometry(), false);
        settings.save2Settings("numKeyPadState", saveState(), false);
        showFullScreen();
    } else {
        //        setWindowState(windowState() ^ Qt::WindowFullScreen);
        restoreGeometry(settings.value("numKeyPadGeometry").toByteArray());
        restoreState(settings.value("numKeyPadState").toByteArray());
    }
}

//--------------------------------------------------------------------------------

void QRK::closeEvent(QCloseEvent *event)
{
    if (m_qrk_home->isImportMode()) {
        emit setImportMode(false);
        connect(m_qrk_home, &QRKHome::importmodefinished, this, &QRK::exitSlot);

        event->ignore();
        return;
    }

    bool openTickets = false;
#ifdef QRK_GASTRO
    openTickets = QRKGastro::openTickets();
#endif

    disconnect(m_qrk_home, &QRKHome::importmodefinished, Q_NULLPTR, Q_NULLPTR);

    if (!m_forceexit) {
        QString messageText = openTickets ? tr("Es gibt noch offene Bonierungen. Möchten Sie wirklich beenden ?")
                                          : tr("Möchten Sie wirklich beenden ?");
        QrkTimedMessageBox mb(5, tr("Beenden"), messageText, QMessageBox::Question,
            QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape, QMessageBox::NoButton,
            this);
        mb.setButtonText(QMessageBox::Yes, tr("Ja"));
        mb.setButtonText(QMessageBox::No, tr("Nein"));
        mb.setDefaultButton(QMessageBox::Yes);

        if (mb.exec() != QMessageBox::Yes) {
            event->ignore();
            return;
        }
    }
    QrkSettings settings;
    settings.removeSettings("QRK_RUNNING", false);
    if (!isFullScreen()) {
        settings.save2Settings("mainWindowGeometry", saveGeometry(), false);
        settings.save2Settings("mainWindowState", saveState(), false);
    }
    PluginManager::instance()->uninitialize();
    //        DatabaseManager::removeCurrentThread("CN");
    waitForThreadsWithTimeout();
    QApplication::exit();
}

//--------------------------------------------------------------------------------

void QRK::waitForThreadsWithTimeout(int timeout)
{
    int elapsedTime = 0;
    int checkInterval = 500;

    QThreadPool *threadPool = QThreadPool::globalInstance();
    qDebug() << "Function Name:" << Q_FUNC_INFO
             << "Waiting for tasks to finish, active threads:" << threadPool->activeThreadCount();

    QRKProgress progress;
    progress.setText(tr("Warten bis alle offenen (%1) Aufgaben beendet sind.").arg(threadPool->activeThreadCount()));
    progress.setWaitMode(false);
    if (threadPool->activeThreadCount() > 0) {
        progress.show();
        while (threadPool->activeThreadCount() > 0 && elapsedTime < timeout) {
            int progressPercentage = (elapsedTime * 100) / timeout;
            progress.progress(progressPercentage);
            qApp->processEvents();
            QThread::msleep(checkInterval);
            elapsedTime += checkInterval;
        }
    }

    qDebug() << "Function Name:" << Q_FUNC_INFO << "All threads finished.";
    Backup::cleanBackupTempDirectory();
}

void QRK::exitSlot()
{
    waitForThreadsWithTimeout();

    QCloseEvent event; // = new QCloseEvent();
    closeEvent(&event);
}

void QRK::actionLeaveDemoMode()
{
    Database::resetAllData();
    DemoMode::leaveDemoMode();
    restartApplication();
}

void QRK::actionResetDemoData()
{
    Database::resetAllData();
    restartApplication();
}

void QRK::closeCashRegister()
{
    if (RBAC::Instance()->Login()) {
        // userLogin->show();
        //        RBAC::Instance()->setuserId(-1);
        logOnOff(true);
    } else {
        RBAC::Instance()->setuserId(0);
    }

    if (!RBAC::Instance()->hasPermission("close_cashregister", true)) return;
    Reports rep(this, true);
    rep.endOfMonth();
    rep.createNullReceipt(PAYED_BY_CONCLUSION_RECEIPT);
    Database::setCashRegisterInAktive();
    backupDEP();
    restartApplication();
}

void QRK::actionResuscitationCashRegister()
{
    backupDEP();
    Database::resetAllData();
    RKSignatureModule::setDEPactive(false);
    restartApplication();
}

void QRK::backupDEP()
{
    if (Export::getLastMonthReceiptId() == -1) {
        QMessageBox::information(this, tr("DEP-7 Datensicherung"),
            tr("Es wurde kein Monatsbeleg gefunden. Versuchen Sie die Sicherung "
               "nach einen Monatsabschlusses erneut."));
        return;
    }

    QrkSettings settings;
    QString directoryname = settings.value("externalDepDirectory", "").toString();
    if (!Utils::isDirectoryWritable(directoryname)) {
        QMessageBox messageBox(QMessageBox::Question, QObject::tr("Externes DEP-7 Backup"),
            QObject::tr("Das externe Medium %1 ist nicht vorhanden oder nicht "
                        "beschreibbar.\nBitte beheben Sie den Fehler und drücken "
                        "OK. Wenn das Backup zu einen späteren Zeitpunkt "
                        "durchgeführt werden soll klicken Sie abbrechen.")
                .arg(directoryname),
            QMessageBox::Yes | QMessageBox::No, Q_NULLPTR);
        messageBox.setButtonText(QMessageBox::Yes, QObject::tr("OK"));
        messageBox.setButtonText(QMessageBox::No, QObject::tr("Abbrechen"));

        if (messageBox.exec() == QMessageBox::No) {
            return;
        }
    }

    if (Utils::isDirectoryWritable(directoryname)) {
        Export xDep;
        bool ok = xDep.createBackup();
        Spread::Instance()->setProgressBarValue(-1);
        if (ok) {
            QMessageBox::information(this, tr("DEP-7 Datensicherung"), tr("DEP-7 Datensicherung abgeschlossen."));
            return;
        } else {
            QMessageBox::information(this, tr("DEP-7 Datensicherung"), tr("DEP-7 Datensicherung fehlgeschlagen."));
            return;
        }
    }

    backupDEP();
}

void QRK::backup()
{
    setStatusBarProgressBarWait(true);
    Backup::create();
    setStatusBarProgressBarWait(false);
    QMessageBox::information(this, tr("Datensicherung"), tr("Datensicherung abgeschlossen."));
}

void QRK::restartApplication()
{
    // Spawn a new instance of myApplication:
    QString app = QApplication::applicationFilePath();
    QStringList arguments = QApplication::arguments();
    arguments << "-r";
    QString wd = QDir::currentPath();
    QProcess::startDetached(app, arguments, wd);
    QApplication::exit(0);
}

void QRK::newApplicationVersionAvailable(QJsonObject jsondata)
{
    disconnect(m_versionChecker, &VersionChecker::Version, Q_NULLPTR, Q_NULLPTR);

    QString currentVersion = qApp->applicationVersion().right(6);
    QString version = jsondata.value("version").toString();
    if (QString::compare(version.right(6), currentVersion, Qt::CaseInsensitive) > 0) {
        QMessageBox messageBox(QMessageBox::Information, QObject::tr("Out of Date"),
            QObject::tr("Hinweis: Ihre QRK Version (%1) ist möglicherweise "
                        "veraltet. QRK Version %2 ist verfügbar")
                .arg(qApp->applicationVersion())
                .arg(version),
            QMessageBox::Yes, this);
        if (jsondata.contains("detailedtext")) messageBox.setDetailedText(jsondata.value("detailedtext").toString());
        messageBox.setButtonText(QMessageBox::Yes, QObject::tr("OK"));
        messageBox.setWindowFlags(messageBox.windowFlags() | Qt::WindowStaysOnTopHint);

        messageBox.exec();
    }
    connect(m_versionChecker, &VersionChecker::Version, this, &QRK::newApplicationVersionAvailable);
}

void QRK::sendDatagram(QString what, QString data)
{
    //    qDebug() << "Function Name: " << Q_FUNC_INFO << what << " data: " <<
    //    data;
    QByteArray datagram;
    QDataStream out(&datagram, QIODevice::WriteOnly);
    out << what << data;

    udpSocket.writeDatagram(datagram, QHostAddress::Broadcast, 5824);
}

void QRK::userChanged()
{
    int userid = RBAC::Instance()->getUserId();
    if (userid < 0) return;

    QrkSettings settings;
    bool autostartregister = settings.value("autostartregister", false).toBool();
    if (autostartregister)
        QTimer::singleShot(1, [this] {
            emit onRegisterButton_clicked();
        });
}

void QRK::waiterLockRequest(QString data)
{
    if (data.isEmpty()) {
        emit logOnOff(true);
        return;
    }
    int userid = RBAC::Instance()->getUserIdByAcsKey(data);
    if (userid > 0) {
        RBAC::Instance()->setuserId(userid);
        emit waiterLockAccept(true);
    }
}

void QRK::handleMessage(const QString &message)
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << "Message: " << message;

    emit needToShow();
    return;
}
