/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "coupondelegate.h"
#include "coupondialog.h"
#include "database.h"

#include <QDateTime>
#include <QStandardItemModel>

CouponDelegate::CouponDelegate(COUPON_COLUMN column, int extraColumn, QObject *parent)
    : QStyledItemDelegate(parent)
    , m_current_column(column)
    , m_extraColumn(extraColumn)

{
}

QString CouponDelegate::displayText(const QVariant &value, const QLocale &locale) const
{

    QString formattedNum;

    switch (m_current_column) {
        case CURRENCY:
        case CREDIT:
            formattedNum = locale.toString(value.toDouble() / 100, 'f', 2) + " " + Database::getShortCurrency();
            break;
        case TAX:
            if (value.toDouble() == 0.0)
                formattedNum = "-";
            else
                formattedNum = locale.toString(value.toDouble() / 100, 'f', 2) + " %";
            break;
        case TYPE:
            formattedNum = value.toInt() == COUPON_TYPE::TYPE_MULTI ? "🗂 " + tr("Mehrzweck Gutschein")
                                                                    : "📄 " + tr("Einzweck Gutschein");
            break;
        case DELETED:
            if (value.toBool())
                formattedNum = tr("Kein Guthaben");
            else
                formattedNum = "OK";
            break;
        case DATETIME:
            formattedNum = locale.toString(QDateTime::fromString(value.toString(), Qt::ISODate), QLocale::ShortFormat);
            break;
        case CODE:
            formattedNum = value.toString().toUpper();
            break;
        case CHECKSUM:
            formattedNum = "🖨️";
            break;
        default:
            break;
    }

    return formattedNum;
}

void CouponDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    QStyleOptionViewItem alignedOption(option);
    alignedOption.displayAlignment = Qt::AlignLeft | Qt::AlignVCenter;
    int deleted;
    if (m_current_column == CURRENCY || m_current_column == CREDIT) {
        alignedOption.displayAlignment = Qt::AlignRight | Qt::AlignVCenter;
        if (m_extraColumn >= 0) {

            QModelIndex idx = index.sibling(index.row(), m_extraColumn);
            deleted = index.model()->data(idx, Qt::DisplayRole).toInt();
            if (deleted != 0) alignedOption.font.setStrikeOut(true);
        }
        // int deleted = index.model()->data(index, Qt::DisplayRole).toInt();
        //        alignedOption.font.setStrikeOut(true);
        int value = index.model()->data(index, Qt::DisplayRole).toInt();
        if (value <= 0) {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::red));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::red));
        }
    }
    if (m_current_column == CODE) {
        alignedOption.displayAlignment = Qt::AlignCenter | Qt::AlignVCenter;
    }
    if (m_current_column == TYPE) {
        alignedOption.displayAlignment = Qt::AlignLeft | Qt::AlignVCenter;
    }
    if (m_current_column == TAX) {
        alignedOption.displayAlignment = Qt::AlignRight | Qt::AlignVCenter;
    }
    if (m_current_column == DELETED) {
        if (index.model()->data(index, Qt::DisplayRole).toInt() == 0) {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::green));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::green));
        } else {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::red));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::red));
        }
    }

    QStyledItemDelegate::paint(painter, alignedOption, index);
}
