/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "givendialog.h"
#include "3rdparty/ckvsoft/numerickeypad.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#ifdef QRK_PROFEATURES
#    include "3rdparty/profeatures/profeatures.h"
#endif
#include "database.h"
#include "defines.h"
#include "qrksettings.h"
#include "ui_givendialog.h"

#include <QDoubleValidator>
#include <QFontDatabase>
#include <QMessageBox>

GivenDialog::GivenDialog(double &sum, QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::GivenDialog)
{
    ui->setupUi(this);
    setWindowFlags(windowFlags() & ~Qt::WindowContextHelpButtonHint);

    /*
    int id = QFontDatabase::addApplicationFont(":/src/font/digital.ttf");
    QString family = QFontDatabase::applicationFontFamilies(id).at(0);
    QFont digital(family);
    digital.setBold(false);
    digital.setPointSize(40);

    ui->lcdNumber->setFont(digital);
    */

    m_numericKeyPad = new NumericKeypad(false, this);
    ui->numericKeyPadLayout->addWidget(m_numericKeyPad);
    ui->mixedFrame->setHidden(true);
    ui->lcdNumber->setText(QLocale().toString(0.0, 'f', 2) + " " + Database::getCurrency());

    ui->buttonGroup->setId(ui->cash, PAYED_BY_CASH);
    ui->buttonGroup->setId(ui->creditCard, PAYED_BY_CREDITCARD);
    ui->buttonGroup->setId(ui->debitCard, PAYED_BY_DEBITCARD);
    m_numericKeyPad->setHidden(true);
    ui->cash->setHidden(true);

    //    QDoubleValidator *doubleVal = new QDoubleValidator(0.0, 9999999.99, 2, this);
    //    doubleVal->setNotation(QDoubleValidator::StandardNotation);

    QRegExp rx("^?(?:0|[1-9][0-9]*),?[0-9][0-9]$");
    QValidator *doubleVal = new QRegExpValidator(rx, this);

    ui->givenEdit->setValidator(doubleVal);
    m_sum = sum;
    ui->toPayLabel->setText(tr("Zu bezahlen: %1 %2").arg(QLocale().toString(sum, 'f', 2)).arg(Database::getCurrency()));

    QPalette palette = ui->lcdNumber->palette();

    palette.setColor(ui->lcdNumber->backgroundRole(), Qt::darkGreen);
    palette.setColor(ui->lcdNumber->foregroundRole(), Qt::darkGreen);
    ui->lcdNumber->setPalette(palette);

    QrkSettings settings;
    m_cashrepayment = settings.value("cashrepayment", false).toBool();
    m_numericKeyPad->setVisible(
        settings.value("virtualNumPad", false).toBool() || qApp->property("appBaseName").toString().contains("GASTRO"));

    connect(ui->givenEdit, &QLineEdit::textChanged, this, &GivenDialog::textChanged);
    connect(ui->finishButton, &QPushButton::clicked, this, &GivenDialog::accept);
    connect(ui->cancelButton, &QPushButton::clicked, this, &GivenDialog::close);
    connect(ui->mixedButton, &QPushButton::clicked, this, &GivenDialog::mixedButton);

#if (QT_VERSION >= QT_VERSION_CHECK(5, 15, 0))
    connect(ui->buttonGroup, static_cast<void (QButtonGroup::*)(int, bool)>(&QButtonGroup::idToggled), this,
        &GivenDialog::mixedPay);
#else
    connect(ui->buttonGroup, static_cast<void (QButtonGroup::*)(int, bool)>(&QButtonGroup::buttonToggled), this,
        &GivenDialog::mixedPay);
#endif
    connect(m_numericKeyPad, &NumericKeypad::textChanged, [=]() {
        ui->givenEdit->setText(m_numericKeyPad->text());
    });
    connect(ui->numericKeyPadPushButton, &QrkPushButton::clicked, this, &GivenDialog::numPadToogle);

    layout()->setSizeConstraint(QLayout::SetFixedSize);
    ui->givenEdit->setFocus();
}

GivenDialog::~GivenDialog()
{
    delete ui;
}

void GivenDialog::mixedButton()
{
    ui->mixedFrame->setHidden(false);
    ui->mixedButton->setEnabled(false);
    if (m_isCoupon)
        mixedPay(PAYED_BY_CASH, true);
    else
        mixedPay(PAYED_BY_DEBITCARD, true);
}

void GivenDialog::mixedPay(int id, bool checked)
{
    m_mixed = true;
    ui->mixedButton->setEnabled(false);
    ui->finishButton->setEnabled(true);
    if (m_isCoupon) ui->givenEdit->setEnabled(false);

    QBCMath mixed(m_sum - QLocale().toDouble(ui->givenEdit->text().trimmed()));
    mixed.round(2);

    if (checked && id == PAYED_BY_DEBITCARD) {
        ui->mixTypeLabel->setText(tr("Mit Bankomatkarte:"));
        ui->mixedCashLabel->setText(mixed.toLocale() + " " + Database::getCurrency());
        if (m_isCoupon) ui->givenEdit->clear();
        setLCDPalette(Qt::darkGreen);

    } else if (checked && id == PAYED_BY_CREDITCARD) {
        ui->mixTypeLabel->setText(tr("Mit Kreditkarte:"));
        ui->mixedCashLabel->setText(mixed.toLocale() + " " + Database::getCurrency());
        if (m_isCoupon) ui->givenEdit->clear();
        setLCDPalette(Qt::darkGreen);
    } else if (checked && id == PAYED_BY_CASH) {
        ui->mixTypeLabel->setText(tr("Mit Bargeld:"));
        ui->mixedCashLabel->setText(mixed.toLocale() + " " + Database::getCurrency());
        ui->givenEdit->setEnabled(true);
        setLCDPalette(Qt::darkGreen);
    } else {
        ui->debitCard->setChecked(false);
        ui->creditCard->setChecked(false);
        ui->mixTypeLabel->setText("");
        ui->mixedCashLabel->setText("");
        ui->givenEdit->setEnabled(true);
        m_mixed = false;
    }

    QBCMath retourMoney(m_sum - QLocale().toDouble(ui->givenEdit->text().trimmed()) - mixed.toDouble());
    retourMoney.round(2);
    ui->lcdNumber->setText(retourMoney.toLocale() + " " + Database::getCurrency());
}

void GivenDialog::setCoupon(double coupon)
{
    m_coupon = coupon;
    m_isCoupon = true;
    // ui->givenEdit->setText("0.00");
    ui->mixedButton->setEnabled(true);
    ui->cash->setHidden(false);
    ui->cash->setChecked(true);
    mixedButton();
}

void GivenDialog::accept()
{
    if (m_mixed) {
        QString mixedCash = ui->mixedCashLabel->text().replace(Database::getCurrency(), "").replace(",", ".");
        QBCMath mixed(mixedCash.trimmed());
        mixed.round(2);
        if (ui->debitCard->isChecked()) {
            mixedMap.insert(PAYED_BY_DEBITCARD, mixed.toDouble());
        } else if (ui->creditCard->isChecked()) {
            mixedMap.insert(PAYED_BY_CREDITCARD, mixed.toDouble());
        }
    }

    mixedMap.insert(PAYED_BY_CASH, getGivenValue());

#ifdef QRK_PROFEATURES
    /* If the database does not have a table with the name cashbook, the standard constructor is returned. When
     * converting to double, this results in &ok = false. thus the process is ignored as it should be. */
    bool ok;
    QBCMath diff(ProFeatures::getCheckoutContents(false).toDouble(&ok));
    diff /= 100;
    diff += m_sum; // diff only becomes smaller if the amount is a minus amount
    diff.round(2);
    bool reach_minus = diff.toDouble() < 0.00;

    if (reach_minus && ok) {
        QString warning_reach_minus = tr("Achtung.\n"
                                         "Ihr Kassabuch würde ein minus von %1 %2 erreichen.\n"
                                         "Zahlen Sie den fehlenden Betrag vorher in die Kassa ein.")
                                          .arg(diff.toLocale())
                                          .arg(QLocale().currencySymbol());

        QMessageBox::warning(this, tr("Storno"), warning_reach_minus);
        QDialog::reject();
        return;
    }
#endif
    QDialog::accept();
}

void GivenDialog::textChanged(QString given)
{

    if (given.isEmpty()) {
        resetGiven();
        return;
    }

    m_mixed = false;

    double retourMoney = QLocale().toDouble(given) - m_sum;

    if (retourMoney >= 0.0) {
        setLCDPalette(Qt::darkGreen);
        ui->mixedButton->setEnabled(false);
        ui->mixedFrame->setHidden(true);
        ui->finishButton->setEnabled(true);
        emit mixedPay(0, false);
    } else {
        setLCDPalette(Qt::red);
        ui->mixedButton->setEnabled(true);
        ui->finishButton->setEnabled(false);
        ui->mixedFrame->setHidden(true);
    }

    if (m_sum < 0.0) ui->mixedButton->setEnabled(false);
    ui->lcdNumber->setText(QLocale().toString(retourMoney, 'f', 2) + " " + Database::getCurrency());
}

void GivenDialog::resetGiven()
{
    m_mixed = false;

    setLCDPalette(Qt::darkGreen);
    ui->mixedButton->setEnabled(false);
    ui->mixedFrame->setHidden(true);
    ui->finishButton->setEnabled(true);
    ui->debitCard->setChecked(false);
    ui->creditCard->setChecked(false);
    ui->mixTypeLabel->setText("");
    ui->mixedCashLabel->setText("");

    double retourMoney = 0.0;
    ui->lcdNumber->setText(QLocale().toString(retourMoney, 'f', 2) + " " + Database::getCurrency());
}

void GivenDialog::setLCDPalette(QColor color)
{
    QPalette palette = ui->lcdNumber->palette();
    palette.setColor(ui->lcdNumber->backgroundRole(), color);
    palette.setColor(ui->lcdNumber->foregroundRole(), color);
    ui->lcdNumber->setPalette(palette);
}

double GivenDialog::getGivenValue()
{
    bool ok = false;
    double value = QLocale().toDouble(ui->givenEdit->text(), &ok);
    if (ok) return value;

    return 0.0;
}

QMap<int, double> GivenDialog::getGiven()
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << " given: " << mixedMap;
    return mixedMap;
}

void GivenDialog::numPadToogle(bool)
{
    bool hidden = m_numericKeyPad->isHidden();
    if (hidden) {
        m_numericKeyPad->setVisible(hidden);
    } else {
        m_numericKeyPad->setVisible(hidden);
        emit m_numericKeyPad->clear();
    }
}
