#include "primesigncardos_53.h"
#include <QDebug>

PrimeSignCARDOS_53::PrimeSignCARDOS_53(QString device_name, bool shared)
    : PrimeSignSmartCard(device_name, shared)
{
}

PrimeSignCARDOS_53::~PrimeSignCARDOS_53()
{
}

QString PrimeSignCARDOS_53::getCardType()
{
    return QObject::tr("PrimeSign CARDOS_53");
}

QString PrimeSignCARDOS_53::getExpireInfo()
{
    return parseExpiryDate("Dezember 2027", getCardType());
}

bool PrimeSignCARDOS_53::selectDF_QES()
{
    // if (m_DF_QES_Selected) return true;

    transmit(PRIMESIGN_SELECT_DF_QES, 7);
    m_DF_QES_Selected = true;
    return true;
}

QString PrimeSignCARDOS_53::signReceipt(QString data)
{
    QString jwsDataToBeSigned = RKSignatureModule::getDataToBeSigned(data);
    QString hashValue = RKSignatureModule::HashValue(jwsDataToBeSigned);

    QByteArray ba = 0;
    ba.append(hashValue.toUtf8());
    ba = QByteArray::fromHex(ba);
    unsigned char *hash = (unsigned char *)ba.data();

    QByteArray JWS_Signature;
    ASignResponse response = signHash(hash);

    for (uint i = 0; i < response.length; i++)
        JWS_Signature[i] = response.data[i]; // (const char)response.data[i];

    if (response.length > 0) {
        JWS_Signature = JWS_Signature.toBase64(QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
    } else
        JWS_Signature.append(base64Url_encode("Sicherheitseinrichtung ausgefallen"));

    return jwsDataToBeSigned + "." + JWS_Signature;
}

// Auslesen des Signaturzertifikates
QString PrimeSignCARDOS_53::getCertificate(bool base64)
{
    /*
     1. SELECT DF_QES
     2. SELECT EF_C_X509_CH_DS
     3. READ BINARY
    */

    selectDF_QES();

    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    QByteArray certificate;
    certificate = ReadFile();

    if (base64) return certificate.toBase64();

    return certificate;
}

/*
QString PrimeSignCARDOS_53::getCertificateSerial(bool hex)
{
    unsigned char data[256];

    selectDF_QES();
    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    DWORD i = response.length;
    if (i == 0) return QString::number(0);

    while (i--) {
        data[i] = response.data[i];
    }

    QByteArray ba = 0;
    ba.append((char *)data, 256);

    int len = static_cast<unsigned int>(static_cast<unsigned char>(ba.at(14)));
    long serial = 0;
    for (int i = 0; i < len; i++) {
        serial = (serial << 8) + static_cast<unsigned int>(static_cast<unsigned char>(ba.at(15 + i)));
    }

    QString serialHex = QString::number(serial, 16).toUpper();

    // put Certificate to Database
    if (!isCertificateInDB(serialHex)) putCertificate(serialHex, getCertificate(true));

    if (hex) {
        return serialHex;
    }

    return QString::number(serial);
}
*/

QString PrimeSignCARDOS_53::getCertificateSerial(bool hex)
{
    selectDF_QES();
    transmit(PRIMESIGN_SELECT_EF_C_X509_CH_DS, 7);

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    if (response.length < 15) return QStringLiteral("0"); // Antwort zu kurz

    QByteArray ba(reinterpret_cast<const char *>(response.data), qMin(response.length, DWORD(256)));

    int len = static_cast<unsigned char>(ba.at(14));
    if (ba.size() < 15 + len) return QStringLiteral("0"); // Seriennummer zu kurz

    QByteArray serialBytes = ba.mid(15, len);

    // Führendes 0x00 entfernen (optional)
    if (serialBytes.size() > 1 && (uchar)serialBytes[0] == 0x00) serialBytes.remove(0, 1);

    QString serialHex = serialBytes.toHex().toUpper();

    if (!isCertificateInDB(serialHex)) putCertificate(serialHex, getCertificate(true));

    if (hex) return serialHex;

    // Dezimalwert als String, auch für sehr große Werte
    QString serialDec = PrimeSignSmartCard::hexToDecimalString(serialBytes);
    return serialDec;
}

QByteArray encodePin(const QString &pin)
{
    int pinLength = pin.length();
    if (pinLength < 6 || pinLength > 12) return QByteArray(); // ungültige Länge, leer zurückgeben

    QByteArray pinField;
    pinField.append(0x20 + pinLength); // 2y = 0x20 + Länge

    // PIN in BCD kodieren: je 2 Ziffern in 1 Byte
    for (int i = 0; i < pinLength; i += 2) {
        int highNibble = pin[i].digitValue();
        int lowNibble = 0xF; // Default lowNibble = 0xF (für ungerade Anzahl)

        if (i + 1 < pinLength) lowNibble = pin[i + 1].digitValue();

        if (highNibble < 0 || lowNibble < 0) return QByteArray(); // ungültige Ziffer

        pinField.append((highNibble << 4) | lowNibble);
    }

    // Auffüllen mit 0xFF bis 8 Bytes insgesamt
    while (pinField.size() < 8)
        pinField.append(0xFF);

    return pinField;
}

ASignResponse PrimeSignCARDOS_53::signHash(const unsigned char hash[32])
{

    bool blocked = false;
    if (verifyPin(blocked)) {
        unsigned char hashcmd[32 + 6] = PRIMESIGN_OS53_CMD_PUT_HASH;
        memcpy(&hashcmd[5], hash, 32);
        return PrimeSignSmartCard::transmit(hashcmd, sizeof(hashcmd));
    }

    // PIN invalid → return empty or error response
    qWarning() << "PIN verification failed – cannot sign hash";

    ASignResponse resp;
    resp.code[0] = 0x69; // "Command not allowed"
    resp.code[1] = 0x82; // Security condition not satisfied
    return resp;
}

bool PrimeSignCARDOS_53::verifyPin(bool &blocked)
{
    selectDF_QES();

    const QByteArray pinBytes = m_pin.trimmed();
    const int pinLength = pinBytes.size();
    const int bcdLength = (pinLength + 1) / 2;

    unsigned char cmd[14] = PRIMESIGN_CMD_VERIFY;
    cmd[5] = static_cast<unsigned char>((2 << 4) | pinLength);

    // BCD encoding
    for (int i = 0; i < bcdLength; ++i) {
        char high = pinBytes[i * 2] - '0';
        char low = ((i * 2 + 1) < pinLength) ? (pinBytes[i * 2 + 1] - '0') : 0x0F;
        cmd[6 + i] = static_cast<unsigned char>((high << 4) | low);
    }

    // Fill unused bytes with 0xFF
    for (int i = 6 + bcdLength; i <= 12; ++i)
        cmd[i] = 0xFF;

    ASignResponse resp = PrimeSignSmartCard::transmit(cmd, 13);
    quint16 sw = (resp.code[0] << 8) | resp.code[1];

    if (sw == 0x9000) {
        return true; // PIN is correct
    } else if (sw == 0x6983) {
        blocked = true;
        qWarning() << "Function Name: " << Q_FUNC_INFO << "PIN is blocked!";
    } else if ((resp.code[0] == 0x63) && ((resp.code[1] & 0xF0) == 0xC0)) {
        int retries = resp.code[1] & 0x0F;
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Incorrect PIN. Remaining attempts:" << retries;
    } else {
        qWarning() << "Function Name: " << Q_FUNC_INFO
                   << "Unknown error during PIN verification. SW=" << QString("0x%1").arg(sw, 4, 16, QChar('0'));
    }

    return false;
}
