/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "qrkgastrofinishticket.h"
#include "database.h"
#include "defines.h"
#include "documentprinter.h"
#include "givendialog.h"
#include "pluginmanager/pluginmanager.h"
#include "qrkgastro.h"
#include "qrkpaymentdialog.h"
#include "qrkprogress.h"
#include "qrksettings.h"
#include "qrktimedmessagebox.h"
#include "receiptitemmodel.h"
#include "reports.h"
#include "utils/utils.h"

#ifdef QRK_PROFEATURES
#    include "3rdparty/profeatures/profeatures.h"
#endif

#include <QApplication>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QSqlDatabase>
#include <QSqlError>


#include <QDebug>

QRKGastroFinishTicket::QRKGastroFinishTicket(bool paycash, QWidget *parent)
    : QWidget(parent)
    , m_receiptitemmodel(new ReceiptItemModel(this))
    , m_paycash(paycash)
{
    QrkSettings settings;
    m_useGivenDialog = settings.value("useGivenDialog", false).toBool();
    m_receiptPrintDialog = settings.value("useReceiptPrintedDialog", false).toBool();
    m_proofs_guestname = settings.value("Gastro/proofs_guestname", false).toBool();
    m_minstockDialog = settings.value("useMinstockDialog", false).toBool();
}

QRKGastroFinishTicket::~QRKGastroFinishTicket()
{
}

void QRKGastroFinishTicket::calculateReceipt(QList<int> ticketList, const QString roomtableName)
{

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery orderQuery(dbc, Q_FUNC_INFO);

    // calc sum
    QBCMath sum;
    QJsonObject Root; // root object
    QJsonArray items;
    query.prepare("SELECT SUM(count * gross) as gross FROM ticketorders WHERE "
                  "ticketId=:ticketId");

    orderQuery.prepare("SELECT ticketorders.count, products.name, products.id, "
                       "ticketorders.gross, ticketorders.id FROM ticketorders "
                       " LEFT JOIN products ON ticketorders.product=products.id"
                       " WHERE ticketorders.ticketId=:id");


    // check for orderextras
    CSqlQuery extras(dbc, Q_FUNC_INFO);
    extras.prepare("SELECT orderextras.type, products.name FROM orderextras "
                   " LEFT JOIN products ON orderextras.product=products.id"
                   " WHERE ticketId=:ticketId");


    CSqlQuery taxtypes(dbc, Q_FUNC_INFO);
    taxtypes.prepare(QString("SELECT tax, comment FROM taxtypes WHERE "
                             "taxlocation=:taxlocation ORDER BY id"));
    query.bindValue(":taxlocation", Database::getTaxLocation());
    taxtypes.exec();
    while (taxtypes.next()) {
        Root[taxtypes.value(1).toString()] = 0.0;
    }

    QMap<double, double> taxes; // <tax-percent, sum>
    int positions = 0;

    for (int i = 0; i < ticketList.count(); i++) {
        query.bindValue(":ticketId", ticketList[i]);
        query.exec();
        query.next();

        sum += query.value("gross").toDouble();
        sum.round(2);

        orderQuery.bindValue(":id", ticketList[i]);
        orderQuery.exec();

        extras.bindValue(":ticketId", ticketList[i]);
        extras.exec();
        qDebug() << Database::getLastExecutedQuery(extras);

        while (orderQuery.next()) {
            QString product = orderQuery.value("name").toString();
            int id = -1;
            QJsonObject item = findValueFromJsonArray(items, "product", product, id);
            QBCMath count = orderQuery.value("count").toInt();
            QBCMath singlePrice(orderQuery.value("gross").toDouble());
            QBCMath gross = singlePrice * count;
            gross.round(2);

            QBCMath tax(Database::getTaxFromProduct(orderQuery.value("id").toInt()));
            tax.round(2);


            QJsonArray OrderExtras;
            while (extras.next()) {
                QJsonObject extra;
                extra["product"] = extras.value("name").toString();
                extra["type"] = extras.value("type").toString();
                OrderExtras.append(extra);
            }

            if (item.isEmpty()) {
                positions++;
                item["count"] = count.toInt();
                item["product"] = product;
                item["singleprice"] = singlePrice.toDouble();
                item["gross"] = gross.toDouble();
                item["tax"] = tax.toDouble();
                if (taxes.contains(tax.toDouble()))
                    taxes[tax.toDouble()] += Utils::getTax(gross.toDouble(), tax.toDouble());
                else
                    taxes[tax.toDouble()] = Utils::getTax(gross.toDouble(), tax.toDouble());

                item["extra"] = OrderExtras;

                items.append(item);
            } else {
                item["count"] = item["count"].toInt() + count.toInt();
                item["gross"] = item["gross"].toInt() + gross.toDouble();
                item["extra"] = OrderExtras;
                taxes[tax.toDouble()] += Utils::getTax(gross.toDouble(), tax.toDouble());
                items.replace(id, item);
            }
        }
    }

    QJsonArray Taxes;
    QList<double> keys = taxes.keys();
    for (int i = 0; i < keys.count(); i++) {
        QJsonObject tax;
        tax["t1"] = QString("%1%").arg(keys[i]);
        tax["t2"] = QString::number(taxes[keys[i]], 'f', 2);
        Taxes.append(tax);
    }
    // Header
    Root["printHeader"] = Database::getHeaderText();
    // Footer
    Root["printFooter"] = Database::getFooterText();

    Root["version"] = QString("%1.%2").arg(QRK_VERSION_MAJOR).arg(QRK_VERSION_MINOR);
    Root["action"] = JOURNAL_RECEIPT;
    Root["kasse"] = Database::getCashRegisterId();
    Root["actionText"] = tr("Zwischenrechnung");
    Root["receiptNum"] = QDateTime::currentDateTime().toString("hhmmss").toInt();
    Root["typeText"] = "ZR";
    Root["shopName"] = Database::getShopName();
    Root["shopMasterData"] = Database::getShopMasterData();
    Root["comment"] = tr("Zwischenrechnung\n%1").arg(roomtableName);
    Root["receiptTime"] = QDateTime::currentDateTime().toString(Qt::ISODate);
    Root["currentRegisterYear"] = QDate::currentDate().year();
    Root.insert("Orders", items);
    Root["positions"] = positions;
    Root["Taxes"] = Taxes;
    Root["taxesCount"] = taxes.count();
    Root["sum"] = sum.toDouble();
    Root.insert("interimCalculation", true);

    DocumentPrinter dp(this);
    dp.printReceipt(Root);

    if (m_receiptPrintDialog) {
        QrkTimedMessageBox messageBox(10, QMessageBox::Information, tr("Drucker"),
            tr("Zwischenrechnung %1 wurde gedruckt. "
               "Nächster Vorgang wird gestartet.")
                .arg(Root["receiptNum"].toInt()),
            QMessageBox::Yes | QMessageBox::Default);

        messageBox.setDefaultButton(QMessageBox::Yes);
        messageBox.setButtonText(QMessageBox::Yes, QObject::tr("OK"));
        messageBox.exec();
    }
}

bool QRKGastroFinishTicket::createReceipt(int ticket)
{

    /*
     * {"receipt":[
     *  {"customertext": "Customer Text",
     *   "payedBy": "0",
     *   "items":[
     *     { "count": "3", "name": "Kupplung", "gross": "122,70", "tax": "20" },
     *     { "count": "1", "name": "Bremsbeläge", "gross": "32,30", "tax": "10" },
     *     { "count": "2", "name": "Benzinschlauch", "gross": "17,80", "tax": "20"
     * }, { "count": "1", "name": "Ölfilter", "gross": "104,50", "tax": "13" }
     *    ]
     *   }
     * ]}
     */
    bool ret = Reports().mustDoEOAny(QDateTime::currentDateTime());
    if (ret) {
        if (QTime::currentTime() < QTime(8, 0, 0)) {
            Database::setCurfewTime(QTime::currentTime().addSecs(120), true);
            m_curfewoveride = true;
        } else {
            QRKGastro::infoMessage(true);
            return false;
        }
    }

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT tableId FROM tickets WHERE "
                  "id=:ticketId");
    query.bindValue(":ticketId", ticket);
    query.exec();
    query.next();
    int tableId = query.value("tableId").toInt();
    m_receiptitemmodel->setRoomTableName(QRKGastro::getRoomNameFromTableId(tableId), QRKGastro::getTableName(tableId));

    // calc sum
    query.prepare("SELECT SUM(count * gross) as gross FROM ticketorders WHERE "
                  "ticketId=:ticketId");
    query.bindValue(":ticketId", ticket);
    query.exec();
    query.next();
    QBCMath sum(query.value("gross").toDouble());
    sum.round(2);

    int code;
    if (m_paycash) {
        code = QRKPaymentDialog::CASHRECEIPT_TICKET;
    } else {
        QRKPaymentDialog question(sum, true, this);
        if (ProFeatures::isActive("Coupon")) {
            question.couponPayment(true);
        }
        code = question.exec();
    }
    if (code == QDialog::Rejected) return false;

    QJsonObject mainitem;
    QJsonArray items;

    /*
    query.prepare("SELECT open FROM tickets WHERE id=:ticketId");
    query.bindValue(":ticketId", ticket);
    query.exec();
    query.next();
  */

    QDateTime invoiceTime = QDateTime::currentDateTime();

    int payedBy = 0;

    if (code == QRKPaymentDialog::CASHRECEIPT_TICKET)
        payedBy = PAYED_BY_CASH;
    else if (code == QRKPaymentDialog::CREDITCARD_TICKET)
        payedBy = PAYED_BY_CREDITCARD;
    else if (code == QRKPaymentDialog::DEBITCARD_TICKET)
        payedBy = PAYED_BY_DEBITCARD;
    else if (code == QRKPaymentDialog::PRIVATE_TICKET)
        payedBy = PAYED_BY_PRIVATE;
    else if (code == QRKPaymentDialog::EMPLOYEE_TICKET)
        payedBy = PAYED_BY_EMPLOYEE;
    else if (code == QRKPaymentDialog::ADVERTISING_TICKET)
        payedBy = PAYED_BY_ADVERTISING;
    else if (code == QRKPaymentDialog::COUPON_TICKET)
        payedBy = PAYED_BY_COUPON;

    mainitem["payedBy"] = QString::number(payedBy);

    //   mainitem["customertext"] = "Customer Text";
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    orders.prepare("SELECT ticketorders.count, products.name, products.id, "
                   "ticketorders.gross, ticketorders.id AS ticketorderId FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:id");
    orders.bindValue(":id", ticket);
    orders.exec();

    while (orders.next()) {
        QJsonObject item;
        QBCMath gross(orders.value("gross").toDouble());
        gross.round(2);
        QBCMath tax(Database::getTaxFromProduct(orders.value("id").toInt()));
        tax.round(2);
        item["count"] = orders.value("count").toString();
        item["name"] = orders.value("name").toString();
        //        item["discount"] = (payedBy < 0) ? "100.00" : "0.00";
        item["gross"] = (payedBy < PAYED_BY_CASH) ? "0.00" : gross.toString();
        item["tax"] = tax.toString();
        item["ticketorder"] = orders.value("ticketorderId").toInt();
        item["productId"] = orders.value("id").toInt();

        items.append(item);
    }
    mainitem["items"] = items;

    m_receiptitemmodel->newOrder();
    if (m_proofs_guestname) m_receiptitemmodel->setCustomerText(QRKGastro::getGuestName(ticket));

    if (m_receiptitemmodel->setReceiptImportMode(mainitem)) {
        // COUPON begin
        QBCMath redeem;
        QString coupon_code = "";
        bool isSingle = false;
        if (payedBy == PAYED_BY_COUPON) {
            if (!ProFeatures::CouponDialog(*m_receiptitemmodel, sum, redeem, coupon_code, isSingle)) return false;
        }

        QBCMath rest((sum - redeem));
        rest.round(2);
        bool force = rest > 0 && isSingle;
        if (force && payedBy == PAYED_BY_COUPON) {
            QRKPaymentDialog pDialog(rest, false, this);
            int code = pDialog.exec();
            if (code == QRKPaymentDialog::CASHRECEIPT_TICKET)
                payedBy = PAYED_BY_CASH;
            else if (code == QRKPaymentDialog::CREDITCARD_TICKET)
                payedBy = PAYED_BY_CREDITCARD;
            else if (code == QRKPaymentDialog::DEBITCARD_TICKET)
                payedBy = PAYED_BY_DEBITCARD;
        }

        if ((m_useGivenDialog && payedBy == PAYED_BY_CASH)) {
            double s = sum.toDouble();
            GivenDialog given(s, this);
            if (given.exec() == 0) {
                return false;
            }
            m_receiptitemmodel->setGiven(given.getGiven());
        }
        if (payNow(payedBy)) {
            if (redeem > 0) {
                ProFeatures::CouponUpdate(
                    m_receiptitemmodel->getReceiptNum(), coupon_code, (redeem * 100).getIntPart());
            }
            query.prepare("UPDATE tickets SET timestamp=:timestamp, open=0, "
                          "payedBy=:payedBy WHERE id=:ticketId");
            query.bindValue(":timestamp", invoiceTime.toString(Qt::ISODate));
            query.bindValue(":ticketId", ticket);
            query.bindValue(":payedBy", payedBy);
            query.exec();
            return true;
        }
    }

    return false;
}

bool QRKGastroFinishTicket::payNow(int payedBy)
{

    QDateTime dateTime;
    dateTime = QDateTime::currentDateTime();

    qApp->processEvents();

    QRKProgress waitBar;
    waitBar.setText(tr("Beleg wird erstellt."));
    waitBar.setWaitMode();
    waitBar.show();

    Reports rep;
    bool ret = rep.checkEOAny(dateTime);
    if (!ret) {
        return false;
    }

    qDebug() << "Function Name: " << Q_FUNC_INFO
             << "Reportcheck, Time elapsed: " << dateTime.msecsTo(QDateTime::currentDateTime()) << "ms";
    //    emit sendDatagram("topay",
    //    ui->sumLabel->text().replace(Database::getCurrency() ,""));

    /*
    if (m_receiptitemmodel->rowCount() > 0) {
        int rc = m_receiptitemmodel->rowCount();

        for (int row = 0; row < rc; row++) {
            QJsonObject itemdata;
            itemdata["name"]
                = m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_PRODUCT, QModelIndex()))
                      .toString();
            itemdata["tax"]
                = m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_TAX, QModelIndex())).toDouble();
            itemdata["net"] = (payedBy < PAYED_BY_CASH)
                ? 0.0
                : m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_NET, QModelIndex())).toDouble();
            itemdata["gross"] = (payedBy < PAYED_BY_CASH)
                ? 0.0
                : m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_SINGLE, QModelIndex()))
                      .toDouble();
            itemdata["itemnum"]
                = m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_PRODUCTNUMBER, QModelIndex()))
                      .toString();
            itemdata["visible"] = (payedBy == PAYED_BY_COUPON) ? 0 : 1;

            int productId
                = m_receiptitemmodel->data(m_receiptitemmodel->index(row, REGISTER_COL_PRODUCT), Qt::UserRole).toInt();
            Database::addProduct(itemdata, productId);
        }
    }
    */

    bool retVal = false;
    QSqlDatabase dbc = Database::database();
    dbc.transaction();

    m_currentReceipt = m_receiptitemmodel->createReceipts();

    bool sql_ok = true;
    if (m_currentReceipt > 0) {
        if (m_receiptitemmodel->createOrder()) {
            if (finishReceipts(payedBy)) {
                sql_ok = true;
            } else {
                sql_ok = false;
            }
        } else {
            sql_ok = false;
        }
    }
    qDebug() << "Function Name: " << Q_FUNC_INFO
             << "Create Receipt Time elapsed: " << dateTime.msecsTo(QDateTime::currentDateTime()) << "ms";

    if (sql_ok && m_currentReceipt > 0) {
        dbc.commit();
        if (m_receiptPrintDialog) {
            QrkTimedMessageBox messageBox(10, QMessageBox::Information, tr("Drucker"),
                tr("Beleg %1 wurde fertiggestellt. Nächster Vorgang wird gestartet.").arg(m_currentReceipt),
                QMessageBox::Yes | QMessageBox::Default);

            messageBox.setDefaultButton(QMessageBox::Yes);
            messageBox.setButtonText(QMessageBox::Yes, QObject::tr("OK"));
            messageBox.exec();
        }

        QStringList stockList = Database::getStockInfoList();
        if (m_minstockDialog && stockList.count() > 0) {
            QMessageBox messageBox(QMessageBox::Information, tr("Lagerbestand"),
                tr("Mindestbestand wurde erreicht oder unterschritten!"), QMessageBox::Yes | QMessageBox::Default);

            messageBox.setDefaultButton(QMessageBox::Yes);
            messageBox.setButtonText(QMessageBox::Yes, tr("OK"));
            messageBox.setDetailedText(stockList.join('\n'));

            messageBox.exec();
        }
        retVal = true;
    } else {
        sql_ok = dbc.rollback();
        QMessageBox::warning(this, tr("Fehler"),
            tr("Datenbank und/oder Signatur Fehler!\nAktueller BON kann nicht "
               "erstellt werden. (Rollback: %1).\nÜberprüfen Sie ob genügend "
               "Speicherplatz für die Datenbank vorhanden ist. Weitere Hilfe gibt "
               "es im Forum. https:://www.ckvsoft.at")
                .arg(sql_ok ? tr("durchgeführt") : tr("fehlgeschlagen")));
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << dbc.lastError().text();
    }

    //    emit sendDatagram("finished", "");
    if (m_curfewoveride) {
        Database::setCurfewTime(QTime::currentTime(), true);
        m_curfewoveride = false;
    }

    qDebug() << "Function Name: " << Q_FUNC_INFO
             << "Finished Time elapsed: " << dateTime.msecsTo(QDateTime::currentDateTime()) << "ms";

    return retVal;
}

bool QRKGastroFinishTicket::finishReceipts(int payedBy, int id, bool isReport)
{

    QDateTime dt = QDateTime::currentDateTime();
    //    m_receiptitemmodel->setCustomerText(ui->customerText->text());
    m_receiptitemmodel->setReceiptTime(dt);
    bool ret = m_receiptitemmodel->finishReceipts(payedBy, id, isReport);
    return ret;
}

QJsonObject QRKGastroFinishTicket::findValueFromJsonArray(QJsonArray arr, QString key, QVariant val, int &id)
{

    for (int i = 0; i < arr.count(); i++) {
        const auto obj = arr.at(i);
        if (obj.toObject().value(key) == val) {
            id = i;
            return obj.toObject();
        }
    }

    return QJsonObject();
}
