#include "qrkviewer.h"
#include "3rdparty/ckvsoft/RK/rk_signaturemodule.h"
#include "3rdparty/ckvsoft/databasemanager.h"
#include "3rdparty/ckvsoft/datetimedelegate.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "database.h"
#include "defines.h"
#include "documentprinter.h"
#include "qrkdelegate.h"
#include "qrksettings.h"
#include "reports.h"
#include "utils/utils.h"

#include "ui_qrkviewer.h"

#include <QDir>
#include <QFileDialog>
#include <QHeaderView>
#include <QItemSelection>
#include <QItemSelectionModel>
#include <QJsonObject>
#include <QMessageBox>
#include <QPrintDialog>
#include <QPrintPreviewDialog>
#include <QPrinter>
#include <QPrinterInfo>
#include <QSqlDatabase>
#include <QSqlError>

#include <QDebug>

QrkViewer::QrkViewer(QWidget *parent)
    : QMainWindow(parent)
    , ui(new Ui::QrkViewer)
{
    ui->setupUi(this);
    connect(ui->actionQRK_Dokument, &QAction::triggered, this, &QrkViewer::documentTriggered);
    connect(ui->printPushButton, &QrkPushButton::clicked, this, &QrkViewer::filePrintPreview);
}

QrkViewer::~QrkViewer()
{
    delete ui;
}

void QrkViewer::documentTriggered(bool)
{
    QrkSettings settings;

    QString lastUsedFileName = settings.value("Viewer/document", QDir::homePath()).toString();

    m_documentfile = QFileDialog::getOpenFileName(
        this, tr("Datei laden"), lastUsedFileName, "Qrk (*.db)", Q_NULLPTR, QFileDialog::DontUseNativeDialog);

    disconnect(ui->documentList->selectionModel(), &QItemSelectionModel::selectionChanged, this,
        &QrkViewer::onDocumentSelectionChanged);

    if (!m_documentfile.isEmpty()) {

        DatabaseManager::clear();
        Database db;
        db.open(false, "QSQLITE", m_documentfile);
        QSqlDatabase dbc = Database::database();

        m_documentListModel = new SortFilterSqlQueryModel(this);
        ui->documentList->setModel(m_documentListModel);
        ui->documentList->repaint();
        ui->documentList->setItemDelegateForColumn(
            DOCUMENT_COL_TOTAL, new QrkDelegate(QrkDelegate::NUMBERFORMAT_DOUBLE, this));

        ui->documentList->setItemDelegateForColumn(4, new DateTimeDelegate(this));
        ui->documentList->setItemDelegateForColumn(2, new DateDelegate(this));

        ui->documentList->horizontalHeader()->setStretchLastSection(false);
        ui->documentList->resizeColumnsToContents();
        ui->documentList->horizontalHeader()->setStretchLastSection(true);

        settings.save2Settings("Viewer/document", m_documentfile);

        m_documentListModel->setQuery(QString("SELECT receipts.receiptNum, actiontypes.actionText, "
                                              "strftime('%Y-%m-%d',receipts.infodate) AS infodate, "
                                              "ROUND(receipts.gross,2) AS gross, receipts.timestamp FROM receipts "
                                              "INNER JOIN actiontypes ON receipts.payedBy=actiontypes.actionId WHERE "
                                              "payedBy > %1")
                                          .arg(PAYED_BY_REPORT_EOD),
            dbc);

        m_documentListModel->setFilterColumn("receiptNum");
        m_documentListModel->setFilterFlags(Qt::MatchStartsWith);
        m_documentListModel->setFilter("");
        m_documentListModel->select();
        m_documentListModel->sort(DOCUMENT_COL_RECEIPT, Qt::DescendingOrder);
        while (m_documentListModel->canFetchMore())
            m_documentListModel->fetchMore();

        if (m_documentListModel->lastError().isValid())
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << m_documentListModel->lastError();

        m_documentListModel->setHeaderData(DOCUMENT_COL_RECEIPT, Qt::Horizontal, tr("Beleg"));
        m_documentListModel->setHeaderData(DOCUMENT_COL_TYPE, Qt::Horizontal, tr("Type"));
        m_documentListModel->setHeaderData(DOCUMENT_COL_INFO, Qt::Horizontal, tr("Info"));
        m_documentListModel->setHeaderData(DOCUMENT_COL_TOTAL, Qt::Horizontal, tr("Summe"));
        m_documentListModel->setHeaderData(DOCUMENT_COL_DATE, Qt::Horizontal, tr("Erstellungsdatum"));

        connect(ui->documentList->selectionModel(), &QItemSelectionModel::selectionChanged, this,
            &QrkViewer::onDocumentSelectionChanged);
    }
}

void QrkViewer::onDocumentSelectionChanged(const QItemSelection &, const QItemSelection &)
{
    QModelIndexList indexList = ui->documentList->selectionModel()->selectedIndexes();
    int row = -1;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }

    qrcode = false;
    int receiptNum
        = m_documentListModel->data(m_documentListModel->index(row, DOCUMENT_COL_RECEIPT, QModelIndex())).toInt();

    QString payedByText
        = m_documentListModel->data(m_documentListModel->index(row, DOCUMENT_COL_TYPE, QModelIndex())).toString();

    int type = Database::getActionTypeByName(payedByText);

    if (type == PAYED_BY_REPORT_EOD || type == PAYED_BY_REPORT_EOM) { /* actionType Tagesbeleg*/
        // ui->customerTextLabel->setHidden(true);
        ui->tableView->setHidden(true);
        ui->textBrowser->setHidden(false);
        ui->textBrowser->setHtml(Reports::getReport(receiptNum, m_report_by_productgroup, false, false));
        if (qrcode == false && RKSignatureModule::isDEPactive()) {
            qrcode = true;
            QTextCursor cursor(ui->textBrowser->document());
            cursor.movePosition(QTextCursor::Up, QTextCursor::MoveAnchor);
            bool isDamaged;
            QImage img = Utils::getQRCode(ui->documentList->model()
                                              ->data(m_documentListModel->index(row, REGISTER_COL_COUNT, QModelIndex()))
                                              .toInt(),
                isDamaged)
                             .toImage();
            cursor.insertImage(img);
            if (isDamaged) cursor.insertHtml("</br><small>Sicherheitseinrichtung ausgefallen</small>");
        }

    } else {
        ui->tableView->setHidden(false);
        ui->textBrowser->setHidden(true);

        QString stornoText = "";
        if (Database::getStorno(receiptNum) == 1)
            stornoText = tr("(Stornierter Beleg, siehe Beleg Nr: %1)").arg(Database::getStornoId(receiptNum));
        else if (Database::getStorno(receiptNum) == 2)
            stornoText = tr("(Storno Beleg für Beleg Nr: %1)").arg(Database::getStornoId(receiptNum));


        QSqlDatabase dbc = Database::database();
        int id = ui->documentList->model()
                     ->data(m_documentListModel->index(row, REGISTER_COL_COUNT, QModelIndex()))
                     .toInt();

        QSqlQueryModel *documentContentModel = new QSqlQueryModel(this);

        documentContentModel->setQuery(QString("SELECT orders.count, products.itemnum, products.name, "
                                               "round(orders.net,2), orders.tax, orders.gross, "
                                               "orders.discount * (-1), ROUND((orders.count * orders.gross) - "
                                               "((orders.count * orders.gross / 100) * orders.discount),2) AS "
                                               "Price FROM orders INNER JOIN products ON "
                                               "products.id=orders.product WHERE orders.receiptId=%1")
                                           .arg(id),
            dbc);
        documentContentModel->setHeaderData(REGISTER_COL_COUNT, Qt::Horizontal, tr("Anz."));
        documentContentModel->setHeaderData(REGISTER_COL_PRODUCTNUMBER, Qt::Horizontal, tr("Artikelnummer"));
        documentContentModel->setHeaderData(REGISTER_COL_PRODUCT, Qt::Horizontal, tr("Artikel"));
        documentContentModel->setHeaderData(REGISTER_COL_NET, Qt::Horizontal, tr("E-Netto"));
        documentContentModel->setHeaderData(REGISTER_COL_TAX, Qt::Horizontal, tr("MwSt."));
        documentContentModel->setHeaderData(REGISTER_COL_SINGLE, Qt::Horizontal, tr("E-Preis"));
        documentContentModel->setHeaderData(REGISTER_COL_DISCOUNT, Qt::Horizontal, tr("Rabatt %"));
        documentContentModel->setHeaderData(REGISTER_COL_TOTAL, Qt::Horizontal, tr("Preis"));
        ui->tableView->setModel(documentContentModel);
        ui->tableView->setItemDelegateForColumn(
            REGISTER_COL_NET, new QrkDelegate(QrkDelegate::NUMBERFORMAT_DOUBLE, this));
        ui->tableView->setItemDelegateForColumn(REGISTER_COL_TAX, new QrkDelegate(QrkDelegate::COMBO_TAX, this));
        ui->tableView->setItemDelegateForColumn(
            REGISTER_COL_SINGLE, new QrkDelegate(QrkDelegate::NUMBERFORMAT_DOUBLE, this));
        ui->tableView->setItemDelegateForColumn(REGISTER_COL_DISCOUNT, new QrkDelegate(QrkDelegate::DISCOUNT, this));
        ui->tableView->setItemDelegateForColumn(
            REGISTER_COL_TOTAL, new QrkDelegate(QrkDelegate::NUMBERFORMAT_DOUBLE, this));

        ui->tableView->resizeColumnsToContents();
        ui->tableView->horizontalHeader()->setSectionResizeMode(REGISTER_COL_PRODUCT, QHeaderView::Stretch);
    }
}

void QrkViewer::printButtonClicked()
{
}

void QrkViewer::filePrintPreview()
{
    // display print preview dialog
    QPrinterInfo def = QPrinterInfo::defaultPrinter();

    // if there is no default printer set the print preview won't show
    if (def.isNull() || def.printerName().isEmpty()) {
        if (QPrinterInfo::availablePrinters().isEmpty()) {
            QMessageBox::critical(this, tr("Print error"),
                tr("Cannot proceed because there are no available printers in your system."), QMessageBox::Ok);
        } else {
            def = QPrinterInfo::availablePrinters().first();
        }
    }

    QPrinter printer(def);

    QPrintPreviewDialog preview(&printer);
    connect(&preview, &QPrintPreviewDialog::paintRequested, this, &QrkViewer::previewPrint);
    preview.exec();
}

void QrkViewer::previewPrint(QPrinter *printer)
{
    //        ui->textBrowser->print(printer);
    QModelIndexList indexList = ui->documentList->selectionModel()->selectedIndexes();
    int row = -1;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }

    QSqlDatabase dbc = Database::database();
    int id
        = ui->documentList->model()->data(m_documentListModel->index(row, REGISTER_COL_COUNT, QModelIndex())).toInt();

    DocumentPrinter docprint;

    if (ui->textBrowser->isVisible()) {
        docprint.printDocument(printer, ui->textBrowser->document());
    } else {
        ReceiptItemModel reg;
        reg.setCurrentReceiptNum(id);

        QrkSettings settings;
        QJsonObject data = reg.compileData();

        data["isCopy"] = true;
        int storno = Database::getStorno(id);
        if (storno == 2) {
            id = Database::getStornoId(id);
            data["comment"] = (id > 0) ? tr("Storno für Beleg Nr: %1").arg(id)
                                       : settings.value("receiptPrinterHeading", "KASSABON").toString();
        }

        data["headerText"] = Database::getCustomerText(id);

        docprint.printReceipt(printer, data);
    }
}
