/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#ifndef QLONGLONGSPINBOX_H
#define QLONGLONGSPINBOX_H

#include "globals_ckvsoft.h"

#include <QAbstractSpinBox>
#include <QLineEdit>
#include <QWidget>

class QLongLongSpinBoxPrivate;

class CKVSOFT_EXPORT QLongLongSpinBox : public QAbstractSpinBox {
    Q_OBJECT

    Q_PROPERTY(qlonglong minimum READ minimum WRITE setMinimum)
    Q_PROPERTY(qlonglong maximum READ maximum WRITE setMaximum)

    Q_PROPERTY(qlonglong value READ value WRITE setValue NOTIFY valueChanged USER true)

    qlonglong m_minimum;
    qlonglong m_maximum;
    qlonglong m_value;

public:
    explicit QLongLongSpinBox(QWidget *parent = Q_NULLPTR)
        : QAbstractSpinBox(parent)
    {
        connect(lineEdit(), SIGNAL(textEdited(QString)), this, SLOT(onEditFinished()));
    }

    ~QLongLongSpinBox()
    {
    }

    qlonglong value() const
    {
        return m_value;
    }

    qlonglong minimum() const
    {
        return m_minimum;
    }

    void setMinimum(qlonglong min)
    {
        m_minimum = min;
    }

    qlonglong maximum() const
    {
        return m_maximum;
    }

    void setMaximum(qlonglong max)
    {
        m_maximum = max;
    }

    void setRange(qlonglong min, qlonglong max)
    {
        setMinimum(min);
        setMaximum(max);
    }

    virtual void stepBy(int steps)
    {
        auto new_value = m_value;
        if (steps < 0 && new_value + steps > new_value) {
            new_value = std::numeric_limits<qlonglong>::min();
        } else if (steps > 0 && new_value + steps < new_value) {
            new_value = std::numeric_limits<qlonglong>::max();
        } else {
            new_value += steps;
        }

        lineEdit()->setText(textFromValue(new_value));
        setValue(new_value);
    }

protected:
    // bool event(QEvent *event);
    virtual QValidator::State validate(QString &input, int &pos) const
    {
        Q_UNUSED(pos)
        bool ok;
        qlonglong val = input.toLongLong(&ok);
        if (!ok) return QValidator::Invalid;

        if (val < m_minimum || val > m_maximum) return QValidator::Invalid;

        return QValidator::Acceptable;
    }

    virtual qlonglong valueFromText(const QString &text) const
    {
        return text.toLongLong();
    }

    virtual QString textFromValue(qlonglong val) const
    {
        return QString::number(val);
    }
    // virtual void fixup(QString &str) const;

    virtual QAbstractSpinBox::StepEnabled stepEnabled() const
    {
        return StepUpEnabled | StepDownEnabled;
    }

public Q_SLOTS:
    void setValue(qlonglong val)
    {
        if (m_value != val) {
            lineEdit()->setText(textFromValue(val));
            m_value = val;
        }
    }

    void onEditFinished()
    {
        QString input = lineEdit()->text();
        int pos = 0;
        if (QValidator::Acceptable == validate(input, pos))
            setValue(valueFromText(input));
        else
            lineEdit()->setText(textFromValue(m_value));
    }

Q_SIGNALS:
    void valueChanged(qlonglong v);

private:
    Q_DISABLE_COPY(QLongLongSpinBox)

    Q_DECLARE_PRIVATE(QLongLongSpinBox)
};
#endif // QLONGLONGSPINBOX_H
