/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "coupongenerator.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "database.h"

#include <QDateTime>
#include <QDebug>


CouponGenerator::CouponGenerator(QObject *parent)
    : QObject(parent)
{
}

CouponGenerator::~CouponGenerator()
{
}

#if (QT_VERSION >= QT_VERSION_CHECK(5, 12, 0))
#    include <QRandomGenerator>
QString CouponGenerator::generateCouponCode(int length)
{
    const QString charset
        = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789"; // Zeichen, die im Gutscheincode verwendet werden sollen
    QString couponCode;

    // Initialisiere den Zufallszahlengenerator mit der aktuellen Zeit als Seed
    QRandomGenerator randomGenerator(QDateTime::currentMSecsSinceEpoch());

    for (int i = 0; i < length; ++i) {
        int index = randomGenerator.bounded(charset.length()); // Zufälliger Index im Zeichensatz
        couponCode.append(charset.at(index)); // Füge das zufällig ausgewählte Zeichen zum Gutscheincode hinzu
    }

    return couponCode;
}
#else
#    include <chrono>
#    include <random>
QString CouponGenerator::generateCouponCode(int length)
{
    const QString charset = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
    QString couponCode;

    // Benutze std::random_device für den Seed
    std::random_device rd;

    // Benutze std::seed_seq mit mehreren Entropiequellen
    std::seed_seq seed { rd(), static_cast<unsigned>(std::chrono::system_clock::now().time_since_epoch().count()) };

    // Initialisiere den Mersenne-Twister-Generator
    std::mt19937 gen(seed);

    std::uniform_int_distribution<> dis(0, charset.length() - 1);

    for (int i = 0; i < length; ++i) {
        int index = dis(gen);
        couponCode.append(charset.at(index));
    }

    return couponCode;
}
#endif

bool CouponGenerator::isCouponCodeUnique(const QString &couponCode)
{
    QSqlDatabase dbc = Database::database("COUPON");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT COUNT(*) FROM coupons WHERE code = :code");
    query.bindValue(":code", couponCode);
    query.exec();

    if (query.next()) {
        int count = query.value(0).toInt();
        return count == 0; // Wenn die Anzahl 0 ist, ist der Gutscheincode eindeutig
    }

    return false; // Fehler beim Ausführen der Abfrage
}

QString CouponGenerator::generateUniqueCouponCode(int length)
{
    QString couponCode;
    bool unique = false;

    while (!unique) {
        couponCode = generateCouponCode(length);
        unique = isCouponCodeUnique(couponCode);
    }

    return couponCode;
}
