/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "settings.h"
#include "abstractdatabase.h"
#include "csqlquery.h"
#include "journal.h"

#include <QApplication>
#include <QDebug>
#include <QSettings>
#include <QSqlError>

Settings::Settings(QObject *parent)
    : QSettings(parent)
{
    QString appName = qApp->property("configuration").toString();
    if (appName.isEmpty())
        appName = qApp->applicationName();
    else
        appName.prepend(qApp->applicationName().append("_"));

    m_settings = new QSettings(QSettings::IniFormat, QSettings::UserScope, "ckvsoft", appName, this);
}

Settings::~Settings()
{
}

void Settings::beginGroup(const QString &prefix)
{
    m_settings->beginGroup(prefix);
}

void Settings::endGroup()
{
    m_settings->endGroup();
}

void Settings::save2Database(QString name, QString value)
{

    QSqlDatabase dbc = AbstractDataBase::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QVariant oldValue;
    QString oldStrValue;
    AbstractDataBase::select_globals(name, oldValue, oldStrValue);

    if ((oldStrValue.isEmpty() && !value.isEmpty()) || oldStrValue != value) {
        AbstractDataBase::insert2globals(name, QVariant(), value);
        globalStringValues.insert(name, value);
        QString text;
    }
}

void Settings::save2Settings(QString name, QVariant value, Journal *journaling)
{
    QVariant oldValue = m_settings->value(name);
    if (oldValue != value) {
        m_settings->setValue(name, value);
        AbstractDataBase::updateGlobals(name, Q_NULLPTR, Q_NULLPTR);
        QString text = QString("Parameter '%1' aus der Konfigurationsdatei wurde "
                               "von '%2' auf '%3' geändert")
                           .arg(name)
                           .arg(oldValue.toString())
                           .arg(value.toString());

        if (journaling) journaling->journalInsertLine("Settings", text);
    }
}

void Settings::removeSettings(QString name, Journal *journaling)
{
    QVariant oldValue = m_settings->value(name);
    if (oldValue.isValid()) {
        QString text = QString("Parameter '%1' aus der Konfigurationsdatei wurde "
                               "von '%2' auf 'standard' geändert")
                           .arg(name)
                           .arg(oldValue.toString());
        if (journaling) journaling->journalInsertLine("Settings", text);
    }

    QString f = m_settings->fileName();

    m_settings->remove(name);
}

QVariant Settings::value(QString key, QVariant defaultValue)
{
    return m_settings->value(key, defaultValue);
}

QString Settings::fileName()
{
    return m_settings->fileName();
}

QString Settings::getConfigName()
{
    QString name = qApp->property("configuration").toString();
    if (!name.isEmpty()) name = "[ " + name + " ]";

    return name;
}
