/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "cashbookdelegate.h"
#include "cashbookdialog.h"
#include "database.h"

#include <QDateTime>
#include <QStandardItemModel>

CashBookDelegate::CashBookDelegate(CASHBOOK_COLUMN column, int extraColumn, QObject *parent)
    : QStyledItemDelegate(parent)
    , m_current_column(column)
    , m_extraColumn(extraColumn)

{
}

QString CashBookDelegate::displayText(const QVariant &value, const QLocale &locale) const
{

    QString formattedNum;

    switch (m_current_column) {
        case CURRENCY:
            formattedNum = locale.toString(value.toDouble() / 100, 'f', 2) + " " + Database::getShortCurrency();
            break;
        case FLOW:
            formattedNum = value.toInt() == CB_FLOW::FLOW_IN ? tr("Eingang") : tr("Ausgang");
            break;
        case TYPE:
            formattedNum = value.toInt() <= CB_TYPE::TYPE_BUSINESS ? tr("Betrieblich") : tr("Privat");
            break;
        case STORNO:
            switch (value.toInt()) {
                case 1:
                    formattedNum = tr("Storno von Kassenbeleg");
                    break;
                case 2:
                    formattedNum = tr("Storniert");
                    break;
                default:
                    formattedNum = "OK";
                    break;
            }
            break;
        case DATETIME:
            formattedNum = locale.toString(QDateTime::fromString(value.toString(), Qt::ISODate), QLocale::ShortFormat);
            break;
        default:
            break;
    }

    return formattedNum;
}

void CashBookDelegate::paint(QPainter *painter, const QStyleOptionViewItem &option, const QModelIndex &index) const
{
    QStyleOptionViewItem alignedOption(option);
    alignedOption.displayAlignment = Qt::AlignLeft | Qt::AlignVCenter;
    int deleted;
    if (m_current_column == CURRENCY) {
        alignedOption.displayAlignment = Qt::AlignRight | Qt::AlignVCenter;
        if (m_extraColumn >= 0) {

            QModelIndex idx = index.sibling(index.row(), m_extraColumn);
            deleted = index.model()->data(idx, Qt::DisplayRole).toInt();
            if (deleted != 0) alignedOption.font.setStrikeOut(true);
        }
        // int deleted = index.model()->data(index, Qt::DisplayRole).toInt();
        //        alignedOption.font.setStrikeOut(true);
        int value = index.model()->data(index, Qt::DisplayRole).toInt();
        if (value < 0) {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::red));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::red));
        }
    }
    if (m_current_column == FLOW) {
        alignedOption.displayAlignment = Qt::AlignCenter;
        if (index.model()->data(index, Qt::DisplayRole).toInt() == CB_FLOW::FLOW_OUT) {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::red));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::red));
        } else {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::green));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::green));
        }
    }
    if (m_current_column == TYPE) {
        alignedOption.displayAlignment = Qt::AlignCenter;
    }
    if (m_current_column == STORNO) {
        if (index.model()->data(index, Qt::DisplayRole).toInt() == 0) {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::green));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::green));
        } else {
            alignedOption.palette.setColor(QPalette::Text, QColor(Qt::red));
            alignedOption.palette.setColor(QPalette::HighlightedText, QColor(Qt::red));
        }
    }

    QStyledItemDelegate::paint(painter, alignedOption, index);
}
