/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "couponredemptionwidget.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "database.h"

#include <QApplication>
#include <QStyledItemDelegate>
#include <QTextOption>

class RightAlignPercentageDelegate : public QStyledItemDelegate {
public:
    RightAlignPercentageDelegate(QObject *parent = nullptr)
        : QStyledItemDelegate(parent)
    {
    }

    QString displayText(const QVariant &value, const QLocale &locale) const override
    {
        if (value.canConvert<int>()) {
            int intValue = value.toInt();
            QBCMath percentageValue(intValue);
            percentageValue /= 100;
            percentageValue.round(2);
            return percentageValue.toLocale() + " " + Database::getShortCurrency();
        }
        return QStyledItemDelegate::displayText(value, locale);
    }

    void initStyleOption(QStyleOptionViewItem *option, const QModelIndex &index) const override
    {
        QStyledItemDelegate::initStyleOption(option, index);
        option->displayAlignment = Qt::AlignRight | Qt::AlignVCenter;
    }
};

CouponRedemptionWidget::CouponRedemptionWidget(QWidget *parent)
    : QWidget(parent)
{
    QVBoxLayout *layout = new QVBoxLayout(this);

    tableView = new QTableView(this);
    tableView->horizontalHeader()->setSectionResizeMode(QHeaderView::Stretch);
    progressBar = new QProgressBar(this);

    layout->addWidget(tableView);
    layout->addWidget(progressBar);

    setLayout(layout);

    setupTableView();
}

void CouponRedemptionWidget::updateData(int couponId, int maxAmount)
{
    this->couponId = couponId;
    this->maxAmount = maxAmount;
    progressBar->setRange(0, maxAmount);
    progressBar->setValue(0);

    getCurrentData();
}

void CouponRedemptionWidget::getCurrentData()
{
    QSqlDatabase dbc = Database::database();

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT redemption_date, redemption_amount FROM coupon_redemptions WHERE coupon_id = :couponId");
    query.bindValue(":couponId", couponId);

    if (!query.exec()) {
        return;
    }

    redeemedAmounts.clear();
    redemptionDates.clear();
    model->removeRows(0, model->rowCount());
    progressBar->setValue(0);

    while (query.next()) {
        redemptionDates.append(query.value(0).toDateTime());
        redeemedAmounts.append(query.value(1).toInt());
    }

    for (int i = 0; i < redemptionDates.size(); ++i) {
        QList<QStandardItem *> rowItems;
        rowItems << new QStandardItem(QLocale().toString(redemptionDates[i], QLocale::ShortFormat));
        rowItems << new QStandardItem(QString::number(redeemedAmounts[i]));
        model->appendRow(rowItems);
        progressBar->setValue(progressBar->value() + redeemedAmounts[i]);
    }

    QBCMath pValue(progressBar->value());
    pValue /= 100;
    pValue.round(2);
    QBCMath aValue(maxAmount);
    aValue /= 100;
    aValue.round(2);

    progressBar->setFormat(pValue.toLocale() + " / " + aValue.toLocale());
}

void CouponRedemptionWidget::setupTableView()
{
    model = new QStandardItemModel(this);
    model->setColumnCount(2);
    model->setHorizontalHeaderLabels(QStringList() << tr("Einlösungsdatum") << tr("Einlösebetrag"));

    tableView->setModel(model);

    // Kombiniertes Delegate setzen für Rechtsausrichtung und Prozentanzeige
    tableView->setItemDelegateForColumn(1, new RightAlignPercentageDelegate(tableView));

    // Tabelle sortierbar machen
    tableView->setSortingEnabled(true);
    tableView->sortByColumn(0, Qt::AscendingOrder); // Standardmäßig nach dem Datum sortieren

    // Zeilennummerierung ausschalten
    tableView->verticalHeader()->hide();
}
