/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 * LillePOS parts Copyright 2010, Martin Koller, kollix@aon.at
 */

#include "qrkgastroopenticketwidget.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "database.h"
#include "qrkgastro.h"
#include "qrkgastrotableorder.h"

#include <QEvent>
#include <QHeaderView>
#include <QLabel>
#include <QPushButton>
#include <QSqlError>
#include <QTreeWidget>
#include <QVBoxLayout>

TreeWidget::TreeWidget(QWidget *parent)
    : QTreeWidget(parent)
{
    setTextElideMode(Qt::ElideMiddle);
    setExpandsOnDoubleClick(false);
}

bool TreeWidget::viewportEvent(QEvent *event)
{
    if (event->type() == QEvent::MouseButtonPress) emit selected();

    return QTreeWidget::viewportEvent(event);
}

QrkGastroOpenTicketWidget::QrkGastroOpenTicketWidget(QWidget *parent, int theId, int theTablem, QString theName)
    : QFrame(parent)
    , id(theId)
    , table(theTablem)
    , selected(false)
    , toServeWidget(Q_NULLPTR)
{
    setAutoFillBackground(true);
    setFixedWidth(300);
    setFrameShape(QFrame::Box);
    setSelected(selected);

    QVBoxLayout *vbox = new QVBoxLayout(this);
    QHBoxLayout *hbox = new QHBoxLayout;
    QHBoxLayout *hbox2 = new QHBoxLayout;

    QLabel *tableLabel = new QLabel(this);
    tableLabel->setAlignment(Qt::AlignRight);

    QLabel *tableNameLabel = new QLabel(this);
    if (!theName.isEmpty()) tableNameLabel->setText(tr("Gast: <b>%1</b>").arg(theName));
    hbox->addWidget(tableNameLabel);

    tableLabel->setText(
        tr("Tisch: <b>%1/%2</b>").arg(QRKGastro::getRoomNameFromTableId(table)).arg(QRKGastro::getTableName(table)));
    hbox->addWidget(tableLabel);

    TreeWidget *ticketTree = new TreeWidget(this);
    QTreeWidgetItem *headerItem = ticketTree->headerItem();
    headerItem->setText(0, tr("Anz"));
    headerItem->setText(1, tr("Produkt"));
    headerItem->setText(2, tr("je")); // price per piece

    ticketTree->header()->setStretchLastSection(false);
    ticketTree->header()->setSectionResizeMode(1, QHeaderView::Stretch);
    // ticketTree->header()->setResizeMode(0, QHeaderView::ResizeToContents);
    //    ticketTree->header()->resizeSection(0, 33);
    ticketTree->setRootIsDecorated(false);
    ticketTree->setSelectionMode(QAbstractItemView::NoSelection);

    connect(ticketTree, &TreeWidget::selected, this, &QrkGastroOpenTicketWidget::selectedSlot);
    //    connect(ticketTree, &TreeWidget::selected, this,
    //    &QrkGastroOpenTicketWidget::selectionChanged);

    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    orders.prepare("SELECT ticketorders.count, products.name, products.tax, "
                   "ticketorders.gross, ticketorders.id FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:id");
    orders.bindValue(":id", id);

    orders.exec();
    QBCMath sum(0);
    while (orders.next()) {
        QBCMath gross(orders.value("gross").toDouble());
        gross.round(2);
        QBCMath tax(orders.value("tax").toDouble());
        tax.round(2);

        QTreeWidgetItem *item = new QTreeWidgetItem;
        item->setData(0, Qt::DisplayRole, orders.value("count").toInt()); // count
        item->setData(1, Qt::DisplayRole,
            orders.value("name").toString());                // product name
        item->setData(2, Qt::DisplayRole, gross.toLocale()); // product price
        item->setData(2, QRKGastro::PRODUCT_TAX, tax.toDouble());
        item->setTextAlignment(2, Qt::AlignRight);
        ticketTree->addTopLevelItem(item);

        sum += (gross * orders.value("count").toInt());

        // check for orderextras
        CSqlQuery extras(dbc, Q_FUNC_INFO);
        extras.prepare("SELECT orderextras.type, products.name, products.gross FROM orderextras "
                       " LEFT JOIN products ON orderextras.product=products.id"
                       " WHERE orderId=:id");
        extras.bindValue(":id", orders.value("id").toInt());

        extras.exec();
        while (extras.next()) {
            QBCMath singlegross(extras.value("gross").toDouble());
            singlegross.round(2);
            QTreeWidgetItem *child = new QTreeWidgetItem(item);
            child->setData(0, Qt::DisplayRole, (extras.value("type").toInt() == QRKGastro::TYPE_WITH) ? "+" : "-");
            child->setData(1, Qt::DisplayRole, extras.value("name").toString());
            // child->setData(2, QRKGastro::PRODUCT_PRICE, singlegross.toDouble());

            item->setExpanded(true);
        }
    }

    sum.round(2);
    QLabel *sumLabel = new QLabel(QString("Summe: %1").arg(sum.toLocale()), this);
    sumLabel->setAlignment(Qt::AlignRight);
    hbox2->addWidget(sumLabel);
    vbox->addLayout(hbox);
    vbox->addLayout(hbox2);
    vbox->addWidget(ticketTree);
    resizeColumnsToContents(*ticketTree);
    // show unprinted orders
    orders.prepare("SELECT (ticketorders.count - ticketorders.printed) AS count, "
                   "products.name, ticketorders.id FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:id AND (ticketorders.count > "
                   "ticketorders.printed)");
    orders.bindValue(":id", id);
    orders.exec();
    if (orders.next()) {
        toServeWidget = new QWidget(this);
        QVBoxLayout *vb = new QVBoxLayout(toServeWidget);
        vb->setContentsMargins(0, 0, 0, 0);
        QHBoxLayout *hb = new QHBoxLayout;
        hb->addWidget(new QLabel(tr("Zu Servieren"), this));
        QPushButton *servedButton = new QPushButton(tr("Serviert"));
        hb->addWidget(servedButton);
        connect(servedButton, &QPushButton::clicked, this, &QrkGastroOpenTicketWidget::ordersServed);
        vb->addLayout(hb);

        QTreeWidget *toServe = new TreeWidget(this);
        QTreeWidgetItem *headerItem = toServe->headerItem();
        headerItem->setText(0, tr("Anz"));
        headerItem->setText(1, tr("Produkt"));

        toServe->setColumnCount(2);
        //        toServe->header()->resizeSection(0, 40);
        toServe->setRootIsDecorated(false);
        toServe->setSelectionMode(QAbstractItemView::NoSelection);

        do {
            QTreeWidgetItem *item = new QTreeWidgetItem;
            item->setData(0, Qt::DisplayRole, orders.value("count").toInt()); // count
            item->setData(1, Qt::DisplayRole,
                orders.value("name").toString()); // product name
            toServe->addTopLevelItem(item);

            // check for orderextras
            CSqlQuery extras(dbc, Q_FUNC_INFO);
            extras.prepare("SELECT orderextras.type, products.name, products.gross, products.tax FROM orderextras "
                           " LEFT JOIN products ON orderextras.product=products.id"
                           " WHERE orderId=:id");
            extras.bindValue(":id", orders.value("id").toInt());
            extras.exec();
            while (extras.next()) {
                QBCMath singlegross(extras.value("gross").toDouble());
                singlegross.round(2);

                QTreeWidgetItem *child = new QTreeWidgetItem(item);
                child->setData(0, Qt::DisplayRole, (extras.value("type").toInt() == QRKGastro::TYPE_WITH) ? "+" : "-");
                child->setData(1, Qt::DisplayRole, extras.value("name").toString());
                child->setData(2, QRKGastro::PRODUCT_PRICE, singlegross.toDouble());

                item->setExpanded(true);
            }
        } while (orders.next());

        vb->addWidget(toServe);

        vbox->addWidget(toServeWidget);
        resizeColumnsToContents(*toServe);

        connect(toServe, SIGNAL(selected()), this, SLOT(selectedSlot()));
    }
}

void QrkGastroOpenTicketWidget::setSelected(bool sel)
{
    selected = sel;

    setBackgroundRole(selected ? QPalette::Highlight : QPalette::AlternateBase);
}

void QrkGastroOpenTicketWidget::selectedSlot()
{
    setSelected(!selected);
    emit selectionChanged();
}

// mark unserverd orders as served
void QrkGastroOpenTicketWidget::ordersServed()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    orders.prepare("UPDATE ticketorders SET printed=count WHERE ticketId=:ticketId");
    orders.bindValue(":ticketId", id);
    orders.exec();
    toServeWidget->hide();
}

void QrkGastroOpenTicketWidget::resizeColumnsToContents(QTreeWidget &treeWidget)
{
    int cCols = treeWidget.columnCount();
    int cItems = treeWidget.topLevelItemCount();
    int w;
    int col;
    int i;
    for (col = 0; col < cCols; col++) {
        w = treeWidget.header()->sectionSizeHint(col);
        for (i = 0; i < cItems; i++)
            w = qMax(w, treeWidget.topLevelItem(i)->text(col).size() * 7 + (col == 0 ? treeWidget.indentation() : 0));
        treeWidget.header()->resizeSection(col, w);
    }
}
