/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "importtest.h"
#include "3rdparty/ckvsoft/RK/rk_signaturemodule.h"
#include "3rdparty/ckvsoft/databasemanager.h"
#include "database.h"
#include "import/importworker.h"
#include "qrksettings.h"

#include <QDateTime>
#include <QDebug>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QTemporaryFile>
#include <QThread>

/**
 * Returns a random integer between low and high.
 */
int ImportTest::IntRand(int low, int high)
{
    if (low > high) qSwap(low, high);
    return (qrand() % ((high + 1) - low) + low);
}

/**
 * Returns a random double between 0.0 and 1.0.
 * Replaces the old bit-shifting logic to avoid Clang 'Dead Store' warnings
 * and ensure cross-platform compatibility.
 */
double ImportTest::DoubleRand()
{
    return (qrand() % 10001) / 10000.0;
}

/**
 * Generates a random alphanumeric string.
 */
QString ImportTest::GetRandomString()
{
    const QString possibleCharacters("ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789");
    const int randomStringLength = IntRand(5, 15);

    QString randomString;
    randomString.reserve(randomStringLength);
    for (int i = 0; i < randomStringLength; ++i) {
        int index = IntRand(0, possibleCharacters.length() - 1);
        randomString.append(possibleCharacters.at(index));
    }
    return randomString;
}

/**
 * Generates and processes a random 'receipt' JSON.
 */
bool ImportTest::receipt()
{
    QJsonObject root;
    QJsonObject receiptObj;

    receiptObj.insert("customertext", "Customer Text");
    receiptObj.insert("payedBy", "0");

    QJsonArray items;
    int toCount = IntRand(1, 10);
    QStringList taxList;
    taxList << "0" << "10" << "13" << "20";

    for (int i = 0; i < toCount; i++) {
        QJsonObject item;
        item.insert("count", QString::number(IntRand(1, 5)));
        item.insert("name", QString("Item - %1").arg(GetRandomString()));
        item.insert("gross", QString::number(DoubleRand() * 100, 'f', 2));
        item.insert("tax", taxList.at(IntRand(0, taxList.size() - 1)));
        items.append(item);
    }

    receiptObj.insert("items", items);

    QJsonArray receiptArray;
    receiptArray.append(receiptObj);
    root.insert("receipt", receiptArray);

    ImportWorker worker;
    QByteArray jsonData = QJsonDocument(root).toJson(QJsonDocument::Compact);

    return worker.processJson(jsonData, "dummy_receipt.json");
}

/**
 * Generates and processes a random 'r2b' (Receipt to Bill) JSON.
 */
bool ImportTest::r2b()
{
    QJsonObject root;
    QJsonObject r2bObj;

    QString re = QString::number(QDateTime::currentMSecsSinceEpoch());
    r2bObj.insert("receiptNum", "RE" + re.toUpper());
    r2bObj.insert("gross", QString::number(DoubleRand() * 100, 'f', 2));
    r2bObj.insert("payedBy", "0");
    r2bObj.insert("customerText", "Optional Customer Text");

    QJsonArray r2bArray;
    r2bArray.append(r2bObj);
    root.insert("r2b", r2bArray);

    ImportWorker worker;
    QByteArray jsonData = QJsonDocument(root).toJson(QJsonDocument::Compact);

    return worker.processJson(jsonData, "dummy_r2b.json");
}

/**
 * Main test logic: setup encrypted DB, perform imports, and verify integrity.
 */
bool ImportTest::test()
{
    qDebug() << "Starting ImportTest sequence...";

    // 1. Create temporary file for the database
    QTemporaryFile tempfile;
    if (!tempfile.open()) {
        qCritical() << "Failed to create temporary file!";
        return false;
    }
    QString filename = tempfile.fileName();
    tempfile.close();

    // 2. Open with SQLCIPHER
    Database db;
    if (!db.open(false, "QSQLCIPHER", filename)) {
        qWarning() << "Could not open encrypted test database!";
        return false;
    }

    QrkSettings settings;
    Database::insert2globals("demomode", 1, QVariant());
    Database::insert2globals("lastReceiptNum", 0, QVariant());

    settings.save2Database("shopCashRegisterId", "CipherTest1");
    settings.save2Database("shopName", "CipherTest");

    ReceiptItemModel rec;
    if (!rec.createStartReceipt()) {
        qCritical() << "Failed to create start receipt!";
        return false;
    }

    // Wait briefly for filesystem/timestamp stability
    QThread::msleep(500);

    // 3. Run random imports
    for (int n = 0; n < 10; n++) {
        bool ok = (IntRand(0, 1) == 0) ? receipt() : r2b();
        if (!ok) {
            qCritical() << "Import failed at step" << n;
            return false;
        }
        QThread::msleep(100);
    }

    // 4. Finalize and verify
    Reports rep(Q_NULLPTR, true);
    if (rep.checkEOAny()) {
        if (!rep.endOfDay()) return false;
        if (!rep.endOfMonth()) return false;

        if (rec.createNullReceipt(CONCLUSION_RECEIPT)) {
            Database::setCashRegisterInAktive();
            RKSignatureModule::setDEPactive(false);
        } else {
            qCritical() << "Failed to create conclusion receipt!";
            return false;
        }
    } else {
        qCritical() << "Reporting check (checkEOAny) failed!";
        return false;
    }

    // 5. Encryption check
    if (!AbstractDataBase::isSqlCipher(filename)) {
        qCritical() << "ERROR: Database file is in PLAIN TEXT despite Cipher driver usage!";
        return false;
    }

    qDebug() << "Success: Database is encrypted and readable.";
    return true;
}
