/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "coupondialog.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/rbac/crypto.h"
#include "3rdparty/profeatures/profeatures.h"
#include "coupondelegate.h"
#include "couponexportdialog.h"
#include "couponinoutdialog.h"
#include "couponprinter.h"
#include "couponredemptionwidget.h"
#include "couponsettingswidget.h"
#include "couponstornodialog.h"
#include "database.h"
#include "defines.h"
#include "qrkjournal.h"
#include "qrksettings.h"
#include "qrktimedmessagebox.h"

#include "ui_coupondialog.h"

#include <QIcon>
#include <QLocale>
#include <QMap>
#include <QPixmap>
#include <QSqlDatabase>
#include <QSqlError>

#include <QSqlRelationalTableModel>
#include <QSqlTableModel>
#include <QStandardItem>
#include <QStandardItemModel>
#include <QTimer>

CouponDialog::CouponDialog(QDialog *parent)
    : QDialog(parent)
    , ui(new Ui::CouponDialog)
{

    ui->setupUi(this);

    ui->tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    ui->tableView->setSelectionMode(QAbstractItemView::SingleSelection);
    ui->tableView->setEditTriggers(QAbstractItemView::NoEditTriggers);

    // ui->exportPushButton->setEnabled(true);
    connect(ui->exportPushButton, &QPushButton::clicked, this, [this]() {
        CouponExportDialog dialog(m_model);
        dialog.exec();
    });
    /****************/

    connect(ui->inoutPushButton, &QrkPushButton::clicked, this, &CouponDialog::insertNew);
    connect(ui->closePushButton, &QrkPushButton::clicked, this, &CouponDialog::close);
    connect(ui->fromDateEdit, &QDateTimeEdit::dateTimeChanged, this, &CouponDialog::setupView);
    connect(ui->toDateEdit, &QDateTimeEdit::dateTimeChanged, this, &CouponDialog::setupView);
    connect(ui->searchLineEdit, &QLineEdit::textChanged, this, &CouponDialog::setupView);
    connect(ui->stornoPushButton, &QPushButton::clicked, this, &CouponDialog::onStornoButtonClicked);

    ui->stornoPushButton->setHidden(true);
    ui->exportPushButton->setHidden(true);

    if (RBAC::Instance()->hasPermission("plugin_coupon_access_editbox", true)) ui->inoutPushButton->setEnabled(true);
    readSettings();
    init();
}

CouponDialog::~CouponDialog()
{
    writeSettings();
    delete ui;
}

void CouponDialog::printFunction(const QString &voucherCode, const QString &info)
{
    CouponPrinter printer(voucherCode, info);
    printer.exec();
}

void CouponDialog::insertNew()
{
    CouponInOutDialog inoutDialog;
    inoutDialog.setMinimumDateTime(CouponDialog::getLastEntryDate());
    inoutDialog.exec();

    ui->toDateEdit->setDateTime(QDateTime::currentDateTime());
}

void CouponDialog::onStornoButtonClicked()
{
    QModelIndexList indexList = ui->tableView->selectionModel()->selectedIndexes();
    int row = 0;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }
    int id = m_model->data(m_model->index(row, m_model->fieldIndex("id"), QModelIndex())).toInt();
    if (id > 0) {
        CouponStornoDialog storno(id);
        if (storno.exec() == QDialog::Accepted) {
            setupView();
        }
    }
}

void CouponDialog::onSelectionChanged(const QItemSelection &, const QItemSelection &)
{
    QModelIndexList indexList = ui->tableView->selectionModel()->selectedIndexes();
    int row = 0;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }


    QString storniert;

    int deleted = m_model->data(m_model->index(row, m_model->fieldIndex("deleted"), QModelIndex())).toInt();
    int id = m_model->data(m_model->index(row, m_model->fieldIndex("id"), QModelIndex())).toInt();
    int amount = m_model->data(m_model->index(row, m_model->fieldIndex("gross"), QModelIndex())).toInt();

    ui->widget->updateData(id, amount);
    /*
        if (deleted > 0) {
    #if (QT_VERSION >= QT_VERSION_CHECK(5, 8, 0))
            storniert = tr("Storniert: %1\n").arg(QLocale().toString(QDateTime::fromSecsSinceEpoch(deleted)));
    #else
            storniert = tr("Storniert: %1\n").arg(QLocale().toString(QDateTime::fromMSecsSinceEpoch(deleted * 1000)));
    #endif
        }
    */
    QString description
        = tr("Beschreibung: %1\n")
              .arg(m_model->data(m_model->index(row, m_model->fieldIndex("description"), QModelIndex())).toString());
    QString reference
        = tr("Referenz: Bon Nummer %1\n")
              .arg(m_model->data(m_model->index(row, m_model->fieldIndex("reference"), QModelIndex())).toString());

    ui->textBrowser->setText(QString("%1%2%3").arg(storniert).arg(reference).arg(description));

    if (deleted > 0) {
        ui->stornoPushButton->setEnabled(false);
        return;
    }

    QDateTime date = m_model->data(m_model->index(row, m_model->fieldIndex("timestamp"), QModelIndex())).toDateTime();
    bool today
        = date.date().toString(Qt::ISODate).compare(QDateTime::currentDateTime().date().toString(Qt::ISODate)) == 0;
    bool enabled
        = today && m_model->data(m_model->index(row, m_model->fieldIndex("type"), QModelIndex())).toInt() > TYPE_FIRST;
    ui->stornoPushButton->setEnabled(enabled);
    ui->searchLineEdit->setFocus();
}

QString CouponDialog::getLastEntryDate()
{
    QSqlDatabase dbc = Database::database("COUPON");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT timestamp FROM coupons ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) return query.value("timestamp").toDateTime().toString(Qt::ISODate);

    return {};
}

void CouponDialog::init()
{
    QVariant value = 0;
    QString strValue = "";

    Database::select_globals("currency", value, strValue);
    m_currency = strValue;
    value = 0;
    strValue = "";

    if (!checkDatabase()) {
        if (dataBaseExists()) {
            QrkTimedMessageBox messageBox(20, QMessageBox::Critical, tr("Gutschein Fehler"),
                tr("Fehler in der Gutschein Datenbank. Checksum Error."), QMessageBox::Yes | QMessageBox::Default);

            messageBox.setDetailedText(tr(
                "Mögliche Ursachen.\nDaten wurden direkt in der Datenbank gelöscht.\nDatensätze wurden direkt in der Datenbank geändert.\n\nSie können die Datenbank aus einen aktuellen Backup wiederherstellen."));

            messageBox.setDefaultButton(QMessageBox::Yes);
            messageBox.setButtonText(QMessageBox::Yes, tr("OK"));
            foreach (QAbstractButton *button, messageBox.buttons()) {
                if (messageBox.buttonRole(button) == QrkTimedMessageBox::ActionRole) {
                    button->click();
                    break;
                }
            }
            messageBox.exec();
        }
        QTimer::singleShot(10, this, &CouponDialog::reject);
        return;
    }
    setupView();
    setUiData();
}

void CouponDialog::setUiData()
{
    ui->toDateEdit->setDateTime(QDateTime::currentDateTime());
}

void CouponDialog::setupView()
{

    ui->fromDateEdit->setMaximumDateTime(ui->toDateEdit->dateTime());

    m_model = new QSqlTableModel(this, Database::database("COUPON"));
    m_model->setTable("coupons");
    //    m_model->setQuery("SELECT id, flow, CASE WHEN flow = 0 THEN gross ELSE '' END flowin, CASE WHEN flow = 1 THEN
    //    gross ELSE '' END flowout FROM cashbook;");
    if (ui->searchLineEdit->text().isEmpty()) {
        m_model->setFilter(QString("timestamp BETWEEN '%1' AND '%2'")
                .arg(ui->fromDateEdit->date().toString(Qt::ISODate))
                .arg(ui->toDateEdit->dateTime().toString(Qt::ISODate)));
    } else {
        m_model->setFilter(QString("code like '%%1%' AND timestamp BETWEEN '%2' AND '%3'")
                .arg(ui->searchLineEdit->text())
                .arg(ui->fromDateEdit->date().toString(Qt::ISODate))
                .arg(ui->toDateEdit->dateTime().toString(Qt::ISODate)));
    }
    m_model->select();
    while (m_model->canFetchMore())
        m_model->fetchMore();

    ui->tableView->setModel(m_model);

    m_model->setHeaderData(m_model->fieldIndex("id"), Qt::Horizontal, tr("Nr."), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("timestamp"), Qt::Horizontal, tr("Datum"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("code"), Qt::Horizontal, tr("Gutschein Code"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("gross"), Qt::Horizontal, tr("Betrag"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("credit"), Qt::Horizontal, tr("Guthaben"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("deleted"), Qt::Horizontal, "", Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("type"), Qt::Horizontal, tr("Type"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("tax"), Qt::Horizontal, tr("Steuer"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("checksum"), Qt::Horizontal, "", Qt::DisplayRole);

    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("timestamp"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::DATETIME));

    ui->tableView->setItemDelegateForColumn(m_model->fieldIndex("credit"),
        new CouponDelegate(CouponDelegate::COUPON_COLUMN::CREDIT, m_model->fieldIndex("deleted")));
    ui->tableView->setItemDelegateForColumn(m_model->fieldIndex("gross"),
        new CouponDelegate(CouponDelegate::COUPON_COLUMN::CURRENCY, m_model->fieldIndex("deleted")));

    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("deleted"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::DELETED));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("type"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::TYPE));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("credit"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::CREDIT));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("code"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::CODE));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("tax"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::TAX));

    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("checksum"), new CouponDelegate(CouponDelegate::COUPON_COLUMN::CHECKSUM));


    ui->tableView->setColumnHidden(m_model->fieldIndex("id"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("customer"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("description"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("reference"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("storno"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("stornotimestamp"), true);
    //    ui->tableView->setColumnHidden(m_model->fieldIndex("checksum"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("userid"), true);

    ui->tableView->horizontalHeader()->resizeSections(QHeaderView::ResizeToContents);
    ui->tableView->sortByColumn(m_model->fieldIndex("id"), Qt::DescendingOrder);
    ui->tableView->horizontalHeader()->setStretchLastSection(true);

    disconnect(ui->tableView, &QTableView::clicked, this, &CouponDialog::onTableClicked);
    disconnect(ui->tableView->selectionModel(), &QItemSelectionModel::selectionChanged, this,
        &CouponDialog::onSelectionChanged);

    connect(ui->tableView, &QTableView::clicked, this, &CouponDialog::onTableClicked);
    connect(ui->tableView->selectionModel(), &QItemSelectionModel::selectionChanged, this,
        &CouponDialog::onSelectionChanged);
    ui->tableView->selectRow(0);
    ui->searchLineEdit->setFocus();
}

void CouponDialog::onTableClicked(const QModelIndex &index)
{
    if (index.column() == m_model->fieldIndex("checksum")) {
        QString dataColumn0 = m_model->data(m_model->index(index.row(), m_model->fieldIndex("gross"))).toString();
        QString dataColumn1 = m_model->data(m_model->index(index.row(), m_model->fieldIndex("type"))).toString();
        QString dataColumn2 = m_model->data(m_model->index(index.row(), m_model->fieldIndex("tax"))).toString();
        QString voucherCode = m_model->data(m_model->index(index.row(), m_model->fieldIndex("code"))).toString();

        QBCMath gross(dataColumn0);
        gross /= 100;
        gross.round(2);
        QBCMath tax(dataColumn2);
        tax /= 100;
        tax.round(2);

        QString info = QString("%1 - %2 %3")
                           .arg(dataColumn1.toInt() == 0 ? "Mehrzweck" : QString("Einzweck (%1%)").arg(tax.toLocale()))
                           .arg(gross.toLocale())
                           .arg(Database::getCurrency());

        printFunction(voucherCode, info);
    }
}

bool CouponDialog::checkDatabase()
{
    QSqlDatabase dbc = Database::database("COUPON");
    if (!dataBaseExists()) {
        QDialog dialog;
        QVBoxLayout *layout = new QVBoxLayout;
        CouponSettingsWidget *sw = new CouponSettingsWidget;
        layout->addWidget(sw);
        dialog.setLayout(layout);
        connect(sw, &CouponSettingsWidget::saved, &dialog, &QDialog::accept);
        connect(sw, &CouponSettingsWidget::rejected, &dialog, &QDialog::reject);
        int retvalue = dialog.exec();
        if (retvalue != QDialog::Accepted) return false;
        QrkJournal journal;
        journal.journalInsertLine("Gutschein", "Gutschein Datenbanktabelle angelegt.");
    } else {
        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.prepare("UPDATE coupons SET deleted = 1 WHERE credit = 0;");
        query.exec();
    }

    return true;
}

QString CouponDialog::getLastCheckSum()
{
    QSqlDatabase dbc = Database::database("COUPON");
    QString decrypted = "";
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT checksum FROM coupons ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) {
        SecureByteArray saName = query.value("checksum").toByteArray();
        decrypted = Crypto::decrypt(saName, SecureByteArray("Coupons"), false);
    }
    if (decrypted.isEmpty()) return 0;
    QBCMath cash(decrypted.toInt());
    return cash.getIntPart();
}

void CouponDialog::readSettings()
{
    QrkSettings settings;
    settings.beginGroup("Coupon");
    restoreGeometry(settings.value("WindowGeometry").toByteArray());
    settings.endGroup();

    //    m_cash_repayment = settings.value("cashrepayment", false).toBool();
}

void CouponDialog::writeSettings()
{
    QrkSettings settings;
    settings.beginGroup("Coupon");
    settings.save2Settings("WindowGeometry", saveGeometry());
    settings.endGroup();
}

bool CouponDialog::dataBaseExists()
{
    QSqlDatabase dbc = Database::database("COUPON");
    return dbc.tables(QSql::AllTables).contains(QLatin1String("coupons"));
}
