/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */
#include "abstractdatabase.h"
#include "csqlquery.h"
#include "databasedefinition.h"
#include "databasemanager.h"
#include "preferences/settings.h"
#include "rbac/crypto.h"

#include <QApplication>
#include <QDate>
#include <QDir>
#include <QFile>
#include <QJsonObject>
#include <QMap>
#include <QMessageBox>
#include <QSqlDatabase>
#include <QSqlError>
#include <QStandardPaths>
#include <QString>

#include <QDebug>

QMap<QString, QString> globalStringValues;

AbstractDataBase::AbstractDataBase(QObject *parent)
    : QObject(parent)
{
}

AbstractDataBase::~AbstractDataBase()
{
}

QJsonObject AbstractDataBase::getConnectionDefinition()
{
    QJsonObject jobj;
    jobj.insert("dbtype", getDatabaseType());
    jobj.insert("databasename", globalStringValues.value("databasename"));
    jobj.insert("databasehost", globalStringValues.value("databasehost"));
    jobj.insert("databaseusername", globalStringValues.value("databaseusername"));
    jobj.insert("databasepassword", globalStringValues.value("databasepassword"));
    jobj.insert("databaseoptions", globalStringValues.value("databaseoptions"));
    return jobj;
}

bool AbstractDataBase::open(bool dbSelect, const QString &defaultdbtype, QString TESTFILE)
{

    Settings settings;

    QJsonObject ConnectionDefinition = AbstractDataBase::getConnectionDefinition();

    QString dbType = ConnectionDefinition["dbtype"].toString();

    if (dbType.isEmpty()) {
        if (!defaultdbtype.isEmpty()) {
            dbType = defaultdbtype;
            settings.save2Settings("DB_type", dbType);
        }
    } else if (dbSelect) {
        DatabaseDefinition dialog(Q_NULLPTR);

        if (dialog.exec() == QDialog::Rejected) return false;

        dbType = dialog.getDbType();

        settings.save2Settings("DB_type", dbType);
        settings.save2Settings("DB_userName", dialog.getUserName());
        settings.save2Settings("DB_password", dialog.getPassword());
        settings.save2Settings("DB_hostName", dialog.getHostName());
    }

    QDate date = QDate::currentDate();

    QString standardDataDir = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation) + "/data";

    QDir t(qApp->applicationDirPath() + "/data");
    QStringList filter;
    filter << "*.db";
    if (!t.entryList(filter, QDir::NoDotAndDotDot).empty())
        settings.save2Settings("sqliteDataDirectory", qApp->applicationDirPath() + "/data");

    QString dataDir = settings.value("sqliteDataDirectory", standardDataDir).toString();
    QDir dir(dataDir);

    if (!dir.exists()) dir.mkpath(".");

    QFileInfo fi(settings.fileName());
    QString basename = fi.baseName();

    if (!QFile::exists(QString(dataDir + "/%1-%2.db").arg(date.year()).arg(basename))) {
        if (QFile::exists(QString(dataDir + "/%1-%2.db").arg(date.year() - 1).arg(basename))) {

            QString oldDbFile = QString(dataDir + "/%1-%2.db").arg(date.year() - 1).arg(basename);
            QString newDbFile = QString(dataDir + "/%1-%2.db").arg(date.year()).arg(basename);

            QSqlDatabase currentConnection;
            if (currentConnection.isOpen()) currentConnection.close();

            QString migrationDriver = QSqlDatabase::drivers().contains("QSQLCIPHER") ? "QSQLCIPHER" : "QSQLITE";

            currentConnection = QSqlDatabase::addDatabase(migrationDriver, "MIGRATION_CON");
            currentConnection.setDatabaseName(oldDbFile);

            if (currentConnection.open()) {
                QString key = Crypto::encrypt("QSQLCIPHER", SecureByteArray("Globals"));
                CSqlQuery query(currentConnection, Q_FUNC_INFO);

                if (isSqlCipher(oldDbFile) && migrationDriver == "QSQLCIPHER") {
                    query.exec(QString("PRAGMA key = '%1';").arg(key));
                }

                bool exportOk = false;
                if (migrationDriver == "QSQLCIPHER") {
                    query.exec(QString("ATTACH DATABASE '%1' AS newyear KEY '%2';").arg(newDbFile).arg(key));
                    if (query.exec("SELECT sqlcipher_export('newyear');")) {
                        query.exec("DETACH DATABASE newyear;");
                        exportOk = true;
                    }
                }

                if (!exportOk) {
                    if (!query.exec(QString("VACUUM main INTO '%1';").arg(newDbFile))) {
                        qWarning() << "Migration failed, simple copy used.";
                        QFile::copy(oldDbFile, newDbFile);
                    }
                }

                currentConnection.close();

                if (migrationDriver == "QSQLCIPHER") {
                    dbType = "QSQLCIPHER";
                    settings.save2Settings("DB_type", "QSQLCIPHER");
                    globalStringValues.insert("DB_type", "QSQLCIPHER");
                }
            }
            QSqlDatabase::removeDatabase("MIGRATION_CON");

        } else {
            // No old database found, so we ensure the next one created is Cipher if possible
            if (dbType == "QSQLITE" || dbType.isEmpty()) {
                if (QSqlDatabase::drivers().contains("QSQLCIPHER")) {
                    dbType = "QSQLCIPHER";
                } else {
                    dbType = "QSQLITE"; // Fallback, falls kein Cipher-Treiber da
                }
            }
        }
    }

    if (dbType == "QSQLCIPHER" && !QSqlDatabase::drivers().contains("QSQLCIPHER")) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "QSQLCIPHER driver missing! Falling back to QSQLITE.";
        dbType = "QSQLITE";
    }

    QString connectionName = "";
    {
        QSqlDatabase currentConnection;
        if (currentConnection.isOpen()) currentConnection.close();

        if (!TESTFILE.isEmpty()) {
            dbType = "QSQLCIPHER";
            globalStringValues.insert("databasename", TESTFILE);
            globalStringValues.insert("DB_type", "QSQLCIPHER");
            currentConnection = QSqlDatabase::addDatabase("QSQLCIPHER", "CN");
            currentConnection.setDatabaseName(TESTFILE);
        }
        // setup database connection
        else if (dbType == "QSQLITE" || dbType == "QSQLCIPHER") {
            QString databasename = QString(dataDir + "/%1-%2.db").arg(date.year()).arg(basename);
            globalStringValues.insert("databasename", databasename);
            if (isSqlCipher(databasename) || dbType == "QSQLCIPHER") {
                dbType = "QSQLCIPHER";
                settings.save2Settings("DB_type", dbType);
            }
            currentConnection = QSqlDatabase::addDatabase(dbType, "CN");
            currentConnection.setDatabaseName(databasename);
        } else if (dbType == "QMYSQL") {
            QString userName = settings.value("DB_userName", "QRK").toString();
            QString password = settings.value("DB_password", "").toString();
            QString hostName = settings.value("DB_hostName", "localhost").toString();

            currentConnection = QSqlDatabase::addDatabase("QMYSQL", "CN");
            currentConnection.setHostName(hostName);
            currentConnection.setUserName(userName);
            currentConnection.setPassword(password);
            currentConnection.setConnectOptions("MYSQL_OPT_RECONNECT=1;MYSQL_OPT_CONNECT_TIMEOUT=86400;MYSQL_OPT_"
                                                "READ_TIMEOUT=60");
            globalStringValues.insert("databasehost", hostName);
            globalStringValues.insert("databaseusername", userName);
            globalStringValues.insert("databasepassword", password);
            globalStringValues.insert("databaseoptions",
                "MYSQL_OPT_RECONNECT=1;MYSQL_OPT_CONNECT_"
                "TIMEOUT=86400;MYSQL_OPT_READ_TIMEOUT=60");
        }

        bool ok = currentConnection.open();
        // Falls Cipher: Schlüssel
        if (dbType == "QSQLCIPHER") {
            QString key = Crypto::encrypt("QSQLCIPHER", SecureByteArray("Globals")); //
            if (!key.isEmpty()) {
                CSqlQuery q(currentConnection, Q_FUNC_INFO);
                if (!q.exec(QString("PRAGMA key = '%1';").arg(key))) {
                    qWarning() << "PRAGMA key error:" << q.lastError().text();
                }
            }
        }

        if (!ok) {
            QMessageBox errorDialog;
            errorDialog.setIcon(QMessageBox::Critical);
            errorDialog.addButton(QMessageBox::Ok);
            errorDialog.setText(currentConnection.lastError().text());
            errorDialog.setWindowTitle(QObject::tr("Datenbank Verbindungsfehler"));
            errorDialog.exec();
            return false;
        }

        if (dbType == "QMYSQL") {
            CSqlQuery query(currentConnection, Q_FUNC_INFO);
            query.prepare(QString("SHOW DATABASES LIKE '%1'").arg(basename));
            query.exec();
            QString errorText = "";
            if (!query.next()) { // db does not exist
                query.prepare(QString("CREATE DATABASE %1").arg(basename));
                query.exec();
                errorText = query.lastError().text();
                qDebug() << "Function Name: " << Q_FUNC_INFO << query.lastError().text();
            }

            currentConnection.close();
            currentConnection.setDatabaseName(basename);
            globalStringValues.insert("databasename", basename);
            if (!currentConnection.open()) {
                QMessageBox errorDialog;
                errorDialog.setIcon(QMessageBox::Critical);
                errorDialog.addButton(QMessageBox::Ok);
                errorDialog.setText(errorText + "\n" + currentConnection.lastError().text());
                errorDialog.setWindowTitle(QObject::tr("Datenbank Verbindungsfehler"));
                errorDialog.exec();
                return false;
            }
        }

        QString lastError;
        if (dbType == "QSQLITE" || dbType == "QSQLCIPHER") {
            if (QFile::exists(QString(dataDir + "/%1-%2.db-journal").arg(date.year()).arg(basename))) {
                CSqlQuery query(currentConnection, Q_FUNC_INFO);
                currentConnection.transaction();
                query.exec("create table force_journal_cleanup (id integer primary key);");
                currentConnection.rollback();
                lastError = query.lastError().text();
            }

            {
                CSqlQuery query(currentConnection, Q_FUNC_INFO);
                query.exec("PRAGMA integrity_check;");
                query.next();
                QString result = query.value(0).toString();
                lastError = query.lastError().text();
                if (result != "ok") {
                    QMessageBox errorDialog;
                    errorDialog.setIcon(QMessageBox::Critical);
                    errorDialog.addButton(QMessageBox::Ok);
                    errorDialog.setText(lastError + ".\n"
                        + tr("Die SQL-Datenbank wurde beschädigt. Stellen Sie diese von "
                             "einen aktuellen Backup wieder her."));
                    errorDialog.setWindowTitle(QObject::tr("Datenbank Fehler"));
                    errorDialog.exec();
                    return false;
                }
            }

            if (QFile::exists(QString(dataDir + "/%1-%2.db-journal").arg(date.year()).arg(basename))) {
                QMessageBox errorDialog;
                errorDialog.setIcon(QMessageBox::Critical);
                errorDialog.addButton(QMessageBox::Ok);
                errorDialog.setText(lastError + ".\n"
                    + tr("Eventuell müssen andere geöffnete Anwendungen "
                         "geschlossen werden."));
                errorDialog.setWindowTitle(QObject::tr("Datenbank Fehler"));
                errorDialog.exec();
                return false;
            }
        }

        bool database_exists = false;
        if (!doDatabaseStuff(currentConnection, database_exists)) return false;

        CSqlQuery query(currentConnection, Q_FUNC_INFO);

        if (dbType == "QSQLITE" || dbType == "QSQLCIPHER") {
            // 1. Enforce foreign key constraint
            query.exec("PRAGMA foreign_keys = 1;");

            // 2. Handle Journal Mode
            if (query.exec("PRAGMA journal_mode;")) {
                query.next();
                QString mode = query.value(0).toString().toLower();
                if (mode != "wal") {
                    if (query.exec("PRAGMA journal_mode = WAL;")) {
                        qDebug() << "Function Name: " << Q_FUNC_INFO << "Changed SQLite mode from" << mode
                                 << "to \"wal\"";
                    } else {
                        qWarning() << "Function Name: " << Q_FUNC_INFO
                                   << "Failed to set WAL mode:" << query.lastError().text();
                    }
                }
            }
        }

        connectionName = currentConnection.connectionName();
        currentConnection.close();
    }

    QSqlDatabase::removeDatabase(connectionName);
    return true;
}

bool AbstractDataBase::isSqlCipher(const QString &filePath)
{
    QFileInfo fi(filePath);

    if (!fi.exists() || fi.size() == 0) return true;

    QFile file(filePath);
    if (!file.open(QIODevice::ReadOnly)) return true;

    QByteArray header = file.read(16);
    file.close();

    return !header.startsWith("SQLite format 3");
}

void AbstractDataBase::insert2globals(const QString &name, const QVariant &value, const QVariant &strValue)
{
    QSqlDatabase dbc = AbstractDataBase::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    SecureByteArray saName = name.toUtf8();
    SecureByteArray saStrValue = strValue.toByteArray();
    QString secureName = Crypto::encrypt(saName, SecureByteArray("Globals"));
    QString secureStrValue = Crypto::encrypt(saStrValue, SecureByteArray("Globals"));

    bool ok;
    if (exists("globals", secureName, "name")) {
        query.prepare("UPDATE globals SET value=:value, strValue=:strValue WHERE name=:name");
        query.bindValue(":name", secureName);
        query.bindValue(":value", value);
        if (strValue.isNull())
            query.bindValue(":strValue", QString());
        else
            query.bindValue(":strValue", secureStrValue);
        if ((ok = query.exec())) updateGlobals(name, value.toString(), strValue.toString());
    } else {
        query.prepare("INSERT INTO globals (name, value, strValue) VALUES(:name, "
                      ":value, :strValue)");
        query.bindValue(":name", secureName);
        query.bindValue(":value", value);
        if (strValue.isNull())
            query.bindValue(":strValue", QString());
        else
            query.bindValue(":strValue", secureStrValue);
        if ((ok = query.exec())) updateGlobals(name, value.toString(), strValue.toString());
    }
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }
}

int AbstractDataBase::select_globals(const QString &name, QVariant &value, QString &strValue, QString addWhere)
{
    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    SecureByteArray saName = name.toUtf8();
    QString secureName = Crypto::encrypt(saName, SecureByteArray("Globals"));

    value = QVariant();
    strValue = QString();

    QString sql = "SELECT DISTINCT id, value, strValue FROM globals WHERE name=:name";
    if (!addWhere.isEmpty()) sql.append(" " + addWhere);

    query.prepare(sql);
    query.bindValue(":name", secureName);
    bool ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.next()) {
        SecureByteArray saStrValue = query.value("strValue").toByteArray();

        value = query.value("value"); // .toString().isNull()?QVariant():Crypto::decrypt(saValue,
                                      // SecureByteArray("Globals"));
        strValue = query.value("strValue").toString().isNull()
            ? QString()
            : Crypto::decrypt(saStrValue, SecureByteArray("Globals"));
        return query.value("id").toInt();
    }

    return -1;
}

bool AbstractDataBase::select_globals(const QString &name, QMap<QString, QVariant> &values, QString addWhere)
{
    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    SecureByteArray saName = name.toUtf8();
    QString secureName = Crypto::encrypt(saName, SecureByteArray("Globals"));

    values.clear();

    QString sql = "SELECT id, value, strValue FROM globals WHERE name=:name";
    if (!addWhere.isEmpty()) sql.append(" " + addWhere);

    query.prepare(sql);
    query.bindValue(":name", secureName);
    bool ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    while (query.next()) {
        SecureByteArray saStrValue = query.value("strValue").toByteArray();
        values.insert(query.value("strValue").toString().isNull()
                ? QString()
                : Crypto::decrypt(saStrValue, SecureByteArray("Globals")),
            query.value("value"));
    }

    return !values.isEmpty();
}

void AbstractDataBase::delete_globals(const QString &name, const QString &where)
{
    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    SecureByteArray saName = name.toUtf8();
    QString secureName = Crypto::encrypt(saName, SecureByteArray("Globals"));

    QString sql = "DELETE FROM globals WHERE name=:name";
    if (!where.isEmpty()) sql.append(" " + where);

    query.prepare(sql);
    query.bindValue(":name", secureName);
    bool ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }
}

QString AbstractDataBase::updateGlobals(const QString &name, QString defaultvalue, QString defaultStrValue)
{
    if (defaultStrValue.isNull() && defaultvalue.isNull()) {
        globalStringValues.remove(name);
        return "";
    }

    if (!defaultStrValue.isNull()) globalStringValues.insert(name, defaultStrValue);

    QVariant value;
    QString strValue;
    int id = select_globals(name, value, strValue);
    if (id > 0) {
        defaultvalue = value.toString().isNull() ? Q_NULLPTR : value.toString();
        defaultStrValue = strValue.isNull() ? Q_NULLPTR : strValue;
        if (!defaultStrValue.isEmpty()) globalStringValues.insert(name, defaultStrValue);
    }

    return defaultvalue.isNull() ? defaultStrValue : defaultvalue;
}

QSqlDatabase AbstractDataBase::database(const QString &connectionname)
{
    QSqlDatabase dbc = DatabaseManager::database(connectionname);
    if (!dbc.lastError().nativeErrorCode().isEmpty())
        qDebug() << "Function Name: " << Q_FUNC_INFO << dbc.lastError().text() << " #"
                 << dbc.lastError().nativeErrorCode();
    return dbc;
}

bool AbstractDataBase::exists(const QString &name)
{
    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("select p2.id from (select max(version) as version, origin "
                  "from products group by origin) p1 inner join (select * from "
                  "products) as  p2 on p1.version=p2.version and "
                  "p1.origin=p2.origin WHERE name=:name AND visible >= 0");
    query.bindValue(":name", name);

    query.exec();
    if (query.next()) return true;

    return false;
}

bool AbstractDataBase::exists(const QString &type, const int &id, const QString &fieldname)
{

    if (id == 0) return true;

    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("SELECT id FROM %1 WHERE %2=:id").arg(type).arg(fieldname));
    query.bindValue(":id", id);

    query.exec();
    if (query.next()) return true;

    return false;
}

bool AbstractDataBase::exists(const QString &type, const QString &name, const QString &fieldname)
{

    if (name.isEmpty()) return true;

    QSqlDatabase dbc = database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    if (type == "products")
        query.prepare(QString("SELECT id FROM %1 WHERE %2=:name AND visible >= 0").arg(type).arg(fieldname));
    else
        query.prepare(QString("SELECT id FROM %1 WHERE %2=:name").arg(type).arg(fieldname));

    query.bindValue(":name", name);

    query.exec();
    if (query.next()) return true;

    return false;
}

QString AbstractDataBase::getLastExecutedQuery(const CSqlQuery &query)
{
    QString str = query.lastQuery();
    QMapIterator<QString, QVariant> it(query.boundValues());
    it.toBack();

    while (it.hasPrevious()) {
        it.previous();
        str.replace(it.key(), it.value().toString());
    }

    return str;
}

QString AbstractDataBase::getDatabaseType()
{

    if (globalStringValues.contains("DB_type")) return globalStringValues.value("DB_type");

    // read global defintions (DB, ...)
    Settings settings;

    return settings.value("DB_type").toString();
}

QString AbstractDataBase::getBaseDriver(const QSqlDatabase &db)
{
    QString name = db.driverName();
    if (name.isEmpty()) return "UNKNOWN";
    if (name == "QSQLCIPHER") return "QSQLITE";
    return name;
}

bool AbstractDataBase::doDatabaseStuff(QSqlDatabase & /* currentConnection */, bool & /* database_exists */)
{
    return false;
}
