/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */


#include "qrkgastroopentickets.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "database.h"
#include "history.h"
#include "qrkgastro.h"
#include "qrkgastrofinishticket.h"
#include "qrkgastrosplitticketwidget.h"
#include "qrkgastrotableorder.h"
#include "qrkpaymentdialog.h"
#include "qrksettings.h"
#include "reports.h"
#include "ui_qrkgastroopentickets.h"

#include <QAction>
#include <QDateTime>
#include <QMenu>
#include <QMessageBox>
#include <QSqlError>

#include <QTreeWidget>

#include <QDebug>

QRKGastroOpenTickets::QRKGastroOpenTickets(QWidget *parent)
    : QWidget(parent)
    , ui(new Ui::QRKGastroOpenTickets)
{
    ui->setupUi(this);
    connect(ui->newTicket, &QrkPushButton::clicked, this,
        static_cast<void (QRKGastroOpenTickets::*)(bool)>(&QRKGastroOpenTickets::newTicket));
    connect(ui->voidTicket, &QrkPushButton::clicked, this, &QRKGastroOpenTickets::voidTicket);
    connect(ui->changeTicket, &QrkPushButton::clicked, this,
        static_cast<void (QRKGastroOpenTickets::*)(bool)>(&QRKGastroOpenTickets::changeTicket));
    connect(ui->splitTicket, &QrkPushButton::clicked, this, &QRKGastroOpenTickets::splitTicket);
    connect(ui->moveTicket, &QrkPushButton::clicked, this, &QRKGastroOpenTickets::moveTicket);
    connect(ui->leaveTicket, &QrkPushButton::clicked, this, &QRKGastroOpenTickets::leaveTicket);
    connect(ui->interimCalculation, &QPushButton::clicked, this, &QRKGastroOpenTickets::calculateTicket);
    connect(ui->payTicket, &QPushButton::clicked, this,
        static_cast<void (QRKGastroOpenTickets::*)()>(&QRKGastroOpenTickets::payTicket));
    connect(ui->openTickets, &QrkGastroOpenTicketsListWidget::selectionChanged, this,
        &QRKGastroOpenTickets::selectionChanged);

    QAction *payAction = new QAction(tr("Bezahlen"), this);
    QAction *payCashAction = new QAction(tr("Bar bezahlen"), this);

    QMenu *payMenu = new QMenu;
    payMenu->addAction(payAction);
    payMenu->addAction(payCashAction);

    ui->toolButton->setMenu(payMenu);
    ui->toolButton->setDefaultAction(payAction);

    payAction->setIcon(QIcon(":src/icons/pay.png"));
    payCashAction->setIcon(QIcon(":src/icons/money48.png"));

    connect(payAction, &QAction::triggered, this,
        static_cast<void (QRKGastroOpenTickets::*)()>(&QRKGastroOpenTickets::payTicket));
    connect(payCashAction, &QAction::triggered, [=]() {
        m_paycash = true;
        payTicket();
    });
}

QRKGastroOpenTickets::~QRKGastroOpenTickets()
{
    delete ui;
}

void QRKGastroOpenTickets::selectionChanged()
{
    QList<int> selected = ui->openTickets->getSelectedTickets();
    QrkSettings settings;
    bool custompaybutton = settings.value("Gastro/custompaybutton", false).toBool();
    ui->payTicket->setHidden(custompaybutton);
    ui->toolButton->setVisible(custompaybutton);

    if (selected.count() == 0) {
        ui->changeTicket->setEnabled(false);
        ui->payTicket->setEnabled(false);
        ui->toolButton->setEnabled(false);
        ui->splitTicket->setEnabled(false);
        ui->voidTicket->setEnabled(false);
        ui->moveTicket->setEnabled(false);
        ui->interimCalculation->setEnabled(false);
    } else {
        ui->changeTicket->setEnabled(true);
        ui->payTicket->setEnabled(true);
        ui->toolButton->setEnabled(true);
        ui->splitTicket->setEnabled(true);
        ui->voidTicket->setEnabled(true);
        ui->moveTicket->setEnabled(true);
        ui->interimCalculation->setEnabled(true);
    }
}

int QRKGastroOpenTickets::setTableId(int id)
{

    m_currentTable = id;
    ui->openTickets->refreshTickets(m_currentTable);
    int size = ui->openTickets->getTickets().size();
    if (size == 0) {
        if (!QRKGastro::isHotelRoom(m_currentTable) && Reports().mustDoEOAny(QDateTime::currentDateTime()))
            QRKGastro::infoMessage();
        else
            emit newTicket(m_currentTable);
    }

    emit selectionChanged();
    return size;
}

void QRKGastroOpenTickets::newTicket(bool clicked)
{

    Q_UNUSED(clicked)
    if (!QRKGastro::isHotelRoom(m_currentTable) && Reports().mustDoEOAny(QDateTime::currentDateTime())) {
        QRKGastro::infoMessage();
        return;
    }
    History history;
    history.historyInsertLine(
        tr("Bestellung"), tr("Bestellung neu Tisch %1").arg(QRKGastro::getTableName(m_currentTable)));

    emit newTicket(m_currentTable);
}

void QRKGastroOpenTickets::changeTicket(bool clicked)
{

    Q_UNUSED(clicked)

    if (!QRKGastro::isHotelRoom(m_currentTable) && Reports().mustDoEOAny(QDateTime::currentDateTime())) {
        QRKGastro::infoMessage();
        return;
    }

    int tableId = 0;
    int ticketId = 0;

    getSelectedTicket(ticketId, tableId);
    if (ticketId == 0) return;

    History history;
    history.historyInsertLine(tr("Bestellung"), tr("Bestellung ändern Tisch %1").arg(QRKGastro::getTableName(tableId)));

    emit changeTicket(tableId, ticketId);
}

void QRKGastroOpenTickets::voidTicket()
{
    if (!RBAC::Instance()->hasPermission("gastro_void_ticked", true)) {
        return;
    }

    int ticketId, tableId;
    getSelectedTicket(ticketId, tableId);
    if (ticketId == 0) return;

    // Bestätigungsdialog anzeigen
    QMessageBox::StandardButton reply = QMessageBox::question(
        this, tr("Bon löschen"), tr("Möchten Sie diesen Bon wirklich löschen?"), QMessageBox::Yes | QMessageBox::No,
        QMessageBox::No // Standardmäßig auf "Nein" setzen
    );

    // Wenn der Benutzer "Nein" auswählt, die Funktion beenden
    if (reply == QMessageBox::No) {
        return;
    }

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    // Starte eine Transaktion
    if (!dbc.transaction()) {
        // Fehler bei Transaktion
        return;
    }

    // Lösche die orderextras
    query.prepare("DELETE FROM orderextras WHERE orderId IN (SELECT id FROM ticketorders WHERE ticketId=:ticketId);");
    query.bindValue(":ticketId", ticketId);
    if (!query.exec()) {
        dbc.rollback();
        return;
    }

    // Lösche die ticketorders
    query.prepare("DELETE FROM ticketorders WHERE ticketId=:ticketId");
    query.bindValue(":ticketId", ticketId);
    if (!query.exec()) {
        dbc.rollback();
        return;
    }

    // Lösche das ticket
    query.prepare("DELETE FROM tickets WHERE id=:ticketId");
    query.bindValue(":ticketId", ticketId);
    if (!query.exec()) {
        dbc.rollback();
        return;
    }

    // Bestätige die Transaktion, wenn alle Löschoperationen erfolgreich waren
    if (!dbc.commit()) {
        // Fehler bei der Bestätigung der Transaktion
    }

    QString text = tr("Bestellung Tisch %1 (id:%2) storniert.").arg(QRKGastro::getTableName(tableId)).arg(ticketId);
    History history;
    history.historyInsertLine(tr("Bestellung"), text);
    QrkJournal journal;
    journal.journalInsertLine("STORNO", text);

    refresh();
}


void QRKGastroOpenTickets::getSelectedTicket(int &id, int &table)
{

    id = table = 0;

    QList<int> tickets = ui->openTickets->getTickets();
    QList<int> selected = ui->openTickets->getSelectedTickets();

    if (selected.count() > 1) {
        QMessageBox::information(this, tr("Mehrfache Selektion"), tr("Nur ein Bon darf selektiert sein"));
        return;
    }

    // if there is only 1 ticket, use this even if not selected
    if (tickets.count() == 1) {
        id = tickets[0];
        table = ui->openTickets->getTableOfTicket(id);
        return;
    }

    if (selected.count() == 1) {
        id = selected[0];
        table = ui->openTickets->getTableOfTicket(id);
        return;
    }
}

void QRKGastroOpenTickets::moveTicket()
{
    emit splitTicket(true);
}

void QRKGastroOpenTickets::splitTicket(bool moveticket)
{

    int ticket, table;
    getSelectedTicket(ticket, table);
    if (ticket == 0) return;

    QRKGastroSplitTicketWidget splitWidget(moveticket, this);
    splitWidget.show();

    if (splitWidget.exec(ticket, table) == QRKGastroSplitTicketWidget::SPLITTED) {
        ui->openTickets->refreshTickets(m_currentTable);

        // select the new ticket
        ui->openTickets->selectTicket(ticket);
        History history;
        if (moveticket)
            history.historyInsertLine(
                tr("Bestellung"), tr("Bestellung auf Tisch %1 verschoben").arg(QRKGastro::getTableName(table)));
        else
            history.historyInsertLine(
                tr("Bestellung"), tr("Bestellung separieren Tisch %1").arg(QRKGastro::getTableName(m_currentTable)));
    }
}

bool QRKGastroOpenTickets::finishTicket(int ticket)
{

    QRKGastroFinishTicket finish(m_paycash, this);
    bool retval = finish.createReceipt(ticket);

    m_paycash = false;
    if (!retval) return false;

    ui->openTickets->refreshTickets(m_currentTable);

    // if ( code == QRKGastroFinishTicketDialog::WITH_PRINTOUT )
    //  printInvoice(ticket);
    History history;
    history.historyInsertLine(
        tr("Bestellung"), tr("Bestellung bezahlen Tisch %1").arg(QRKGastro::getTableName(m_currentTable)));

    return true;
}

void QRKGastroOpenTickets::payTicket(int tickeId, bool paycash)
{
    m_paycash = paycash;
    ui->openTickets->selectTicket(tickeId);
    emit payTicket();
}

void QRKGastroOpenTickets::calculateTicket()
{
    QList<int> selected = ui->openTickets->getSelectedTickets();

    if (selected.count() == 0) return;

    QRKGastroFinishTicket(false).calculateReceipt(selected,
        QString("%1 - %2")
            .arg(QRKGastro::getRoomNameFromTableId(m_currentTable))
            .arg(QRKGastro::getTableName(m_currentTable)));
    refresh();
}

void QRKGastroOpenTickets::payTicket()
{
    QList<int> selected = ui->openTickets->getSelectedTickets();

    if (selected.count() == 0) return;

    if (selected.count() > 1) {
        payGroupTicket(selected);
        refresh();
        return;
    }

    QList<int> tickets = ui->openTickets->getTickets();
    if (tickets.count() == 1) // if there's only 1, use this
        finishTicket(tickets[0]);
    else if (selected.count() == 1) // if there are more, use only if 1 is selected
        finishTicket(selected[0]);

    refresh();
}

void QRKGastroOpenTickets::payGroupTicket(QList<int> tickets)
{
    if (tickets.count() <= 1) return;

    if (QMessageBox::question(this, tr("Bons zusammenlegen"),
            tr("Möchten sie diese %1 Bons wirklich zusammenlegen ?").arg(tickets.count()), QMessageBox::Yes,
            QMessageBox::No)
        == QMessageBox::No)
        return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    int firstId = tickets[0];
    for (int i = 1; i < tickets.count(); i++) {
        query.prepare("UPDATE ticketorders set ticketId=:firstId WHERE ticketId=:ticketId");
        query.bindValue(":firstId", firstId);
        query.bindValue(":ticketId", tickets[i]);
        query.exec();
        query.prepare("DELETE FROM tickets where id=:ticketId");
        query.bindValue(":ticketId", tickets[i]);
        query.exec();
    }

    // update for merged ticket
    ui->openTickets->refreshTickets(m_currentTable);

    // select the merged one
    ui->openTickets->selectTicket(firstId);

    finishTicket(firstId);
}

void QRKGastroOpenTickets::refresh()
{
    QrkSettings settings;
    bool autoleave = settings.value("Gastro/autoleavetable", false).toBool();
    ui->openTickets->refreshTickets(m_currentTable);
    int size = ui->openTickets->getTickets().size();
    if (size == 0) {
        if (!Reports().mustDoEOAny(QDateTime::currentDateTime()) && !autoleave) {
            emit newTicket(m_currentTable);
            return;
        } else
            emit leaveTicket();
    }
    emit selectionChanged();

    //    emit leaveTicket();
}
