/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "qrkgastroquickproduct.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "database.h"

#include "ui_qrkgastroquickproduct.h"

#include <QCompleter>
#include <QDebug>
#include <QJsonArray>
#include <QJsonObject>
#include <QJsonValue>

QrkGastroQuickProduct::QrkGastroQuickProduct(QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::QrkGastroQuickProduct)
{
    ui->setupUi(this);
    ui->infoLabel->setHidden(true);

    setTaxes();
    setCompleter();

    QRegExp rx("^?(?:0|[1-9][0-9]*)[,.]?[0-9][0-9]$");
    QValidator *doubleVal = new QRegExpValidator(rx, this);
    ui->grossLineEdit->setValidator(doubleVal);

    populateGroupComboBox();

    QJsonArray printers = Database::getPrinters();
    ui->printerComboBox->addItem(tr("kein Drucker"), 0);
    foreach (const QJsonValue &value, printers) {
        QJsonObject obj = value.toObject();
        ui->printerComboBox->addItem(obj["name"].toString(), obj["id"].toInt());
    }

    QVariant value;
    QString strValue;
    Database::select_globals("quickproductprinter", value, strValue);
    int i;
    for (i = 0; i <= ui->printerComboBox->count(); i++) {
        if (value.toInt() == i) break;
    }

    if (i > ui->printerComboBox->count()) i = 0;

    ui->printerComboBox->setCurrentIndex(i);
    m_printerId = i;

    connect(ui->cancelPushButton, &QPushButton::clicked, this, &QrkGastroQuickProduct::close);
    connect(ui->okPushButton, &QPushButton::clicked, this, &QrkGastroQuickProduct::accept);
    connect(ui->printerComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), this,
        &QrkGastroQuickProduct::printerChanged);
    connect(ui->productLineEdit, &QLineEdit::textChanged, this, &QrkGastroQuickProduct::checkProduct);
    connect(ui->groupComboBox, &QComboBox::currentTextChanged, this, &QrkGastroQuickProduct::checkProduct);

    connect(ui->grossLineEdit, &QLineEdit::textChanged, [=]() {
        QString input = ui->grossLineEdit->text();
        input.replace(",", ".");
        m_gross = input;
        m_gross.round(2);
    });
}

QrkGastroQuickProduct::~QrkGastroQuickProduct()
{
    delete ui;
}

void QrkGastroQuickProduct::setTaxes()
{
    QString taxlocation = Database::getTaxLocation();
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare(QString("SELECT tax FROM taxtypes WHERE taxlocation=:taxlocation ORDER BY id"));
    query.bindValue(":taxlocation", taxlocation);
    query.exec();
    while (query.next()) {
        QBCMath tax(query.value(0).toString());
        tax.round(2);
        ui->taxComboBox->addItem(tax.toString(), tax.toDouble());
    }
    ui->taxComboBox->setCurrentIndex(ui->taxComboBox->findText(Database::getDefaultTax()));
}

void QrkGastroQuickProduct::populateGroupComboBox()
{
    QSqlDatabase dbc = Database::database();

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id, name FROM groups WHERE id > 1");

    if (!query.exec()) {
        return;
    }

    while (query.next()) {
        int id = query.value("id").toInt();
        QString name = query.value("name").toString();
        ui->groupComboBox->addItem(name, id); // `name` als Text, `id` als Datenwert speichern
    }

    ui->groupComboBox->setCurrentIndex(ui->groupComboBox->findData(2));
}

void QrkGastroQuickProduct::setCompleter()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("select DISTINCT p2.name from (select max(version) as version, "
                  "origin from products group by origin) p1 inner join (select * "
                  "from products) as  p2 on p1.version=p2.version and "
                  "p1.origin=p2.origin where groupid=2 AND visible >= 0");
    query.exec();

    QStringList list;

    while (query.next()) {
        QString value = query.value("name").toString();
        list << value;
    }

    QCompleter *editorCompleter = new QCompleter(list);
    editorCompleter->setCaseSensitivity(Qt::CaseInsensitive);
    editorCompleter->setFilterMode(Qt::MatchContains);
    ui->productLineEdit->setCompleter(editorCompleter);
}

void QrkGastroQuickProduct::printerChanged(int id)
{
    m_printerId = id;
    Database::insert2globals("quickproductprinter", id, QVariant());
}

void QrkGastroQuickProduct::checkProduct()
{
    int id = Database::getProductIdByName(ui->productLineEdit->text(), ui->groupComboBox->currentData().toInt());
    if (id > 0) {
        ui->infoLabel->setHidden(false);
        QJsonObject itemdata = Database::getProductById(id, 0);
        m_printerId = Database::getPrinterIdFromProduct(id);
        qDebug() << "Function: " << Q_FUNC_INFO << " itemData " << itemdata;

        QBCMath tax(itemdata.value("tax").toDouble());
        tax.round(2);

        ui->infoLabel->setText(tr(
            "Artikel '%1' ist bereits in der Warengruppe '%2' vorhanden.\nDer bestehende Steuersatz wird von diesen Artikel übernommen.")
                .arg(itemdata["name"].toString())
                .arg(ui->groupComboBox->currentText()));

        qDebug() << "Function: " << Q_FUNC_INFO << " Tax: " << tax.toString()
                 << " index: " << ui->taxComboBox->findData(tax.toDouble());
        ui->taxComboBox->setCurrentIndex(ui->taxComboBox->findData(tax.toDouble()));
        ui->taxComboBox->setEnabled(false);
        ui->printerComboBox->setCurrentIndex(ui->printerComboBox->findData(m_printerId));

    } else {
        ui->infoLabel->setHidden(true);
        ui->taxComboBox->setEnabled(true);
    }
}

void QrkGastroQuickProduct::accept()
{
    if (ui->productLineEdit->text().trimmed().isEmpty()) {
        ui->productLineEdit->setFocus();
        return;
    }

    if (ui->grossLineEdit->text().trimmed().isEmpty()) {
        ui->grossLineEdit->setFocus();
        return;
    }

    int groupId = ui->groupComboBox->currentData().toInt();
    QBCMath tax(ui->taxComboBox->currentText());
    tax.round(2);
    QBCMath net(m_gross / (1.0 + tax.toDouble() / 100));
    net.round(2);
    m_itemdata = QJsonObject();
    m_itemdata["name"] = ui->productLineEdit->text().trimmed();
    m_itemdata["itemnum"] = ""; // wurde wieder entfrnt -> D = Dummy Product No itemnumber Update
    m_itemdata["tax"] = tax.toDouble();
    m_itemdata["net"] = net.toDouble();
    m_itemdata["gross"] = m_gross.toDouble();
    m_itemdata["visible"] = 0;
    m_itemdata["group"] = groupId;

    m_productId = Database::getProductIdByName(m_itemdata["name"].toString(), groupId);
    m_itemdata["productId"] = m_productId;

    if (m_productId > 0) {
        // Database::updateProductTax(m_itemdata["tax"].toDouble(), m_productId);
        // Database::updateProductPrice(m_itemdata["gross"].toDouble(), m_itemdata["tax"].toDouble(), m_productId);
        Database::updateProductPrinter(m_printerId, m_productId);
    } else if (Database::addProduct(m_itemdata, m_productId)) {
        Database::updateProductPrinter(m_printerId, m_productId);
    }
    m_productId = Database::getProductIdByName(m_itemdata["name"].toString(), groupId);
    m_itemdata["productId"] = m_productId;

    emit QDialog::accept();
}

QJsonObject QrkGastroQuickProduct::getProduct()
{

    return m_itemdata;
}
