/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "cashbookinoutdialog.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/rbac/crypto.h"
#include "3rdparty/profeatures/profeatures.h"
#include "cashbookdialog.h"
#include "database.h"
#include "qrkjournal.h"
#include "qrksettings.h"
#include "ui_cashbookinoutdialog.h"

#include <QMessageBox>
#include <QSqlDatabase>
#include <QSqlError>

#include <QDebug>

CashBookInOutDialog::CashBookInOutDialog(QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::CashBookInOutDialog)
{
    ui->setupUi(this);
    ui->dateTimeEdit->setDateTime(QDateTime::currentDateTime());

    ui->flowComboBox->addItem(tr("Einzahlung"), CB_FLOW::FLOW_IN);
    ui->flowComboBox->addItem(tr("Auszahlung"), CB_FLOW::FLOW_OUT);

    ui->typeComboBox->addItem(tr("Betrieblich"), CB_TYPE::TYPE_BUSINESS);
    ui->typeComboBox->addItem(tr("Privat"), CB_TYPE::TYPE_PRIVATE);

    ui->grossLabel->setText(tr("Brutto Betrag in %1").arg(Database::getCurrency()));

    QRegExp rx("^?(?:0|[1-9][0-9]*),?[0-9][0-9]$");
    QValidator *doubleVal = new QRegExpValidator(rx, this);
    ui->grossEdit->setValidator(doubleVal);

    connect(ui->savePushButton, &QrkPushButton::clicked, this, &CashBookInOutDialog::save);
    connect(ui->cancelPushButton, &QrkPushButton::clicked, this, &CashBookInOutDialog::close);
    connect(ui->grossEdit, &QLineEdit::textChanged, this, &CashBookInOutDialog::grossEditChanged);
    connect(ui->flowComboBox, &QComboBox::currentTextChanged, this, &CashBookInOutDialog::grossEditChanged);

    CSqlQuery query(Database::database("CASHBOOK"), Q_FUNC_INFO);
    query.prepare("SELECT timestamp FROM cashbook ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) ui->dateTimeEdit->setMinimumDateTime(query.value("timestamp").toDateTime());
    readSettings();
}

CashBookInOutDialog::~CashBookInOutDialog()
{
    writeSettings();
    delete ui;
}

void CashBookInOutDialog::setMinimumDateTime(QString datetimeIsoString)
{
    ui->dateTimeEdit->setMinimumDateTime(QDateTime::fromString(datetimeIsoString, Qt::ISODate));
}

void CashBookInOutDialog::grossEditChanged()
{
    int checkoutcontent = ProFeatures::getCheckoutContents(false).toInt();
    int gross = QLocale().toDouble(ui->grossEdit->text()) * 100;
    if (checkoutcontent - gross < 0 && ui->flowComboBox->currentData() == FLOW_OUT) {
        ui->infoLabel->setText(tr("Der eingegebene Betrag würde die Kasse in den Minusbereich bringen."));
        ui->savePushButton->setDisabled(true);
    } else {
        if (QLocale().toDouble(ui->grossEdit->text()) > 0.00)
            ui->savePushButton->setDisabled(false);
        else
            ui->savePushButton->setDisabled(true);
        ui->infoLabel->setText("");
    }
}

void CashBookInOutDialog::save()
{
    QSqlDatabase dbc = Database::database("CASHBOOK");
    if (!dbc.transaction()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "Could not start transaction:" << dbc.lastError().text();
        QMessageBox::critical(this, tr("Datenbankfehler"), tr("Konnte keine Datenbank-Transaktion starten."));
        return;
    }

    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(R"(
        INSERT INTO cashbook (timestamp, flow, type, gross, reference, checksum, description, userid)
        VALUES (:timestamp, :flow, :type, :gross, :reference, :checksum, :description, :userid)
    )");

    query.bindValue(":timestamp", ui->dateTimeEdit->dateTime().toString(Qt::ISODate));
    query.bindValue(":flow", ui->flowComboBox->currentData().toInt());
    query.bindValue(":type", ui->typeComboBox->currentData().toInt());

    int gross = int(QLocale().toDouble(ui->grossEdit->text()) * 100);
    if (ui->flowComboBox->currentData().toInt() == FLOW_OUT) gross *= -1;
    query.bindValue(":gross", gross);

    query.bindValue(":reference", ui->referenceLineEdit->text());
    query.bindValue(":description", ui->descriptionTextEdit->toPlainText());
    query.bindValue(":userid", RBAC::Instance()->getUserId());

    int checksumvalue = CashBookDialog::getLastCheckSum().toInt() + gross;
    query.bindValue(":checksum",
        Crypto::encrypt(SecureByteArray(QString::number(checksumvalue).toLatin1()), SecureByteArray("CashBook")));

    if (!query.exec()) {
        QMessageBox::critical(this, tr("Datenbankfehler"), tr("Fehler beim Einfügen des Eintrags ins Kassabuch."));
        dbc.rollback();
        return;
    }

    if (!dbc.commit()) {
        qWarning() << "Commit failed:" << dbc.lastError().text();
        QMessageBox::critical(this, tr("Datenbankfehler"), tr("Die Transaktion konnte nicht abgeschlossen werden."));
        dbc.rollback();
        return;
    }

    // Journal entries (not transactional, but logged)
    QrkJournal journal;
    journal.journalInsertLine(
        "Kassabuch", tr("Manueller %1 - %2").arg((gross > 0) ? tr("Eingang") : tr("Ausgang")).arg(gross));
    journal.journalInsertLine("Kassabuch", tr("Kassastand - %1").arg(ProFeatures::getCheckoutContents()));
    journal.journalInsertLine("Kassabuch",
        tr("Prüfsumme: %1")
            .arg(Crypto::encrypt(
                SecureByteArray(QString::number(checksumvalue).toLatin1()), SecureByteArray("CashBook"))));

    QDialog::accept();
}

void CashBookInOutDialog::readSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    restoreGeometry(settings.value("InOutDialogWindowGeometry").toByteArray());
    settings.endGroup();
}

void CashBookInOutDialog::writeSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    settings.save2Settings("InOutDialogWindowGeometry", saveGeometry());
    settings.endGroup();
}
