/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "couponprinter.h"
#include "qrkpushbutton.h"
#include "qrksettings.h"

#include <QCheckBox>
#include <QFileDialog>
#include <QFontDatabase>
#include <QGraphicsPixmapItem>
#include <QGraphicsScene>
#include <QGraphicsTextItem>
#include <QGraphicsView>
#include <QGridLayout>
#include <QGroupBox>
#include <QLabel>
#include <QPainter>
#include <QPrintDialog>
#include <QPrintPreviewDialog>
#include <QSpinBox>
#include <QVBoxLayout>
#include <QtPrintSupport/QPrinter>

CouponPrinter::CouponPrinter(const QString &voucherCode, const QString &infoText, QWidget *parent)
    : QDialog(parent)
    , m_voucherCode(voucherCode)
    , m_infoText(infoText)
{

    setWindowFlags(Qt::Dialog | Qt::CustomizeWindowHint | Qt::WindowTitleHint);
    setWindowTitle(tr("Gutschein - Druck"));
    int id = QFontDatabase::addApplicationFont(":font/free3of9.ttf");
    QFontDatabase::applicationFontFamilies(id).at(0);

    QVBoxLayout *mainLayout = new QVBoxLayout(this);
    QGraphicsView *graphicsView = new QGraphicsView(this);
    graphicsScene = new QGraphicsScene(this);
    graphicsView->setScene(graphicsScene);
    mainLayout->addWidget(graphicsView);

    showInfoTextCheckBox = new QCheckBox("Show Info Text", this);
    showInfoTextCheckBox->setChecked(true);
    mainLayout->addWidget(showInfoTextCheckBox);

    extendCheckBox = new QCheckBox("Extend Settings", this);
    mainLayout->addWidget(extendCheckBox);

    settingsGroup = new QGroupBox("Settings");
    QGridLayout *gridLayout = new QGridLayout(settingsGroup);
    int row = 0;
    addSetting(tr("Barcode Größe:"), &barcodeSizeSpinBox, gridLayout, row);
    // New settings for text and code font sizes
    addSetting(tr("Text Font Größe:"), &textSizeSpinBox, gridLayout, row, 2);
    addSetting(tr("Code Font Größe:"), &codeSizeSpinBox, gridLayout, row++, 4);

    addSettingPair("Code X Position:", "Code Y Position:", &xSpinBox, &ySpinBox, gridLayout, row++);
    addSettingPair("Barcode X Position:", "Barcode Y Position:", &barcodeXSpinBox, &barcodeYSpinBox, gridLayout, row++);
    addSettingPair("Text X Position:", "Text Y Position:", &textXSpinBox, &textYSpinBox, gridLayout, row++);
    addSettingPair(
        "Template X Position:", "Template Y Position:", &templateXSpinBox, &templateYSpinBox, gridLayout, row++, true);

    settingsGroup->setLayout(gridLayout);
    mainLayout->addWidget(settingsGroup);

    connect(extendCheckBox, &QCheckBox::stateChanged, this, &CouponPrinter::toggleExtendedSettings);
    connect(barcodeSizeSpinBox, static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged), this,
        &CouponPrinter::updateBarcodeSize);

    QList<QSpinBox *> spinBoxesToUpdate = { xSpinBox, ySpinBox, barcodeXSpinBox, barcodeYSpinBox, textXSpinBox,
        textYSpinBox, textSizeSpinBox, codeSizeSpinBox };
    for (auto spinBox : spinBoxesToUpdate) {
        connect(spinBox, static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged), this,
            &CouponPrinter::updatePreview);
    }

    connect(showInfoTextCheckBox, &QCheckBox::stateChanged, this, &CouponPrinter::toggleInfoText);

    toggleExtendedSettings();


    QHBoxLayout *printButtonsLayout = new QHBoxLayout();
    addButtonToLayout(
        tr("Druck Vorschau"), ":/src/icons/print.png", printButtonsLayout, &CouponPrinter::showPrintPreview);
    addButtonToLayout(tr("Drucken"), ":/src/icons/ok-print.png", printButtonsLayout, &CouponPrinter::directPrint);
    mainLayout->addLayout(printButtonsLayout);

    QHBoxLayout *templateButtonsLayout = new QHBoxLayout();
    addButtonToLayout(
        tr("Default Template"), ":/src/icons/restore.png", templateButtonsLayout, &CouponPrinter::loadDefaultTemplate);
    addButtonToLayout(
        tr("Template laden"), ":/src/icons/new.png", templateButtonsLayout, &CouponPrinter::selectTemplate);
    addButtonToLayout(tr("Beenden"), ":/src/icons/exit.png", templateButtonsLayout, &CouponPrinter::quit);

    mainLayout->addLayout(templateButtonsLayout);

    setLayout(mainLayout);

    QString defaultTemplateImagePath = ":coupon/images/gutscheinvorlage.png";
    if (QFile::exists(defaultTemplateImagePath)) {
        templateImage = QImage(defaultTemplateImagePath);
        templateImagePath = defaultTemplateImagePath;
    } else {
        templateImage = createDefaultTemplate();
    }
    templatePixmapItem = new QGraphicsPixmapItem(QPixmap::fromImage(templateImage));
    graphicsScene->addItem(templatePixmapItem);
    codeTextItem = new QGraphicsTextItem(m_voucherCode);
    QFont font = codeTextItem->font();
    font.setPointSize(20);
    codeTextItem->setFont(font);
    codeTextItem->setDefaultTextColor(Qt::black);
    graphicsScene->addItem(codeTextItem);

    textItem = new QGraphicsTextItem(m_infoText);
    font = textItem->font();
    font.setPointSize(20);
    textItem->setFont(font);
    textItem->setDefaultTextColor(Qt::black);
    graphicsScene->addItem(textItem);

    barcodeImage = generateBarcodeImage("*" + m_voucherCode + "*", barcodeSizeSpinBox->value() + 1);
    barcodePixmapItem = new QGraphicsPixmapItem(QPixmap::fromImage(barcodeImage));
    graphicsScene->addItem(barcodePixmapItem);

    connect(textSizeSpinBox, static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged), [this](int size) {
        updateFontSize(size, textItem);
    });
    connect(codeSizeSpinBox, static_cast<void (QSpinBox::*)(int)>(&QSpinBox::valueChanged), [this](int size) {
        updateFontSize(size, codeTextItem);
    });


    loadSettings();
    adjustWindowSize(templateImage.size());
    updatePreview();
}

CouponPrinter::~CouponPrinter()
{
    saveSettings();
}

void CouponPrinter::toggleInfoText()
{
    bool showInfoText = showInfoTextCheckBox->isChecked();
    textItem->setVisible(showInfoText);
    updatePreview();
}

void CouponPrinter::updatePreview()
{
    templatePixmapItem->setPos(templateXSpinBox->value(), templateYSpinBox->value());
    codeTextItem->setPos(xSpinBox->value(), ySpinBox->value());
    barcodePixmapItem->setPos(barcodeXSpinBox->value(), barcodeYSpinBox->value());
    textItem->setPos(textXSpinBox->value(), textYSpinBox->value());
}

void CouponPrinter::updateBarcodeSize(int size)
{
    barcodeImage = generateBarcodeImage("*" + m_voucherCode + "*", size);
    barcodePixmapItem->setPixmap(QPixmap::fromImage(barcodeImage));
    updatePreview();
}

void CouponPrinter::updateFontSize(int size, QGraphicsTextItem *textItem)
{
    QFont font = textItem->font();
    font.setPointSize(size);
    textItem->setFont(font);
    updatePreview();
}

void CouponPrinter::showPrintPreview()
{
    QPrinter printer;
    QPrintPreviewDialog previewDialog(&printer, this);
    connect(&previewDialog, &QPrintPreviewDialog::paintRequested, this, &CouponPrinter::printVoucher);
    previewDialog.exec();
}

void CouponPrinter::directPrint()
{
    QPrinter printer;
    QPrintDialog printDialog(&printer, this);
    if (printDialog.exec() == QDialog::Accepted) {
        printVoucher(&printer);
    }
}

void CouponPrinter::printVoucher(QPrinter *printer)
{
    QPainter painter(printer);
    painter.drawImage(QPoint(0, 0), templateImage);
    painter.drawImage(QPoint(barcodeXSpinBox->value(), barcodeYSpinBox->value()), barcodeImage);

    QFont font = painter.font();
    font.setPointSize(codeSizeSpinBox->value()); // Code font size
    painter.setFont(font);
    painter.drawText(
        QRectF(xSpinBox->value(), ySpinBox->value(), templateImage.width(), templateImage.height()), m_voucherCode);

    font.setPointSize(textSizeSpinBox->value()); // Text font size
    painter.setFont(font);
    if (showInfoTextCheckBox->isChecked()) {

        painter.drawText(
            QRectF(textXSpinBox->value(), textYSpinBox->value(), templateImage.width(), templateImage.height()),
            m_infoText);
    }
}

void CouponPrinter::toggleExtendedSettings()
{
    bool showExtended = extendCheckBox->isChecked();
    settingsGroup->setVisible(showExtended);
    adjustWindowSize(templateImage.size());
}

void CouponPrinter::selectTemplate()
{
    QString selectedFilePath
        = QFileDialog::getOpenFileName(this, tr("Template Auswahl"), "", "Images (*.png *.jpg *.bmp)");
    if (!selectedFilePath.isEmpty()) {
        templateImage = QImage(selectedFilePath);
        templateImagePath = selectedFilePath;
        templatePixmapItem->setPixmap(QPixmap::fromImage(templateImage));
        adjustWindowSize(templateImage.size());
        updatePreview();
    }
}

void CouponPrinter::loadDefaultTemplate()
{
    templateImage = QImage(":coupon/images/gutscheinvorlage.png");
    templateImagePath = ":coupon/images/gutscheinvorlage.png";
    templatePixmapItem->setPixmap(QPixmap::fromImage(templateImage));
    adjustWindowSize(templateImage.size());
    updatePreview();
}

void CouponPrinter::addSetting(
    const QString &labelText, QSpinBox **spinBox, QGridLayout *layout, int row, int column, bool hidden)
{
    QLabel *label = new QLabel(labelText, this);
    *spinBox = new QSpinBox(this);
    (*spinBox)->setRange(-1000, 1000);

    // Add widgets to layout
    layout->addWidget(label, row, column);
    layout->addWidget(*spinBox, row, column + 1);

    // Set visibility
    label->setVisible(!hidden);
    (*spinBox)->setVisible(!hidden);
}

void CouponPrinter::addSettingPair(const QString &label1, const QString &label2, QSpinBox **spinBox1,
    QSpinBox **spinBox2, QGridLayout *layout, int row, bool hidden)
{
    QLabel *labelX = new QLabel(label1, this);
    *spinBox1 = new QSpinBox(this);
    (*spinBox1)->setRange(0, 1000);
    layout->addWidget(labelX, row, 0);
    layout->addWidget(*spinBox1, row, 1);

    QLabel *labelY = new QLabel(label2, this);
    *spinBox2 = new QSpinBox(this);
    (*spinBox2)->setRange(0, 1000);
    layout->addWidget(labelY, row, 2);
    layout->addWidget(*spinBox2, row, 3);

    labelX->setVisible(!hidden);
    (*spinBox1)->setVisible(!hidden);
    labelY->setVisible(!hidden);
    (*spinBox2)->setVisible(!hidden);
}

void CouponPrinter::addButtonToLayout(
    const QString &buttonText, const QString &iconPath, QHBoxLayout *layout, void (CouponPrinter::*function)())
{
    QrkPushButton *button = new QrkPushButton(buttonText, this);
    button->setIcon(QIcon(iconPath));
    button->setIconSize(QSize(32, 32));
    connect(button, &QrkPushButton::clicked, this, function);
    layout->addWidget(button);
}

void CouponPrinter::adjustWindowSize(const QSize &size)
{
    setMinimumSize(size + QSize(100, 100));
}

QImage CouponPrinter::createDefaultTemplate()
{
    QImage defaultImage(500, 300, QImage::Format_ARGB32);
    defaultImage.fill(Qt::white);
    QPainter painter(&defaultImage);
    painter.setPen(Qt::black);
    painter.drawRect(0, 0, 499, 299);
    painter.setFont(QFont("Arial", 30));
    painter.drawText(defaultImage.rect(), Qt::AlignCenter, "Voucher Template");
    return defaultImage;
}

QImage CouponPrinter::generateBarcodeImage(const QString &barcodeText, int size)
{
    // Use 3-of-9 TrueType font to render barcode
    QFont barcodeFont("free 3 of 9");
    barcodeFont.setPointSize(size); // Adjust font size based on user selection

    QFontMetrics barcodeFontMetrics(barcodeFont);
    QSize barcodeSize = barcodeFontMetrics.size(0, barcodeText);
    int barcodeWidth = barcodeSize.width() + 5;
    int barcodeHeight = barcodeSize.height() + 5;
    QImage barcodeImage(barcodeWidth, barcodeHeight, QImage::Format_ARGB32);
    barcodeImage.fill(Qt::white);
    QPainter painter(&barcodeImage);
    painter.setPen(Qt::black);

    painter.setFont(barcodeFont);
    painter.drawText(barcodeImage.rect(), Qt::AlignCenter, barcodeText);

    return barcodeImage;
}

void CouponPrinter::loadSettings()
{
    QrkSettings settings;
    settings.beginGroup("Coupon");

    xSpinBox->setValue(settings.value("codeXPos", 130).toInt());
    ySpinBox->setValue(settings.value("codeYPos", 170).toInt());
    barcodeXSpinBox->setValue(settings.value("barcodeXPos", 100).toInt());
    barcodeYSpinBox->setValue(settings.value("barcodeYPos", 220).toInt());
    templateXSpinBox->setValue(settings.value("templateXPos", 0).toInt());
    templateYSpinBox->setValue(settings.value("templateYPos", 0).toInt());
    textXSpinBox->setValue(settings.value("textXPos", 65).toInt());
    textYSpinBox->setValue(settings.value("textYPos", 135).toInt());
    barcodeSizeSpinBox->setValue(settings.value("barcodeSize", 35).toInt());
    textSizeSpinBox->setValue(settings.value("textSize", 15).toInt());
    codeSizeSpinBox->setValue(settings.value("codeSize", 20).toInt());

    extendCheckBox->setChecked(settings.value("extendedSettingsVisible", false).toBool());
    toggleExtendedSettings();

    QString savedTemplatePath = settings.value("templatePath", "").toString();
    if (!savedTemplatePath.isEmpty()) {
        templateImage = QImage(savedTemplatePath);
        templateImagePath = savedTemplatePath;
        templatePixmapItem->setPixmap(QPixmap::fromImage(templateImage));
        adjustWindowSize(templateImage.size());
    }

    restoreGeometry(settings.value("geometry").toByteArray());
    settings.endGroup();
}

void CouponPrinter::saveSettings()
{
    QrkSettings settings;
    settings.beginGroup("Coupon");

    settings.save2Settings("codeXPos", xSpinBox->value());
    settings.save2Settings("codeYPos", ySpinBox->value());
    settings.save2Settings("barcodeXPos", barcodeXSpinBox->value());
    settings.save2Settings("barcodeYPos", barcodeYSpinBox->value());
    settings.save2Settings("templateXPos", templateXSpinBox->value());
    settings.save2Settings("templateYPos", templateYSpinBox->value());
    settings.save2Settings("textXPos", textXSpinBox->value());
    settings.save2Settings("textYPos", textYSpinBox->value());
    settings.save2Settings("barcodeSize", barcodeSizeSpinBox->value());
    settings.save2Settings("textSize", textSizeSpinBox->value());
    settings.save2Settings("codeSize", codeSizeSpinBox->value());
    settings.save2Settings("extendedSettingsVisible", extendCheckBox->isChecked());

    settings.save2Settings("templatePath", templateImagePath);
    settings.save2Settings("geometry", saveGeometry());
    settings.endGroup();
}

void CouponPrinter::quit()
{
    QDialog::close();
}
