/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "wsdlbackupsettings.h"
#include "../qrkcore/backup.h"
#include "3rdparty/ckvsoft/preferences/settings.h"
#include "3rdparty/profeatures/profeatures.h"
#include "database.h"

#include <QDate>
#include <QFileSystemModel>
#include <QGroupBox>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QPushButton>
#include <QSortFilterProxyModel>
#include <QTreeView>

WsdlBackupSettings::WsdlBackupSettings(QWidget *parent)
    : QWidget(parent)
{

    Settings settings;
    m_path = settings.value("backupDirectory", "./").toString();
    QDir dir(m_path);
    dir.setFilter(QDir::Files | QDir::NoDotAndDotDot);

    QStringList filters;
    filters << "*.zip";

    m_view = new QTreeView();
    m_model = new QFileSystemModel();

    m_model->setRootPath(dir.path());
    m_model->index(dir.path());
    m_model->setFilter(QDir::Files | QDir::NoDotAndDotDot);
    m_model->setNameFilters(filters);
    m_model->setNameFilterDisables(false);
    m_view->setSelectionMode(QAbstractItemView::SingleSelection);
    m_view->setSelectionBehavior(QAbstractItemView::SelectRows);
    m_view->setModel(m_model);
    m_view->setRootIndex(m_model->index(dir.path()));
    m_view->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);

    m_viewGroup = new QGroupBox();
    m_viewGroup->setCheckable(true);
    if (!ProFeatures::isValid()) settings.save2Settings("WsdlBackup/backup_enabled", false);

    m_viewGroup->setChecked(settings.value("WsdlBackup/backup_enabled", false).toBool());

    m_viewGroup->setTitle(tr("Wdsl Backup verwenden"));

    QVBoxLayout *viewLayout = new QVBoxLayout();
    QHBoxLayout *buttonLayout = new QHBoxLayout();

    QPushButton *pushButton = new QPushButton;
    pushButton->setMinimumHeight(60);
    pushButton->setMinimumWidth(150);

    QIcon icon1 = QIcon(":/src/icons/cancel.png");
    QSize size = QSize(32, 32);
    pushButton->setIcon(icon1);
    pushButton->setIconSize(size);
    pushButton->setText(tr("Beenden"));

    m_button = new QPushButton(tr("Wiederherstellen"));
    m_button->setEnabled(false);
    m_button->setMinimumHeight(60);
    m_button->setMinimumWidth(150);
    QIcon icon2 = QIcon(":/src/icons/restore.png");
    m_button->setIcon(icon2);
    m_button->setIconSize(size);

    m_deleteButton = new QPushButton(tr("Löschen"));
    m_deleteButton->setEnabled(false);
    m_deleteButton->setMinimumHeight(60);
    m_deleteButton->setMinimumWidth(150);
    QIcon icon3 = QIcon(":/src/icons/eraser.png");
    m_deleteButton->setIcon(icon3);
    m_deleteButton->setIconSize(size);

    QSpacerItem *spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);

    buttonLayout->addItem(spacer);
    buttonLayout->addWidget(m_deleteButton);
    buttonLayout->addWidget(m_button);
    buttonLayout->addWidget(pushButton);
    viewLayout->addWidget(m_view, 1);
    //    viewLayout->addLayout(buttonLayout);
    viewLayout->setStretchFactor(m_view, 1);
    m_viewGroup->setLayout(viewLayout);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(m_viewGroup);
    mainLayout->addLayout(buttonLayout);
    mainLayout->setStretch(0, 1);
    mainLayout->addSpacing(12);
    setLayout(mainLayout);

    connect(m_button, &QPushButton::clicked, this, &WsdlBackupSettings::buttonClicked);
    connect(m_deleteButton, &QPushButton::clicked, this, &WsdlBackupSettings::deleteButtonClicked);
    connect(m_view, &QTreeView::clicked, this, &WsdlBackupSettings::buttonEnabled);
    connect(pushButton, &QPushButton::clicked, this, &WsdlBackupSettings::accept);
    connect(m_viewGroup, &QGroupBox::clicked, this, &WsdlBackupSettings::viewGroupChanged);
}

void WsdlBackupSettings::buttonEnabled(QModelIndex idx)
{
    if (m_current.isValid() && m_current.row() == idx.row()) {
        m_button->setEnabled(!m_button->isEnabled());
        m_deleteButton->setEnabled(!m_deleteButton->isEnabled());

        if (!m_button->isEnabled()) {
            m_view->clearSelection();
            m_viewGroup->setFocus();
        }
    } else {
        m_button->setEnabled(true);
        m_deleteButton->setEnabled(true);
        m_current = idx;
    }
}

void WsdlBackupSettings::buttonClicked(bool)
{
    QModelIndex idx = m_view->currentIndex();
    QString filename = idx.model()->data(idx).toString();
    if (!filename.isEmpty()) {
        Backup::restore(filename, true, true);
    }
}

void WsdlBackupSettings::deleteButtonClicked(bool)
{
    QModelIndex idx = m_view->currentIndex();
    QString filename = idx.model()->data(idx).toString();
    if (!filename.isEmpty()) QFile::remove(QString("%1/%2").arg(m_path).arg(filename));

    m_view->repaint();
    m_button->setEnabled(false);
    m_deleteButton->setEnabled(false);

    m_view->setRootIndex(m_model->index(m_path));
    m_view->header()->setSectionResizeMode(0, QHeaderView::ResizeToContents);
}

void WsdlBackupSettings::viewGroupChanged(bool checked)
{
    Settings settings;
    settings.save2Settings("WsdlBackup/backup_enabled", checked);
    m_button->setEnabled(false);
    m_deleteButton->setEnabled(false);
}
