/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "a_signsmartcard.h"
#include <QDebug>

/*
        RK Suite defined in Detailspezifikation/ABS 2
        suite for a closed system (closed systems are identified by the ZDA-ID
   AT0) R1_AT0("1", "AT0", "ES256", "SHA-256", 8), suite for an open system (in
   this case with the virtual ZDA identified by AT100) R1_AT100("1", "AT100",
   "ES256", "SHA-256", 8);
*/

/**
 * @brief ASignSmartCard::ASignSmartCard
 * @param device_name
 */
ASignSmartCard::ASignSmartCard(QString device_name, bool shared)
    : RKSignatureSmartCard(device_name, shared)
{
    setPin("123456");
}

/**
 * @brief ASignSmartCard::~ASignSmartCard
 */
ASignSmartCard::~ASignSmartCard()
{
}

/**
 * @brief ASignSmartCard::transmit
 * @param txBuffer
 * @param txLength
 * @return
 */
ASignResponse ASignSmartCard::transmit(const unsigned char *txBuffer, DWORD txLength)
{
    ASignResponse result;
    memset(&result, 0, sizeof(ASignResponse));

    if (RKSignatureSmartCard::transmit(txBuffer, txLength, result.data, &result.length)) {
        result.code[0] = result.data[result.length - 2];
        result.code[1] = result.data[result.length - 1];
        result.length -= 2;
        if (result.code[0] != 0x90 || result.code[1] != '\0') {
            qCritical() << "Function Name: " << Q_FUNC_INFO << getSmartCardMessage(result.code);
        }
    }

    return result;
}


QByteArray ASignSmartCard::ReadFile()
{
    QByteArray ba = 0;

    unsigned char cmd[6];
    memcpy(cmd, ASIGN_CMD_READ_BINARY, sizeof(ASIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 5);

    DWORD offset = response.length;
    for (DWORD i = 0; i < response.length; i++) {
        ba.append(response.data[i]);
    }
    while (256 == response.length) {
        /*
         * 2.1.6.1 Explicit selection
         * For explicit selection the exact offset is given.
         * OH Offset High Byte
         * OL Offset Low Byte
         *
         * 2.1.6.2 Implicit selection
         * When using implicit selection, the data of the EF with the given short
         * identier (SID) is read. The Short Identier always refers to an EF in the
         * DF just selected before. Therefore, 3 bits of OH are used to indicate the
         * implicit selection B8 B7 B6 B5 B4 B3 B2 B1 1  0  0  x  x  x  x  x
         *          XXXXX is the Short File Identier. The Lower Offset OL is always
         * set to 0x00.
         *
         * 2.1.6.3 Le
         * Le (length expected), 0x00 all digits until the le end but max. 256 bytes
         */

        cmd[2] = (offset >> 8) & 0xFF;
        cmd[3] = offset & 0xFF;
        response = transmit(cmd, 5);
        for (DWORD i = 0; i < response.length; i++) {
            ba.append(response.data[i]);
        }
        offset += response.length;
    }

    return ba;
}

bool ASignSmartCard::verifyPinCmd(bool &blocked, const unsigned char cmd[14])
{
    if (m_pin.isEmpty()) {
        qWarning() << Q_FUNC_INFO << "PIN-Verifizierung fehlgeschlagen: Kein PIN";
        return false;
    }

    ASignResponse resp = transmit(cmd, 13);
    unsigned short sw = (resp.code[0] << 8) | resp.code[1];

    QString message = getSmartCardMessage(resp.code);

    if (sw == 0x9000) {
        return true; // PIN korrekt
    }

    m_pin.clear();

    if (sw == 0x6983) {
        blocked = true;
        qWarning() << Q_FUNC_INFO << "PIN blockiert!";
        return false;
    }

    qWarning() << Q_FUNC_INFO << "PIN-Verifizierung fehlgeschlagen:" << message;
    return false;
}
