/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "primesignsmartcard.h"
#include <QDebug>

/*
        RK Suite defined in Detailspezifikation/ABS 2
        suite for a closed system (closed systems are identified by the ZDA-ID
   AT0) R1_AT0("1", "AT0", "ES256", "SHA-256", 8), suite for an open system (in
   this case with the virtual ZDA identified by AT100) R1_AT100("1", "AT100",
   "ES256", "SHA-256", 8);
*/

PrimeSignSmartCard::PrimeSignSmartCard(QString device_name, bool shared)
    : RKSignatureSmartCard(device_name, shared)
{
}

PrimeSignSmartCard::~PrimeSignSmartCard()
{
}

/**
 * @brief PrimeSignSmartCard::transmit
 * @param txBuffer
 * @param txLength
 * @return
 */
ASignResponse PrimeSignSmartCard::transmit(const unsigned char *txBuffer, DWORD txLength)
{
    ASignResponse result;
    memset(&result, 0, sizeof(ASignResponse));

    if (RKSignatureSmartCard::transmit(txBuffer, txLength, result.data, &result.length)) {
        result.code[0] = result.data[result.length - 2];
        result.code[1] = result.data[result.length - 1];
        result.length -= 2;
        if (result.code[0] != 0x90 || result.code[1] != '\0') {
            qCritical() << "Function Name: " << Q_FUNC_INFO << getSmartCardMessage(result.code);
        }
    }

    return result;
}

QByteArray PrimeSignSmartCard::ReadFile()
{
    QByteArray ba = 0;

    unsigned char cmd[8];
    memcpy(cmd, PRIMESIGN_CMD_READ_BINARY, sizeof(PRIMESIGN_CMD_READ_BINARY));
    ASignResponse response = transmit(cmd, 7);

    DWORD offset = response.length;
    for (DWORD i = 0; i < response.length; i++) {
        ba.append(response.data[i]);
    }
    while (256 == response.length) {
        cmd[2] = (offset >> 8) & 0xFF;
        cmd[3] = offset & 0xFF;
        response = transmit(cmd, 7);
        for (DWORD i = 0; i < response.length; i++) {
            ba.append(response.data[i]);
        }
        offset += response.length;
    }

    return ba;
}

QString PrimeSignSmartCard::hexToDecimalString(const QByteArray &hex)
{
    QString result = "0";
    for (uchar byte : hex) {
        int carry = byte;
        QString temp;

        // Multipliziere vorhandene Dezimalzahl mit 256 und addiere neues Byte
        for (int i = result.length() - 1; i >= 0; --i) {
            int digit = result[i].digitValue();
            int value = digit * 256 + carry;

            carry = value / 10;
            temp.prepend(QChar('0' + (value % 10)));
        }
        while (carry > 0) {
            temp.prepend(QChar('0' + (carry % 10)));
            carry /= 10;
        }
        result = temp;
    }
    return result;
}

bool PrimeSignSmartCard::verifyPinCmd(bool &blocked, const unsigned char cmd[14])
{
    if (m_pin.isEmpty()) {
        qWarning() << Q_FUNC_INFO << "PIN-Verifizierung fehlgeschlagen: Kein PIN";
        return false;
    }

    ASignResponse resp = transmit(cmd, 13);
    unsigned short sw = (resp.code[0] << 8) | resp.code[1];

    QString message = getSmartCardMessage(resp.code);

    if (sw == 0x9000) {
        return true; // PIN korrekt
    }

    m_pin.clear();

    if (sw == 0x6983) {
        blocked = true;
        qWarning() << Q_FUNC_INFO << "PIN blockiert!";
        return false;
    }

    qWarning() << Q_FUNC_INFO << "PIN-Verifizierung fehlgeschlagen:" << message;
    return false;
}
