/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "documentprinter.h"
#include "3rdparty/ckvsoft/ckvtemplate.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#ifdef QRK_PROFEATURES
#    include "3rdparty/profeatures/profeatures.h"
#endif
#include "RK/rk_signaturemodule.h"
#include "database.h"
#include "defines.h"
#include "qrkprinter.h"
#include "qrksettings.h"
#include "reports.h"
#include "singleton/spreadsignal.h"
#include "utils/demomode.h"
#include "utils/qrcode.h"
#include "utils/utils.h"

#include <QAbstractTextDocumentLayout>
#include <QApplication>
#include <QDateTime>
#include <QDebug>
#include <QDir>
#include <QFileInfo>
#include <QJsonArray>
#include <QJsonObject>
#include <QPainter>
#include <QPrinter>
#include <QPrinterInfo>
#include <QStandardPaths>
#include <QTextDocument>
#include <cmath>

DocumentPrinter::DocumentPrinter(QObject *parent)
    : QObject(parent)
{

    QrkSettings settings;
    QList<QString> printerFontList = settings.value("printerfont", "Courier-New,10,100").toString().split(",");
    QList<QString> receiptPrinterFontList
        = settings.value("receiptprinterfont", "Courier-New,8,100").toString().split(",");
    QList<QString> taggedPrinterFontList
        = settings.value("taggedprinterfont", "Courier-New,8,100").toString().split(",");


    m_noPrinter = settings.value("noPrinter", false).toBool();

    if (settings.value("Printer/receiptPrinter").toInt() == 0) m_noPrinter = true;

    m_pdfPrinterPath
        = settings.value("pdfDirectory", QStandardPaths::writableLocation(QStandardPaths::AppDataLocation)).toString();
    if (!QDir().exists(m_pdfPrinterPath)) {
        m_pdfPrinterPath = QStandardPaths::writableLocation(QStandardPaths::AppDataLocation);
        settings.save2Settings("pdfDirectory", m_pdfPrinterPath);
    }

    m_printerFont = QFont(printerFontList.at(0));
    m_printerFont.setPointSize(printerFontList.at(1).toInt());
    m_printerFont.setStretch(printerFontList.at(2).toInt());
    m_printCollectionsReceipt = settings.value("printCollectionReceipt", false).toBool();
    m_collectionsReceiptText = settings.value("collectionReceiptText", tr("Abholbon für")).toString();
    //    m_collectionsReceiptCopies = settings.value("collectionReceiptCopies",
    //    1).toInt();

    m_defaultPaperWidth = settings.value("paperWidth", 80).toInt();
    m_receiptPrinterFont = QFont(receiptPrinterFontList.at(0));
    m_receiptPrinterFont.setPointSize(receiptPrinterFontList.at(1).toInt());
    m_receiptPrinterFont.setStretch(receiptPrinterFontList.at(2).toInt());

    m_taggedPrinterFont = QFont(taggedPrinterFontList.at(0));
    m_taggedPrinterFont.setPointSize(taggedPrinterFontList.at(1).toInt());
    m_taggedPrinterFont.setStretch(taggedPrinterFontList.at(2).toInt());


    m_printCompanyNameBold = settings.value("printCompanyNameBold", false).toBool();
    m_printQRCode = settings.value("qrcode", true).toBool();
    m_logoFileName = "";
    if (settings.value("useLogo", false).toBool()) m_logoFileName = settings.value("logo", "logo.png").toString();

    m_advertisingFileName = "";
    if (settings.value("useAdvertising", false).toBool())
        m_advertisingFileName = settings.value("advertising", "advertising.png").toString();

    m_logoRight = settings.value("logoRight", false).toBool();
    m_numberCopies = settings.value("numberCopies", 1).toInt();
    m_paperFormat = settings.value("paperFormat", "A4").toString();
    m_currency = Database::getCurrency();

    m_feedProdukt = settings.value("feedProdukt", 5).toInt();
    m_feedCompanyHeader = settings.value("feedCompanyHeader", 5).toInt();
    m_feedCompanyAddress = settings.value("feedCompanyAddress", 5).toInt();
    m_feedCashRegisterid = settings.value("feedCashRegisterid", 5).toInt();
    m_feedTimestamp = settings.value("feedTimestamp", 5).toInt();
    m_feedTax = settings.value("feedTaxSpin", 5).toInt();
    m_feedPrintHeader = settings.value("feedPrintHeader", 5).toInt();
    m_feedHeaderText = settings.value("feedHeaderText", 5).toInt();
    m_feedQRCode = settings.value("feedQRCode", 20).toInt();

    m_printQrCodeLeft = settings.value("qrcodeleft", false).toBool();
    m_useDecimalQuantity = settings.value("useDecimalQuantity", false).toBool();
    m_countDigits = settings.value("decimalDigits", 2).toInt();

    if (settings.value("noPrinter", false).toBool()) {
        m_useReportPrinter = false;
    } else {
        m_useReportPrinter = settings.value("useReportPrinter", false).toBool();
        if (m_numberCopies > 1 && m_useReportPrinter) {
            m_numberCopies = 1;
        } else {
            m_useReportPrinter = false;
        }
    }
}

DocumentPrinter::~DocumentPrinter()
{
    Spread::Instance()->setProgressBarValue(-1);
}

//--------------------------------------------------------------------------------

void DocumentPrinter::printTestDocument(QFont font)
{
    QTextDocument testDoc;
    testDoc.setHtml(Reports::getReport(2, false, true));
    testDoc.setDefaultFont(font);
    printDocument(&testDoc, "TEST DRUCK");
}

void DocumentPrinter::printDocument(QPrinter *printer, QTextDocument *document)
{
    m_smallPrinter = (printer->pageLayout().pageSize().size(QPageSize::Millimeter).width() <= 60) ? true : false;

    document->setDefaultFont(m_printerFont);
    document->adjustSize();

    if (printer->pageLayout().pageSize().id() == QPageSize::Custom) {
        document->setPageSize(printer->pageLayout().pageSize().size(QPageSize::Millimeter));
        const QRect pageRect = QRect(0, 0, printer->width(),
            printer->height()); // printer.pageRect();

        // Give the document the correct left-to-right width for word wrapping etc
        document->setPageSize(pageRect.size());

        // The total extent of the content (there are no page margin in this)
        QRect contentRect = QRect(QPoint(0, 0), document->size().toSize());

        // This is the part of the content we will drop on a page.  It's a sliding
        // window on the content.
        QRect currentRect(0, 0, pageRect.width(), pageRect.height());

        QPainter painter(printer);
        QFont font(m_receiptPrinterFont);
        font.setPointSize(8);

        document->setDefaultFont(font);
        document->documentLayout()->setPaintDevice(painter.device());

        while (currentRect.intersects(contentRect)) {
            painter.save();
            painter.translate(0, -currentRect.y());
            document->drawContents(&painter,
                currentRect); // draws part of the document
            painter.restore();

            // Translate the current rectangle to the area to be printed for the
            // next page
            currentRect.translate(0, currentRect.height());

            // Inserting a new page if there is still area left to be printed
            if (currentRect.intersects(contentRect)) printer->newPage();
        }
        painter.end();
    } else {
        document->print(printer);
    }
}

void DocumentPrinter::printDocument(QTextDocument *document, const QString &title)
{

    Spread::Instance()->setProgressBarWait(true);

    QrkSettings settings;
    QRKPrinter printer(settings.value("Printer/reportPrinter").toInt());
    QList<QPrinter *> printerList = printer.getPrinterList();
    int counter = 0;
    while (!printerList.isEmpty()) {
        QPrinter *printer = printerList.takeFirst();
        if (m_noPrinter || printer->outputFormat() == QPrinter::PdfFormat) {
            printer->setOutputFormat(QPrinter::PdfFormat);
            QDir dir(m_pdfPrinterPath);
            if (!dir.exists()) {
                dir.mkpath(".");
            }
            QString confname = qApp->property("configuration").toString();
            if (!confname.isEmpty()) confname = "_" + confname;
            if (counter == 0)
                printer->setOutputFileName(QString(m_pdfPrinterPath + "/QRK%1-REPORT_%2.pdf").arg(confname).arg(title));
            else
                printer->setOutputFileName(
                    QString(m_pdfPrinterPath + "/QRK%1-REPORT_%2 (%3).pdf").arg(confname).arg(title).arg(counter));

            ++counter;
        }

        printDocument(printer, document);

        // delete printer;
        // printer = Q_NULLPTR;
    }
    Spread::Instance()->setProgressBarWait(false);
}

void DocumentPrinter::printReceipt(QPrinter *printer, QJsonObject data)
{
    printI(data, printer);
}

void DocumentPrinter::printReceipt(QJsonObject data)
{

    QrkSettings settings;

    m_receiptNum = data.value("receiptNum").toInt();
    Spread::Instance()->setProgressBarValue(1);

    bool onlyOneCopy = false;

    int printerId = settings.value("Printer/receiptPrinter").toInt();
    if (data.value("isInvoiceCompany").toBool()) {
        onlyOneCopy = true;
        printerId = settings.value("Printer/invoiceCompanyPrinter").toInt();
    }

    QRKPrinter printer(printerId);
    QList<QPrinter *> printerList = printer.getPrinterList();
    int counter = 0;
    m_interimCalculation = data["interimCalculation"].toBool();
    while (!printerList.isEmpty()) {
        QPrinter *printer = printerList.takeFirst();

        if (m_noPrinter || printer->outputFormat() == QPrinter::PdfFormat) {
            printer->setOutputFormat(QPrinter::PdfFormat);
            QDir dir(m_pdfPrinterPath);
            if (!dir.exists()) {
                dir.mkpath(".");
            }
            QString description = "BON";

            if (m_interimCalculation) description = tr("ZR");

            QString confname = qApp->property("configuration").toString();
            if (!confname.isEmpty()) confname = "_" + confname;
            if (counter == 0)
                printer->setOutputFileName(
                    QString(m_pdfPrinterPath + "/QRK%1-%2%3.pdf").arg(confname).arg(description).arg(m_receiptNum));
            else
                printer->setOutputFileName(QString(m_pdfPrinterPath + "/QRK%1-%2%3 (%4).pdf")
                                               .arg(confname)
                                               .arg(description)
                                               .arg(m_receiptNum)
                                               .arg(counter));

            ++counter;
        }

        m_smallPrinter = (printer->pageLayout().pageSize().size(QPageSize::Millimeter).width() <= 60) ? true : false;

        printI(data, printer);

        // delete printer;
        // printer = Q_NULLPTR;

        if (onlyOneCopy || data.value("isCopy").toBool()) break;
    }
}

void DocumentPrinter::printTagged(QJsonObject data)
{

    QString description = data.value("customerText").toString();
    if (description.isEmpty()) return;

    QrkSettings settings;
    bool isTestPrint = data.value("isTestPrint").toBool();
    data.remove("customerText");
    data.remove("isTestPrint");
    foreach (const QString &key, data.keys()) {
        int printerid = key.toInt();
        if (printerid == -1) continue;

        QRKPrinter printer(printerid);
        QList<QPrinter *> printerList = printer.getPrinterList();
        int counter = 0;
        while (!printerList.isEmpty()) {
            QPrinter *printer = printerList.takeFirst();
            if (m_noPrinter || printer->outputFormat() == QPrinter::PdfFormat) {
                printer->setOutputFormat(QPrinter::PdfFormat);
                QDir dir(m_pdfPrinterPath);
                if (!dir.exists()) {
                    dir.mkpath(".");
                }
                QString confname = qApp->property("configuration").toString();
                if (!confname.isEmpty()) confname = "_" + confname;
                if (isTestPrint) confname = "_TEST" + confname;
                if (counter == 0)
                    printer->setOutputFileName(
                        QString(m_pdfPrinterPath + "/QRK%1-%2P%3.pdf").arg(confname).arg(description).arg(printerid));
                else
                    printer->setOutputFileName(QString(m_pdfPrinterPath + "/QRK%1-%2P%3 (%4).pdf")
                                                   .arg(confname)
                                                   .arg(description)
                                                   .arg(printerid)
                                                   .arg(counter));

                ++counter;
            }
            m_smallPrinter
                = (printer->pageLayout().pageSize().size(QPageSize::Millimeter).width() <= 60) ? true : false;

            QFont font(m_taggedPrinterFont);
            QFont datefont(m_taggedPrinterFont);
            datefont.setPointSize(8);
            QFontMetrics dateMetr(datefont);

            QFont boldFont(m_taggedPrinterFont);
            boldFont.setBold(true);

            QFontMetrics boldMetr(boldFont);

            QFont numFont(m_taggedPrinterFont);
            numFont.setBold(true);
            numFont.setPointSize(24);

            QFontMetrics sumMetr(numFont);

            const int WIDTH = printer->pageLayout().paintRectPixels(printer->resolution()).size().width();
            int y = 0;

            QPainter painter(printer);
            painter.setFont(font);
            QPen pen(Qt::black);
            painter.setPen(pen);
            QFontMetrics fontMetr = painter.fontMetrics();

            /*
            if ( printer->outputFormat() != QPrinter::PdfFormat ) {
              painter.translate(WIDTH, printer->pageLayout().paintRectPixels(printer->resolution()).height());
              painter.rotate(180);
            }
            */

            y = 0;

            painter.save();
            painter.setFont(numFont);
            // CustomerText
            QString headerText = (isTestPrint) ? QString("TEST%1").arg(description) : description;
            headerText = Utils::wordWrap(headerText, WIDTH, numFont);
            int headerTextHeight = headerText.split(QRegExp("\n|\r\n|\r")).count() * sumMetr.height();
            painter.drawText(0, y, WIDTH, headerTextHeight + 4, Qt::AlignCenter, headerText);
            y += 10 + headerTextHeight;
            painter.restore();

            painter.save();
            painter.setFont(datefont);
            painter.drawText(0, y, WIDTH, dateMetr.height(), Qt::AlignCenter,
                tr("Datum: %1 Uhrzeit: %2")
                    .arg(QDateTime::currentDateTime().toString("dd.MM.yyyy"))
                    .arg(QDateTime::currentDateTime().toString("hh:mm:ss")));
            y += 10 + dateMetr.height();
            painter.restore();

            painter.setFont(boldFont);

            QJsonArray orders = data.value(key).toArray();
            foreach (const QJsonValue &order, orders) {
                QString product = order.toObject().value("product").toString();
                painter.drawText(5, y, 40, fontMetr.height(), Qt::AlignLeft,
                    QString::number(order.toObject().value("count").toInt()));

                product = Utils::wordWrap(product, WIDTH - 40, boldFont);
                int productHeight = product.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
                painter.drawText(40, y, WIDTH - 40, productHeight, Qt::AlignLeft, product);
                y += 5 + productHeight;
                if ((y + fontMetr.height()) > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                    printer->newPage();
                    y = 0;
                }

                painter.save();
                painter.setFont(font);

                QString orderDescription = order.toObject().value("description").toString();
                if (!orderDescription.isEmpty()) {
                    orderDescription = Utils::wordWrap("*) " + orderDescription, WIDTH - 40, boldFont);
                    int orderDescriptionHeight
                        = orderDescription.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
                    painter.drawText(40, y, WIDTH - 40, orderDescriptionHeight, Qt::AlignLeft, orderDescription);
                    y += 5 + orderDescriptionHeight;
                    if ((y + fontMetr.height())
                        > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                        printer->newPage();
                        y = 0;
                    }
                }
                foreach (const QJsonValue &extra, order.toObject().value("extra").toArray()) {
                    QString extratext = extra.toString();
                    extratext = Utils::wordWrap(extratext, WIDTH, boldFont);
                    int productHeight = extratext.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
                    painter.drawText(40, y, WIDTH - 40, productHeight, Qt::AlignLeft, extratext);
                    y += 5 + productHeight;
                    if ((y + fontMetr.height())
                        > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                        printer->newPage();
                        y = 0;
                    }
                }
                painter.restore();
            }
            painter.end();
            // delete printer;
            // printer = Q_NULLPTR;
        }
    }
}

void DocumentPrinter::printCollectionReceipt(QJsonObject data)
{
    QrkSettings settings;
    bool isTestPrint = data.value("isTestPrint").toBool();

    QRKPrinter printer(settings.value("Printer/collectionPrinter").toInt());
    QList<QPrinter *> printerList = printer.getPrinterList();
    int counter = 0;
    while (!printerList.isEmpty()) {
        QPrinter *printer = printerList.takeFirst();
        if (m_noPrinter || printer->outputFormat() == QPrinter::PdfFormat) {
            QString confname = qApp->property("configuration").toString();
            if (!confname.isEmpty()) confname = "_" + confname;
            if (isTestPrint) confname = "_TEST" + confname;
            if (counter == 0)
                printer->setOutputFileName(
                    QString(m_pdfPrinterPath + "/QRK%1-BON%2-ABHOLBON.pdf").arg(confname).arg(m_receiptNum));
            else
                printer->setOutputFileName(QString(m_pdfPrinterPath + "/QRK%1-BON%2-ABHOLBON (%3).pdf")
                                               .arg(confname)
                                               .arg(m_receiptNum)
                                               .arg(counter));

            ++counter;
            printer->setOutputFormat(QPrinter::PdfFormat);
        }
        m_smallPrinter = (printer->pageLayout().pageSize().size(QPageSize::Millimeter).width() <= 60) ? true : false;
        QFont font(m_taggedPrinterFont);
        QFont datefont(m_taggedPrinterFont);
        datefont.setPointSize(8);
        QFontMetrics dateMetr(datefont);

        QFont boldFont(m_taggedPrinterFont);
        boldFont.setBold(true);

        QFontMetrics boldMetr(boldFont);

        QFont numFont(m_taggedPrinterFont);
        numFont.setBold(true);
        numFont.setPointSize(24);

        QFontMetrics sumMetr(numFont);

        const int WIDTH = printer->pageLayout().paintRectPixels(printer->resolution()).size().width();
        int y = 0;

        QString shopName = (isTestPrint) ? "TESTDRUCK MUSTERFIRMA" : data.value("shopName").toString();
        QString shopMasterData = (isTestPrint) ? "" : data.value("shopMasterData").toString();
        QString receiptNumber = QString::number(data.value("receiptNum").toInt());
        QJsonArray Orders = data["Orders"].toArray();
        foreach (const QJsonValue &item, Orders) {
            const QJsonObject &order = item.toObject();
            QBCMath count = order.value("count").toDouble();
            count.round(m_countDigits);

            bool coupon = order.value("coupon").toString() == "1";
            if (!coupon) continue;

            QPainter painter(printer);
            painter.setFont(font);
            QPen pen(Qt::black);
            painter.setPen(pen);
            QFontMetrics fontMetr = painter.fontMetrics();

            y = 0;

            painter.save();
            painter.setFont(numFont);
            painter.drawText(0, y, WIDTH, sumMetr.height() + 4, Qt::AlignCenter, receiptNumber);
            y += 5 + sumMetr.height();
            painter.restore();

            if (m_printCompanyNameBold) {
                painter.save();
                painter.setFont(boldFont);
                painter.drawText(0, y, WIDTH, boldMetr.height() + 4, Qt::AlignCenter, shopName);
                y += 5;
                painter.restore();
            } else {
                painter.drawText(0, y, WIDTH, fontMetr.height() + 4, Qt::AlignCenter, shopName);
                y += 5;
            }

            int shopMasterDataHeight = shopMasterData.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(0, y, WIDTH, shopMasterDataHeight + 4, Qt::AlignCenter, shopMasterData);
            y += 15 + shopMasterDataHeight;

            painter.save();
            painter.setFont(datefont);
            painter.drawText(0, y, WIDTH, dateMetr.height(), Qt::AlignCenter,
                tr("Datum: %1 Uhrzeit: %2")
                    .arg(
                        QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("dd.MM.yyyy"))
                    .arg(
                        QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("hh:mm:ss")));
            y += 5 + dateMetr.height();
            painter.restore();

            painter.setFont(boldFont);

            // CustomerText
            if (!data.value("headerText").toString().isEmpty()) {
                y += 5;
                QString headerText = data.value("headerText").toString();
                headerText = Utils::wordWrap(headerText, WIDTH, font);
                int headerTextHeight = headerText.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
                painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignCenter, headerText);
                y += m_feedHeaderText + headerTextHeight + 4;
                painter.drawLine(0, y, WIDTH, y);
                y += 5;
            }

            y += 5 + boldMetr.height();
            painter.drawText(0, y, WIDTH, boldMetr.height(), Qt::AlignCenter, m_collectionsReceiptText);
            y += 5 + boldMetr.height() * 2;

            QString product = QString("%1 x %2").arg(count.toString()).arg(order.value("product").toString());
            product = Utils::wordWrap(product, WIDTH, boldFont);
            int productHeight = product.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
            painter.drawText(0, y + 10, WIDTH, productHeight, Qt::AlignCenter, product);
            painter.end();
        }
        // delete printer;
        // printer = Q_NULLPTR;
    }
}

void DocumentPrinter::printI(QJsonObject data, QPrinter *printer)
{

    int fontsize = m_receiptPrinterFont.pointSize();

    QPainter painter(printer);
    QFont font(m_receiptPrinterFont);
    ckvTemplate Template;
    Template.set("DATUM", QDate::currentDate().toString());
    Template.set("DISPLAYNAME", data.value("displayname").toString());
    Template.set("UHRZEIT", QTime::currentTime().toString());
    Template.set("BONNUMMER", QString::number(data.value("receiptNum").toInt()));
    Template.set("VERSION", data.value("version").toString());
    Template.set("GASTRO_RAUMNAME", data.value("roomname").toString());
    Template.set("GASTRO_TISCHNAME", data.value("tablename").toString());

    QString sum = QLocale().toString(data.value("sum").toDouble(), 'f', 2);

    if (data.value("isTestPrint").toBool()) {
        sum = "0,0";
    }

    Template.set("SUMME", sum);

    // font.setFixedPitch(true);
    painter.setFont(font);
    QFontMetrics fontMetr = painter.fontMetrics();

    QFont grossFont(font);
    // grossFont.setFixedPitch(true);
    QFontMetrics grossMetrics(grossFont, printer);

    // QFont boldFont("Courier-New", boldsize, QFont::Bold);  // for sum
    QFont boldFont(m_receiptPrinterFont);
    boldFont.setBold(true);
    boldFont.setPointSize(m_receiptPrinterFont.pointSize() + 2);

    QFontMetrics boldMetr(boldFont, printer);

    QPen pen(Qt::black);
    painter.setPen(pen);

    QJsonArray Orders = data["Orders"].toArray();
    int oc = Orders.count() + 100;

    const int WIDTH = printer->pageLayout().paintRectPixels(printer->resolution()).size().width();
    const double FACTOR = getFactor(WIDTH, printer);

    qDebug() << "Function Name: " << Q_FUNC_INFO << " WIDTH: " << WIDTH;
    qDebug() << "Function Name: " << Q_FUNC_INFO << " FACTOR: " << FACTOR;

    int y = 0;

    bool logo = false;
    QFileInfo checkFile(m_logoFileName);
    // check if file exists and if yes: Is it really a file and no directory?
    if (checkFile.exists() && checkFile.isFile()) {
        logo = true;
    }

    bool advertising = false;
    QFileInfo checkAdvertisingFile(m_advertisingFileName);
    // check if file exists and if yes: Is it really a file and no directory?
    if (checkAdvertisingFile.exists() && checkAdvertisingFile.isFile()) {
        advertising = true;
    }

    Spread::Instance()->setProgressBarValue(int((float(15) / float(oc)) * 100));

    bool isInvoiceCompany = data.value("isInvoiceCompany").toBool();

    QPixmap logoPixmap;
    QPixmap advertisingPixmap;

    QString shopName = data.value("shopName").toString();
    QString shopMasterData = data.value("shopMasterData").toString();

    if (!isInvoiceCompany) {
        if (logo) {

            logoPixmap.load(m_logoFileName);
            logoPixmap = logoPixmap.scaled(logoPixmap.size() * FACTOR, Qt::KeepAspectRatio);

            if (m_logoRight) {
                if (logoPixmap.width() > WIDTH / 2.50)
                    logoPixmap = logoPixmap.scaled(WIDTH / 2.50,
                        printer->pageLayout().paintRectPixels(printer->resolution()).height(), Qt::KeepAspectRatio);

                painter.drawPixmap(WIDTH - logoPixmap.width() - 1, y, logoPixmap);

                QRect rect;
                if (m_printCompanyNameBold) {
                    painter.save();
                    painter.setFont(boldFont);
                    rect = painter.boundingRect(
                        0, y, WIDTH - logoPixmap.width(), logoPixmap.height(), Qt::AlignLeft, shopName);
                    painter.drawText(0, y, rect.width(), rect.height(), Qt::AlignLeft, shopName);
                    y += 5;
                    painter.restore();
                } else {
                    rect = painter.boundingRect(
                        0, y, WIDTH - logoPixmap.width(), logoPixmap.height(), Qt::AlignLeft, shopName);
                    painter.drawText(0, y, rect.width(), rect.height(), Qt::AlignLeft, shopName);
                    y += 5;
                }

                rect = painter.boundingRect(
                    0, y, WIDTH - logoPixmap.width(), logoPixmap.height(), Qt::AlignLeft, shopMasterData);
                painter.drawText(0, y, rect.width(), rect.height(), Qt::AlignLeft, shopMasterData);

                y += 5 + qMax(rect.height(), logoPixmap.height()) + 4;
                painter.drawLine(0, y, WIDTH, y);
                y += 5;
            } else {

                if (logoPixmap.width() > WIDTH)
                    logoPixmap = logoPixmap.scaled(WIDTH,
                        printer->pageLayout().paintRectPixels(printer->resolution()).height(), Qt::KeepAspectRatio);
                painter.drawPixmap((WIDTH / 2) - (logoPixmap.width() / 2) - 1, y, logoPixmap);
                y += 5 + logoPixmap.height() + 4;

                if (m_printCompanyNameBold) {
                    painter.save();
                    painter.setFont(boldFont);
                    painter.drawText(0, y, WIDTH, boldMetr.height() + 4, Qt::AlignCenter, shopName);
                    y += 5;
                    painter.restore();

                } else {
                    painter.drawText(0, y, WIDTH, fontMetr.height() + 4, Qt::AlignCenter, shopName);
                    y += 5;
                }

                int shopMasterDataHeight = shopMasterData.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
                painter.drawText(0, y, WIDTH, shopMasterDataHeight + 4, Qt::AlignCenter, shopMasterData);
                y += 5 + shopMasterDataHeight;
            }
        } else {
            if (m_printCompanyNameBold) {
                painter.save();
                painter.setFont(boldFont);
                painter.drawText(0, y, WIDTH, boldMetr.height() + 4, Qt::AlignCenter, shopName);
                y += m_feedCompanyHeader;
                painter.restore();
            } else {
                painter.drawText(0, y, WIDTH, fontMetr.height() + 4, Qt::AlignCenter, shopName);
                y += m_feedCompanyHeader;
            }

            int shopMasterDataHeight = shopMasterData.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(0, y, WIDTH, shopMasterDataHeight + 4, Qt::AlignCenter, shopMasterData);
            y += m_feedCompanyAddress + shopMasterDataHeight;
        }
    }

    if (!data.value("printHeader").toString().isEmpty()) {
        QString printHeader = data.value("printHeader").toString();
        printHeader = Utils::wordWrap(Template.process(printHeader), WIDTH, font);
        int headerTextHeight = printHeader.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
        painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignCenter, printHeader);
        y += m_feedPrintHeader + headerTextHeight + 4;
        painter.drawLine(0, y, WIDTH, y);
    }

    // CustomerText
    if (!data.value("headerText").toString().isEmpty()) {
        y += 5;
        QString headerText = data.value("headerText").toString();
        headerText = Utils::wordWrap(headerText, WIDTH, font);
        int headerTextHeight = headerText.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
        painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignLeft, headerText);
        y += m_feedHeaderText + headerTextHeight + 4;
        painter.drawLine(0, y, WIDTH, y);
        y += 5;
    }

    Spread::Instance()->setProgressBarValue(int((float(20) / float(oc)) * 100));

    painter.save();
    painter.setFont(boldFont);

    // receiptPrinterHeading or cancellationtext by cancellation
    QString comment = data.value("comment").toString();
    if (!comment.isEmpty()) {
        comment = Utils::wordWrap(comment, WIDTH, font);
        int commentHeight = comment.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
        y += 5 + boldMetr.height();
        painter.drawText(0, y, WIDTH, commentHeight, Qt::AlignCenter, comment);
        y += commentHeight;
        y += 5 + boldMetr.height() * 2;
    }

    painter.restore();

    QString copy = "";
    if (data.value("isCopy").toBool()) copy = tr("( Kopie )");

    if (data.value("isTestPrint").toBool()) {
        data["kassa"] = "DEMO-PRINT-1";
        data["receiptNum"] = 0;
        data["typeText"] = "DEMO";
    }

    if (m_smallPrinter) {
        painter.drawText(
            0, y, WIDTH, fontMetr.height(), Qt::AlignLeft, tr("Kasse: %1").arg(data.value("kasse").toString()));
        y += m_feedCashRegisterid + fontMetr.height();

        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Bon-Nr: %1 %2 - %3")
                .arg(data.value("receiptNum").toInt())
                .arg(copy)
                .arg(data.value("typeText").toString()));
        y += m_feedCashRegisterid + fontMetr.height();

        painter.drawText(
            0, y, WIDTH, fontMetr.height(), Qt::AlignLeft, tr("Positionen: %1").arg(data.value("positions").toInt()));
        y += m_feedCashRegisterid + fontMetr.height();

        Spread::Instance()->setProgressBarValue(int((float(30) / float(oc)) * 100));

        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Datum: %1")
                .arg(QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("dd.MM.yyyy")));
        y += m_feedTimestamp + fontMetr.height();

        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Uhrzeit: %1")
                .arg(QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("hh:mm:ss")));
        y += m_feedTimestamp + fontMetr.height();

    } else {
        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Kasse: %1 Positionen: %2").arg(data.value("kasse").toString()).arg(data.value("positions").toInt()));
        // TODO: 1 or 2 Lines from config, in my case the text is to long. Do not
        // fit in one line
        y += m_feedCashRegisterid + fontMetr.height();

        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Bon-Nr: %1 %2 - %3")
                .arg(data.value("receiptNum").toInt())
                .arg(copy)
                .arg(data.value("typeText").toString()));
        y += m_feedCashRegisterid + fontMetr.height();

        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignLeft,
            tr("Datum: %1 Uhrzeit: %2")
                .arg(QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("dd.MM.yyyy"))
                .arg(QDateTime::fromString(data.value("receiptTime").toString(), Qt::ISODate).toString("hh:mm:ss")));
        y += m_feedTimestamp + fontMetr.height();
    }

    Spread::Instance()->setProgressBarValue(int((float(50) / float(oc)) * 100));

    painter.drawLine(0, y, WIDTH, y);
    y += 5;

    // paint orders

    QString countText = tr("Anz");

    // get maximum counter size
    int counter_size = 0;
    foreach (const QJsonValue &item, Orders) {
        const QJsonObject &order = item.toObject();
        QBCMath count;
        if (m_useDecimalQuantity) {
            count = order.value("count").toDouble();
            count.round(m_countDigits);
        } else {
            count = order.value("count").toInt();
        }

        counter_size = qMax(counter_size, count.toString().size());
    }

    int bountingWidth = qMax(fontMetr.boundingRect(countText + " ").width(),
        fontMetr.boundingRect(QString("0").repeated(counter_size) + " ").width());
    const int X_COUNT = 0;
    const int X_NAME = bountingWidth > 25 ? bountingWidth : 25;

    painter.drawText(X_COUNT, y, WIDTH, fontMetr.height(), Qt::AlignLeft, countText);
    painter.drawText(X_NAME, y, WIDTH - X_COUNT, fontMetr.height(), Qt::AlignLeft, tr("Artikel"));
    painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, tr("Preis  M%"));
    y += 5 + fontMetr.height();

    Spread::Instance()->setProgressBarValue(int((float(60) / float(oc)) * 100));

    int progress = 60;

    foreach (const QJsonValue &item, Orders) {

        Spread::Instance()->setProgressBarValue(int((float(progress++) / float(oc)) * 100));

        const QJsonObject &order = item.toObject();

        QBCMath count;
        if (m_useDecimalQuantity) {
            count = order.value("count").toDouble();
            count.round(m_countDigits);
        } else {
            count = order.value("count").toInt();
        }

        QString taxPercent;

        if (Database::getTaxLocation() == "CH")
            taxPercent = QString("%1").arg(QString::number(order.value("tax").toDouble(), 'f', 2));
        else
            taxPercent = QString("%1").arg(QString::number(order.value("tax").toInt()));

        if (taxPercent == "0") taxPercent = "00";

        QBCMath discount = order.value("discount").toDouble();
        discount.round(2);

        QBCMath gross = order.value("gross").toDouble();
        gross.round(2);

        QBCMath singleprice = order.value("singleprice").toDouble();
        singleprice.round(2);

        QString grossText = gross.toLocale();
        QString singleGrossText;
        if (discount > 0)
            singleGrossText = QString("%1 x %2 Rabatt:\u00A0-%3%")
                                  .arg(count.toLocale())
                                  .arg(singleprice.toLocale())
                                  .arg(discount.toLocale());
        else
            singleGrossText = QString("%1 x %2").arg(count.toLocale()).arg(singleprice.toLocale());

        if (data.value("isTestPrint").toBool()) {
            count = 0.0;
            grossText = "0,0";
            singleGrossText = "0 x 0,0";
        }

        int grossWidth = grossMetrics.boundingRect(grossText + "   " + taxPercent).width();
        QString product = order.value("product").toString();
        product = Utils::wordWrap(product, WIDTH - grossWidth - X_NAME, font);
        int productHeight = product.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();

        // check if new drawText is heigher than page height
        if ((y + productHeight + 20) > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
            printer->newPage();
            y = 0;
        }

        QRect usedRect;
        QString countString;
        if (m_useDecimalQuantity)
            countString = count.toLocale();
        else
            countString = QLocale().toString(count.toInt());

        painter.drawText(X_COUNT, y, WIDTH - X_COUNT, fontMetr.height(), Qt::AlignLeft, countString);
        painter.drawText(X_NAME, y, WIDTH, productHeight, Qt::AlignLeft, product, &usedRect);

        if (m_useDecimalQuantity || discount.toDouble() > 0 || count.toDouble() > 1 || count.toDouble() < -1) {
            y += m_feedProdukt + usedRect.height();
            singleGrossText = Utils::wordWrap(singleGrossText, WIDTH - grossWidth - X_NAME, font);
            int singleGrossTextHeight = singleGrossText.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(X_NAME, y, WIDTH - X_NAME - grossWidth - 5, singleGrossTextHeight, Qt::AlignLeft,
                singleGrossText, &usedRect);
            y += usedRect.height() - fontMetr.height();
        } else {
            y += usedRect.height() - fontMetr.height();
        }

        painter.setFont(grossFont);
        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, grossText + "   " + taxPercent);
        painter.setFont(font);

        QString productDescription = order.value("description").toString();
        if (!productDescription.isEmpty()) {
            y += fontMetr.height();
            productDescription = Utils::wordWrap(productDescription, WIDTH - grossWidth - X_NAME, font);
            int productDescriptionHeight = productDescription.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(X_NAME, y, WIDTH, productDescriptionHeight, Qt::AlignLeft, productDescription, &usedRect);
            y += usedRect.height() - fontMetr.height();
        }

        foreach (const QJsonValue &extra, order.value("extra").toArray()) {
            QString prefix = extra.toObject().value("type").toString() == "0" ? "- " : "+ ";
            QString product = prefix + extra.toObject().value("product").toString();
            product = Utils::wordWrap(product, WIDTH, boldFont);
            int productHeight = product.split(QRegExp("\n|\r\n|\r")).count() * boldMetr.height();
            y += productHeight;
            if ((y + fontMetr.height()) > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                printer->newPage();
                y = 0;
            }
            painter.drawText(X_NAME, y, WIDTH, productHeight, Qt::AlignLeft, product, &usedRect);
        }


        y += m_feedProdukt + fontMetr.height();
    }

    Spread::Instance()->setProgressBarValue(int((float(progress + 10) / float(oc)) * 100));

    painter.drawLine(0, y, WIDTH, y);
    y += fontMetr.height() / 2;

    // if there is not enough space for sum+tax lines, start new page
    if ((y + (data.value("taxesCount").toInt() * (5 + fontMetr.height())) + boldMetr.height() + 10)
        > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
        drawVoidLines(painter, y, WIDTH);
        printer->newPage();
        y = 0;
    }

    Spread::Instance()->setProgressBarValue(int((float(progress += 10) / float(oc)) * 100));

    int ySave = y; // save y when QR-Code was printing left
    QString sumText = tr("Gesamt: %1").arg(sum);
    painter.save();
    painter.setFont(boldFont);

    painter.drawText(0, y, WIDTH, boldMetr.height(), Qt::AlignRight, sumText);
    painter.restore();

    y += 5 + boldMetr.height();

    QMap<int, double> givenMap = Database::getGiven(data.value("receiptNum").toInt());

    if (givenMap.size() > 0) {
        QBCMath given(givenMap.take(PAYED_BY_CASH));
        if (given == 0.0) given = givenMap.take(PAYED_BY_COUPON);
        given.round(2);
        QBCMath secondPay(givenMap.value(PAYED_BY_DEBITCARD));
        if (secondPay == 0) secondPay = givenMap.value(PAYED_BY_CREDITCARD);
        if (secondPay == 0) secondPay = givenMap.value(PAYED_BY_COUPON);
        secondPay.round(2);

        QBCMath retourMoney((given + secondPay) - QLocale().toDouble(sum));
        retourMoney.round(2);
        QString givenText = tr("Gegeben: %1").arg(given.toLocale());
        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, givenText);
        y += m_feedTax + fontMetr.height();

        if (retourMoney > 0.0) {
            QString retourMoneyText = tr("Rückgeld: %1").arg(retourMoney.toLocale());
            painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, retourMoneyText);
            y += m_feedTax + fontMetr.height();
        }
        if (secondPay > 0.0) {
            QString paymentMethod;
            switch (givenMap.lastKey()) {
                case PAYED_BY_DEBITCARD:
                    paymentMethod = tr("Bankomat");
                    break;
                case PAYED_BY_CREDITCARD:
                    paymentMethod = tr("Kreditkarte");
                    break;
                case PAYED_BY_CASH:
                    paymentMethod = tr("Barzahlung");
                    break;
                case PAYED_BY_COUPON:
                    paymentMethod = tr("Gutschein");
                    break;
                default:
                    paymentMethod = tr("Unbekannt");
            }
            QString secondText = tr("%1: %2").arg(paymentMethod).arg(secondPay.toLocale());
            painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, secondText);
            y += m_feedTax + fontMetr.height();
        }
    }

    QJsonArray Taxes = data["Taxes"].toArray();
    foreach (const QJsonValue &item, Taxes) {
        const QJsonObject &tax = item.toObject();

        //        QString taxSum =
        //        QString::number(tax.value("t2").toString().toDouble(), 'f',
        //        2).replace(".",",");
        QBCMath taxSum = tax.value("t2").toString();
        taxSum.round(2);
        if (data.value("isTestPrint").toBool()) {
            taxSum = "0,0";
        }

        QString taxValue = tax.value("t1").toString();
        if (taxValue != "0%") {
            painter.drawText(
                0, y, WIDTH, fontMetr.height(), Qt::AlignRight, tr("MwSt %1: %2").arg(taxValue).arg(taxSum.toLocale()));

            y += m_feedTax + fontMetr.height();
        }
    }
    y += 5 + fontMetr.height();

    // Die Währung müsste sonst neben jeden Preis stehen, darum schreiben wir
    // diesen InfoText
    QString currencyText = tr("(Alle Beträge in %1)").arg(m_currency);
    if (m_printQRCode && m_printQrCodeLeft) {
        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignRight, currencyText);
        y += 20;
    } else {
        painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignCenter, currencyText);
    }

    y += 5 + fontMetr.height();

    Spread::Instance()->setProgressBarValue(int((float(progress += 10) / float(oc)) * 100));

    if (!m_interimCalculation) {
        QString qr_code_rep = shopName + " - " + shopMasterData;
        QString ocr_code_rep = shopName + " - " + shopMasterData;

        if (RKSignatureModule::isDEPactive()) {
            QString signature = Utils::getReceiptSignature(data.value("receiptNum").toInt(), true);
            if (signature.split('.').size() == 3) {
                qr_code_rep = signature.split('.').at(1);
                qr_code_rep = RKSignatureModule::base64Url_decode(qr_code_rep);
                ocr_code_rep = qr_code_rep;
                qr_code_rep
                    = qr_code_rep + "_" + RKSignatureModule::base64Url_decode(signature.split('.').at(2)).toBase64();
                ocr_code_rep = ocr_code_rep + "_"
                    + RKSignatureModule::base32_encode(RKSignatureModule::base64Url_decode(signature.split('.').at(2)));
                if (signature.split('.').at(2)
                    == RKSignatureModule::base64Url_encode("Sicherheitseinrichtung ausgefallen"))
                    data["isSEEDamaged"] = true;
                qDebug() << "Function Name: " << Q_FUNC_INFO << " QRCode Representation: " << qr_code_rep;
            } else {
                qInfo() << "Function Name: " << Q_FUNC_INFO
                        << " Print old (before DEP-7) Receipt Id:" << data.value("receiptNum").toInt();
            }
        }

        if (m_printQRCode && m_printQrCodeLeft) {
            QRCode qr;
            QPixmap QR = qr.encodeTextToPixmap(qr_code_rep);
            QR = QR.scaled(QR.size() * FACTOR, Qt::KeepAspectRatio);

            int sumWidth = boldMetr.boundingRect(sumText).width();

            if (QR.width() > (WIDTH - sumWidth))
                QR = QR.scaled(WIDTH - sumWidth - 4,
                    printer->pageLayout().paintRectPixels(printer->resolution()).height(), Qt::KeepAspectRatio);

            painter.drawPixmap(1, ySave, QR);

            y = m_feedQRCode + qMax(ySave + QR.height(), y);

            if (data.value("isSEEDamaged").toBool()) {
                y += 5 + fontMetr.height();
                painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignCenter, "Sicherheitseinrichtung ausgefallen");
                y += 5 + fontMetr.height();
            }
            painter.drawPoint(0, y);
        }

        if (isInvoiceCompany) {
            y += 5;
            QString printFooter = tr("Dient als Steuerbeleg für ihr Finanzamt.\n"
                                     "Wichtig: Beleg bitte aufbewahren.\n"
                                     "Diese Rechnung ist nur in Verbindung\n"
                                     "mit dem angehefteten Kassenbon gültig.\n"
                                     ">> Datum = Liefer- und Rechnungsdatum <<\n");

            int headerTextHeight = printFooter.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignCenter, printFooter);
            y += 5 + headerTextHeight + 4;

        } else if (!data.value("printFooter").toString().isEmpty()) {
            y += 5;
            QString printFooter = data.value("printFooter").toString();
            printFooter = Utils::wordWrap(Template.process(printFooter), WIDTH, font);
            int headerTextHeight = printFooter.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignCenter, printFooter);
            y += 5 + headerTextHeight + 4;
        }

        progress += 10;
        Spread::Instance()->setProgressBarValue(int((float(progress) / float(oc)) * 100));

        if (m_printQRCode && !m_printQrCodeLeft) {
            QRCode qr;
            QPixmap QR = qr.encodeTextToPixmap(qr_code_rep);
            QR = QR.scaled(QR.size() * FACTOR, Qt::KeepAspectRatio);

            if (QR.width() > WIDTH) {
                QR = QR.scaled(
                    WIDTH, printer->pageLayout().paintRectPixels(printer->resolution()).height(), Qt::KeepAspectRatio);
            }

            y += 5;
            painter.drawLine(0, y, WIDTH, y);
            y += 5;

            // check if new drawText is heigher than page height
            if ((y + QR.height() + 20) > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                drawVoidLines(painter, y, WIDTH);
                printer->newPage();
                y = 0;
            }
            painter.drawPixmap((WIDTH / 2) - (QR.width() / 2) - 1, y, QR);

            y += QR.height() + m_feedQRCode;

        } else if (!m_printQRCode && Database::getTaxLocation() == "AT") {
            y += 5;
            painter.drawLine(0, y, WIDTH, y);
            y += 5;

            int id = QFontDatabase::addApplicationFont(":src/font/ocra.ttf");
            QString family = QFontDatabase::applicationFontFamilies(id).at(0);
            QFont ocrfont(family, fontsize);
            painter.save();
            painter.setFont(ocrfont);

            QFontMetrics ocrMetr = painter.fontMetrics();

            ocr_code_rep = Utils::wordWrap(ocr_code_rep, WIDTH, ocrfont);
            int ocrHeight = ocr_code_rep.split(QRegExp("\n|\r\n|\r")).count() * ocrMetr.height();

            // check if new drawText is heigher than page height
            if ((y + ocrHeight + m_feedQRCode)
                > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                drawVoidLines(painter, y, WIDTH);
                printer->newPage();
                y = 0;
            }

            painter.drawText(0, y, WIDTH, ocrHeight, Qt::AlignLeft, ocr_code_rep);
            y += ocrHeight + m_feedQRCode;
            painter.restore();
        }

        if (!m_printQrCodeLeft) {
            if (data.value("isSEEDamaged").toBool()) {
                y += 5 + fontMetr.height();
                painter.drawText(0, y, WIDTH, fontMetr.height(), Qt::AlignCenter, "Sicherheitseinrichtung ausgefallen");
                y += 5 + fontMetr.height();
            }
            painter.drawPoint(0, y);
        }

        if (advertising) {
            y += 5;
            if ((y + advertisingPixmap.height() + 20)
                > printer->pageLayout().paintRectPixels(printer->resolution()).height()) {
                printer->newPage();
                y = 0;
            }

            advertisingPixmap.load(m_advertisingFileName);
            advertisingPixmap = advertisingPixmap.scaled(advertisingPixmap.size() * FACTOR, Qt::KeepAspectRatio);

            if (advertisingPixmap.width() > WIDTH)
                advertisingPixmap = advertisingPixmap.scaled(
                    WIDTH, printer->pageLayout().paintRectPixels(printer->resolution()).height(), Qt::KeepAspectRatio);
            painter.drawPixmap((WIDTH / 2) - (advertisingPixmap.width() / 2) - 1, y, advertisingPixmap);
            y += 5 + advertisingPixmap.height() + 4;
        }

        if (!data.value("printAdvertisingText").toString().isEmpty()) {
            QString printAdvertisingText = data.value("printAdvertisingText").toString();
            printAdvertisingText = Utils::wordWrap(Template.process(printAdvertisingText), WIDTH, font);
            int headerTextHeight = printAdvertisingText.split(QRegExp("\n|\r\n|\r")).count() * fontMetr.height();
            painter.drawText(0, y, WIDTH, headerTextHeight, Qt::AlignCenter, printAdvertisingText);
            y += headerTextHeight;
        }
    } // end m_interimCalculation

    // workaround marginBottom
    int marginBottom = int(printer->pageLayout().margins().bottom());
    if (marginBottom > 0) {
        y += marginBottom;
        painter.drawPoint(0, y);
    }

    if (!createdVersionInfo(data.value("version").toString()).isEmpty()) {
        font.setPixelSize(6);
        painter.setFont(font);
        painter.drawText(
            0, y, WIDTH, fontMetr.height(), Qt::AlignCenter, createdVersionInfo(data.value("version").toString()));
        y += 5 + fontMetr.height();
    }

    drawVoidLines(painter, y, WIDTH);

    painter.end();

    Spread::Instance()->setProgressBarValue(100);

    if (m_printCollectionsReceipt && !m_interimCalculation) printCollectionReceipt(data);
}

double DocumentPrinter::getFactor(int pixel, QPrinter *printer)
{
    const double p = 3.779527559055;
    int rect = int(printer->pageRect(QPrinter::Millimeter).width());
    int ratio = int(pixel / p);
    return static_cast<double>(ratio) / rect;
}

void DocumentPrinter::drawVoidLines(QPainter &painter, int y, int width)
{
    if (!DemoMode::isDemoMode()) return;

    QPen savedPen = painter.pen();
    QFont savedFont = painter.font();

    QFont demoFont = savedFont;
    demoFont.setPointSize(50); // Setze die Schriftgröße entsprechend an
    painter.setFont(demoFont);

    painter.setPen(QPen(savedPen.color(), 3));

    const QString demoText = "DEMO";
    QFontMetrics fm(demoFont);

#if QT_VERSION >= 0x050B00 // Für Qt 5.11 und neuer
    int textWidth = fm.horizontalAdvance(demoText);
#else
    int textWidth = fm.width(demoText);
#endif

    double angle = std::atan2(y, width) * 180 / M_PI;

    painter.drawLine(0, y, width, 0);

    int xPos = 0;
    int yPos = y;
    while (xPos < width && yPos > 0) {
        painter.save();
        painter.translate(xPos, yPos);
        painter.rotate(-angle);
        painter.drawText(0, 0, demoText);
        painter.restore();

        xPos += textWidth + 50;
        yPos -= (textWidth + 50) * std::tan(angle * M_PI / 180);
    }

    painter.setPen(savedPen);
    painter.setFont(savedFont);
}

/*
void DocumentPrinter::drawVoidLines(QPainter &painter, int y, int width)
{
    if (!DemoMode::isDemoMode()) return;
    QPen savedPen = painter.pen();
    painter.setPen(QPen(savedPen.color(), 3));
    painter.drawLine(0, y, width, 0);
    painter.setPen(savedPen);
}
*/

QString DocumentPrinter::createdVersionInfo(QString version)
{
#ifdef QRK_PROFEATURES
    if (ProFeatures::isValid()) return {};
#endif
    return QString("Erstellt mit QRK %1 - www.ckvsoft.at").arg(version);
}
