/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "qrkgastro.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/drag/dragpushbutton.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/verification.h"
#include "database.h"
#include "preferences/registrationtab.h"
#include "qrkgastrocurfewchecker.h"
#include "qrkgastroopentickets.h"
#include "qrkgastroselector.h"
#include "qrkgastrotableorder.h"
#include "qrkjournal.h"
#include "qrktimedmessagebox.h"
#include "reports.h"
#include "ui_qrkgastro.h"

#include <QDateTime>
#include <QDebug>
#include <QMessageBox>
#include <QSqlDatabase>
#include <QSqlError>
#include <QThread>
#include <QTreeWidget>

QRKGastro::QRKGastro(QWidget *parent)
    : QWidget(parent)
    , ui(new Ui::QRKGastro)
{
    ui->setupUi(this);

    m_selector = new QRKGastroSelector(ui->stackedGastroWidget);
    ui->stackedGastroWidget->addWidget(m_selector);

    m_order = new QRKGastroTableOrder(ui->stackedGastroWidget);
    ui->stackedGastroWidget->addWidget(m_order);

    m_openTickets = new QRKGastroOpenTickets(ui->stackedGastroWidget);
    ui->stackedGastroWidget->addWidget(m_openTickets);

    connect(m_selector, &QRKGastroSelector::cancelGastroButton_clicked, this, &QRKGastro::cancelGastroButton_clicked);
    if (isBlackListet()) {
        connect(m_selector, &QRKGastroSelector::tableOrder, [=] {
            notRegistered(-1);
        });
        return;
    }

    connect(m_selector, &QRKGastroSelector::tableOrder, this, &QRKGastro::tableOrder);

    connect(m_openTickets, static_cast<void (QRKGastroOpenTickets::*)(int)>(&QRKGastroOpenTickets::newTicket), this,
        &QRKGastro::newTableOrder);
    connect(m_openTickets, static_cast<void (QRKGastroOpenTickets::*)(int, int)>(&QRKGastroOpenTickets::changeTicket),
        this, &QRKGastro::changeTableOrderTicket);
    connect(m_openTickets, &QRKGastroOpenTickets::leaveTicket, this, &QRKGastro::leaveOpenTickets);
    connect(m_order, &QRKGastroTableOrder::cancelOrder, this, &QRKGastro::cancelTableOrder);
    connect(m_order, &QRKGastroTableOrder::updateOrderButton, this, &QRKGastro::updateButton);
    connect(m_order, &QRKGastroTableOrder::payTicket, m_openTickets,
        static_cast<void (QRKGastroOpenTickets::*)(int, bool)>(&QRKGastroOpenTickets::payTicket));

    m_checkerThread = new QThread;
    m_checker = new QrkGastroCurfewChecker;
    m_checker->moveToThread(m_checkerThread);

    connect(m_checker, &QrkGastroCurfewChecker::curFew, this, &QRKGastro::curfewdiff);
    connect(m_checkerThread, &QThread::started, m_checker, &QrkGastroCurfewChecker::run);
    connect(m_checker, &QrkGastroCurfewChecker::finished, m_checkerThread, &QThread::quit);
    connect(m_checker, &QrkGastroCurfewChecker::finished, m_checker, &QrkGastroCurfewChecker::deleteLater);
    connect(m_checkerThread, &QThread::finished, m_checkerThread, &QThread::deleteLater);
    m_checkerThread->start();
}

QRKGastro::~QRKGastro()
{
    delete ui;
}

void QRKGastro::curfewdiff(int diff)
{

    if (!openTickets()) return;

    QMessageBox mb(tr("Sperrstunde in %1 Minuten").arg(diff / 60),
        tr("Es ist bald Sperrstunde. Bitte offene Bonierungen kassieren. Nicht "
           "abgeschlossene Bonierungen werden auf den nächsten Tag verschoben."),
        QMessageBox::Question, QMessageBox::Yes | QMessageBox::Default, QMessageBox::NoButton, QMessageBox::NoButton,
        this);
    mb.setButtonText(QMessageBox::Yes, tr("Ok"));
    mb.setDefaultButton(QMessageBox::Yes);
    mb.exec();
}

void QRKGastro::notRegistered(int days)
{
    if (days >= 0) {
        QMessageBox mb(tr("Test Version"), tr("Ihre QRK Gastro Version wird in %1 Tagen ablaufen.").arg(days),
            QMessageBox::Information, QMessageBox::Yes | QMessageBox::Default, QMessageBox::NoButton,
            QMessageBox::NoButton, this);
        mb.setButtonText(QMessageBox::Yes, tr("Ok"));
        mb.setDefaultButton(QMessageBox::Yes);
        mb.exec();
    } else {
        QMessageBox mb(tr("Version abgelaufen"),
            tr("Leider ist Ihre QRK Gastro Version abgelaufen oder nicht freigeschalten."), QMessageBox::Information,
            QMessageBox::Yes | QMessageBox::Default, QMessageBox::NoButton, QMessageBox::NoButton, this);
        mb.setButtonText(QMessageBox::Yes, tr("Ok"));
        mb.setDefaultButton(QMessageBox::Yes);
        mb.exec();
        emit cancelGastroButton_clicked();
    }
}

bool QRKGastro::init()
{

    if (RegistrationTab::isActive("QRK-GASTRO")) {
        RegistrationTab reg("QRK-GASTRO", true, this);
        int days;
        if (!reg.isValid(days)) {
            emit notRegistered(days);
            if (days <= 0) return false;
        } else if (days >= 0) {
            emit notRegistered(days);
        }

        m_selector->refresh();
        m_order->refresh();
        return true;
    }
    RegistrationTab reg("QRK-GASTRO", true, this);
    int days;
    reg.isValid(days);
    if (RegistrationTab::isActive("QRK-GASTRO")) {
        m_selector->refresh();
        m_order->refresh();
        return true;
    }
    return false;
}

void QRKGastro::tableOrder(int id)
{
    qDebug() << "Function Name: " << Q_FUNC_INFO << "id: " << id;
    m_lastTableId = id;
    if (m_openTickets->setTableId(id) > 0) ui->stackedGastroWidget->setCurrentWidget(m_openTickets);
}

void QRKGastro::newTableOrder(int id)
{
    changeTableOrderTicket(id, 0);
}

void QRKGastro::changeTableOrderTicket(int tableId, int ticketId)
{

    bool ret = Reports().mustDoEOAny(QDateTime::currentDateTime());
    if (ret) {
        if (QTime::currentTime() < QTime(8, 0, 0)) {
            Database::setCurfewTime(QTime::currentTime().addSecs(120), true);
        } else {
            infoMessage(true);
            return;
        }
    }

    /*    if (!QRKGastro::isHotelRoom(tableId) && Reports().mustDoEOAny(QDateTime::currentDateTime()))
            ui->stackedGastroWidget->setCurrentWidget(m_selector);
        else { */
    m_order->setTableId(tableId);
    m_order->setTicketId(ticketId);
    ui->stackedGastroWidget->setCurrentWidget(m_order);
    //    }
}

void QRKGastro::updateButton(int id)
{

    DragPushButton *pb = m_selector->getTableButton(id);
    if (Q_NULLPTR == pb) return;

    QBCMath sum(0);
    double sumDouble = QRKGastro::getOrderSum(id).toDouble();
    if (sumDouble != 0.00) {
        int customid = pb->getCustomId();
        if (customid == 0) pb->setCustomId(RBAC::Instance()->getUserId());
        customid = pb->getCustomId();
        if (customid > 0 && customid == RBAC::Instance()->getUserId())
            pb->setButtonColor("green");
        else
            pb->setButtonColor("red");

        sum = sumDouble;
    } else {
        pb->restoreButtonColor();
        pb->setCustomId(0);
    }
    sum.round(2);
    pb->setPriceText(sum.toLocale() + " " + QLocale().currencySymbol());
}

void QRKGastro::cancelTableOrder(int tableId, bool leaveTickets)
{

    m_selector->refresh();
    if (m_openTickets->setTableId(tableId) > 0 && !leaveTickets)
        ui->stackedGastroWidget->setCurrentWidget(m_openTickets);
    else
        emit leaveOpenTickets();
}

void QRKGastro::leaveOpenTickets()
{

    m_selector->refresh();
    ui->stackedGastroWidget->setCurrentWidget(m_selector);
}

void QRKGastro::fillOrderList(QTreeWidget *tree, int ticketId)
{

    tree->clear();

    QSqlDatabase dbc = Database::database();
    CSqlQuery orders(dbc, Q_FUNC_INFO);

    orders.prepare("SELECT ticketorders.count, ticketorders.product, ticketorders.gross, "
                   "products.name, products.tax, ticketorders.id FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:ticketId");
    orders.bindValue(":ticketId", ticketId);

    orders.exec();
    CSqlQuery orderdescs(dbc, Q_FUNC_INFO);
    orderdescs.prepare("SELECT description FROM orderdescs WHERE orderId=:id AND type=1");

    while (orders.next()) {
        int orderId = orders.value("id").toInt();
        orderdescs.bindValue(":id", orderId);
        orderdescs.exec();
        QString description;
        if (orderdescs.next()) description = orderdescs.value("description").toString();

        QBCMath productPrice(orders.value("gross").toDouble());
        productPrice.round(2);
        QBCMath tax(orders.value("tax").toDouble());
        tax.round(2);

        QTreeWidgetItem *item = new QTreeWidgetItem;
        item->setData(0, Qt::DisplayRole, orders.value("count").toInt()); // count
        item->setData(1, Qt::DisplayRole,
            orders.value("name").toString()); // product name
        item->setData(1, QRKGastro::PRODUCT_ID, orders.value("product").toInt());
        item->setData(1, QRKGastro::ORDER_ID, orderId);
        item->setData(2, QRKGastro::PRODUCT_PRICE, productPrice.toDouble());
        item->setData(2, QRKGastro::PRODUCT_TAX, tax.toDouble());
        item->setData(3, QRKGastro::ORDER_DESCRIPTION, description);
        item->setIcon(3, QIcon(":src/icons/textfield.png"));
        item->setSizeHint(0, QSize(50, 50));
        tree->addTopLevelItem(item);

        // check for orderextras
        CSqlQuery extras(dbc, Q_FUNC_INFO);
        extras.prepare("SELECT orderextras.type, orderextras.product, "
                       "products.name, products.gross FROM orderextras"
                       " LEFT JOIN products ON orderextras.product=products.id"
                       " WHERE orderId=:orderId");
        extras.bindValue(":orderId", orderId);

        extras.exec();
        while (extras.next()) {
            QBCMath singlegross(extras.value("gross").toDouble());
            singlegross.round(2);
            QTreeWidgetItem *child = new QTreeWidgetItem(item);
            child->setData(0, Qt::DisplayRole, (extras.value("type").toInt() == QRKGastro::TYPE_WITH) ? "+" : "-");
            child->setData(1, QRKGastro::EXTRA_TYPE, extras.value("type").toInt());
            child->setData(1, QRKGastro::PRODUCT_ID, extras.value("product").toInt());
            child->setData(1, Qt::DisplayRole, extras.value("name").toString());
            child->setData(2, QRKGastro::PRODUCT_PRICE, singlegross.toDouble());

            item->setExpanded(true);
        }
    }
}

int QRKGastro::getFirstRoomId()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery room(dbc, Q_FUNC_INFO);
    int id = 0;
    room.prepare("SELECT min(id) as id FROM `rooms` ORDER BY sortorder, name LIMIT 1");
    room.exec();
    if (room.next()) id = room.value("id").toInt();

    return id;
}

QString QRKGastro::getGuestName(int ticketid)
{
    QSqlDatabase dbc = Database::database();
    QString guestName;

    CSqlQuery guest(dbc, Q_FUNC_INFO);
    guest.prepare("SELECT guestname FROM `tickets` WHERE id=:ticketId");
    guest.bindValue(":ticketId", ticketid);
    guest.exec();
    if (guest.next()) {
        guestName = guest.value("guestname").toString();
        return guestName;
    }
    return {};
}

QString QRKGastro::getRoomName(int roomId)
{
    QSqlDatabase dbc = Database::database();
    QString roomName;

    CSqlQuery room(dbc, Q_FUNC_INFO);
    room.prepare("SELECT name FROM `rooms` WHERE id=:roomId");
    room.bindValue(":roomId", roomId);
    room.exec();
    if (room.next()) {
        roomName = room.value("name").toString();
        return roomName;
    }
    return QString::number(roomId);
}

QString QRKGastro::getRoomNameFromTableId(int tableId)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QString roomName;

    query.prepare("SELECT roomId FROM `tables` WHERE id=:tableId");
    query.bindValue(":tableId", tableId);
    query.exec();
    if (query.next()) {
        CSqlQuery room(dbc, Q_FUNC_INFO);
        room.prepare("SELECT name FROM `rooms` WHERE id=:roomId");
        room.bindValue(":roomId", query.value("roomId").toInt());
        room.exec();
        if (room.next()) {
            roomName = room.value("name").toString();
            return roomName;
        }
        return QString::number(query.value("roomId").toInt());
    }

    return "n/a";
}

int QRKGastro::getRoomIdFromTableId(int tableId)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT roomId FROM `tables` WHERE id=:tableId");
    query.bindValue(":tableId", tableId);
    query.exec();
    if (query.next()) {
        return query.value("roomId").toInt();
    }

    return 0;
}

QString QRKGastro::getTableName(int tableId)
{

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT name FROM `tables` WHERE id=:tableId");
    query.bindValue(":tableId", tableId);
    query.exec();
    if (!query.next()) // no name defined for that table num
        return QString::number(tableId);

    QString tableName = query.value("name").toString();
    if (tableName.isEmpty()) tableName = QString::number(tableId);

    return tableName;
}

QString QRKGastro::getOrderSum(int table)
{
    QBCMath sum(0);
    QBCMath total(0);
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("select id from tickets where tickets.tableId=:tableId and open=1;");
    query.bindValue(":tableId", table);
    query.exec();

    CSqlQuery orders(dbc, Q_FUNC_INFO);
    orders.prepare("SELECT ticketorders.count, ticketorders.gross FROM "
                   "ticketorders WHERE ticketorders.ticketId=:ticketId");
    while (query.next()) {
        orders.bindValue(":ticketId", query.value("id").toInt());
        orders.exec();
        while (orders.next()) {
            sum = orders.value("gross").toDouble();
            sum.round(2);
            total += sum * orders.value("count").toInt();
        }
    }

    return total.toString();
}

bool QRKGastro::isOpenTicked(int tableId)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery orders(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id FROM tickets WHERE tableId=:tableId AND open = 1");
    query.bindValue(":tableId", tableId);
    query.exec();

    if (query.next()) return true;

    return false;
}

bool QRKGastro::isOrderNotServed(int tableId)
{
    bool toServe = false;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery orders(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id FROM tickets WHERE tableId=:tableId AND open > 0");
    query.bindValue(":tableId", tableId);
    query.exec();

    orders.prepare("SELECT (ticketorders.count - ticketorders.printed) AS count, "
                   "products.name, ticketorders.id FROM ticketorders "
                   " LEFT JOIN products ON ticketorders.product=products.id"
                   " WHERE ticketorders.ticketId=:id AND (ticketorders.count > "
                   "ticketorders.printed)");

    while (query.next()) {
        orders.bindValue(":id", query.value("id").toInt());
        orders.exec();
        if (orders.next()) {
            toServe = true;
        }
    }

    return toServe;
}

bool QRKGastro::createOrUpdateTicket(QTreeWidget *tree, int &ticketId, int tableId, QString guestname, bool asServed)
{
    if (tree->topLevelItemCount() == 0) { // nothing ordered
        qWarning() << "Function Name: " << Q_FUNC_INFO << " ToplevelCount: 0";
        return false;
    }

    QrkJournal journal;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    if (ticketId == 0) { // a new ticket
        query.prepare("INSERT INTO tickets (`tableId`, `timestamp`, `guestname`) "
                      "VALUES(:tableId, :timestamp, :guestname)");
        query.bindValue(":tableId", tableId);
        query.bindValue(":timestamp", QDateTime::currentDateTime().toString(Qt::ISODate));
        query.bindValue(":guestname", guestname);

        query.exec();
        query.exec("SELECT MAX(id) AS id FROM tickets;");
        query.next();
        ticketId = query.value("id").toInt();
        QString line = QString("Bonierung\tNeu\t\t%1\t%2\tTisch %3")
                           .arg(ticketId)
                           .arg(QDateTime::currentDateTime().toString(Qt::ISODate))
                           .arg(getTableName(tableId));

        journal.journalInsertLine("Bonierung", line);
    } else {
        // change table in DB
        if (guestname.isNull()) {
            query.prepare("UPDATE tickets SET tableId=:tableId WHERE id=:id");
            query.bindValue(":tableId", tableId);
            query.bindValue(":id", ticketId);
        } else {
            query.prepare("UPDATE tickets SET tableId=:tableId, guestname=:guestname WHERE id=:id");
            query.bindValue(":tableId", tableId);
            query.bindValue(":id", ticketId);
            query.bindValue(":guestname", guestname);
        }

        bool ok = query.exec();
        if (!ok) {
            return false;
        }
        QString line = QString("Bonierung\tÄnderung\t\t%1\t%2\tTisch %3")
                           .arg(ticketId)
                           .arg(QDateTime::currentDateTime().toString(Qt::ISODate))
                           .arg(getTableName(tableId));

        journal.journalInsertLine("Bonierung", line);
    }

    for (int i = 0; i < tree->topLevelItemCount(); i++) {
        QTreeWidgetItem *item = tree->topLevelItem(i);
        // QVariant v = item->data(1, QRKGastro::ORDER_ID);
        // QVariant v1 = item->data(1, ORDER_ID);
        int orderId = item->data(1, QRKGastro::ORDER_ID).toInt();

        if (orderId != -1) updateOrderDescription(orderId, item->data(3, QRKGastro::ORDER_DESCRIPTION).toString());

        if (item->isHidden()) {  // shall be removed
            if (orderId != -1) { // was already in DB
                // ordersExtras before orders, as ordersExtras refers to orders.id
                query.prepare("DELETE FROM orderextras WHERE orderId=:orderId");
                query.bindValue(":orderId", orderId);
                bool ok = query.exec();
                if (!ok) {
                    return false;
                }

                query.prepare("DELETE FROM ticketorders WHERE id=:orderId");
                query.bindValue(":orderId", orderId);
                ok = query.exec();
                if (!ok) {
                    return false;
                }
            }
        } else {                 // insert or update
            if (orderId != -1) { // was already in DB
                QBCMath productPrice(item->data(2, PRODUCT_PRICE).toDouble());
                productPrice.round(2);
                CSqlQuery update(dbc, Q_FUNC_INFO);
                update.prepare("UPDATE ticketorders SET count=:count, gross=:gross WHERE id=:id");
                update.bindValue(":count", item->data(0, Qt::DisplayRole).toInt());
                update.bindValue(":id", orderId);
                update.bindValue(":gross", productPrice.toDouble());

                bool ok = update.exec();
                if (!ok) {
                    return false;
                }

                QString line = QString("Bonierung\tÄnderung\t\t%1\t%2\tTisch %3\t%4\t%5")
                                   .arg(orderId)
                                   .arg(QDateTime::currentDateTime().toString(Qt::ISODate))
                                   .arg(getTableName(tableId))
                                   .arg(item->data(0, Qt::DisplayRole).toInt())
                                   .arg(item->data(1, Qt::DisplayRole).toString());

                journal.journalInsertLine("Textposition", line);
            } else {
                QBCMath productPrice(item->data(2, PRODUCT_PRICE).toDouble());
                productPrice.round(2);
                query.prepare("INSERT INTO ticketorders (ticketId, count, product, gross, "
                              "printed) VALUES(:ticketId, :count, :product, :gross, :printed)");
                query.bindValue(":ticketId", ticketId);
                query.bindValue(":count", item->data(0, Qt::DisplayRole).toInt());
                query.bindValue(":product", item->data(1, PRODUCT_ID).toInt());
                query.bindValue(":gross", productPrice.toDouble());
                query.bindValue(":printed", asServed ? item->data(0, Qt::DisplayRole).toInt() : 0);
                bool ok = query.exec();
                if (!ok) {
                    return false;
                }

                query.exec("SELECT MAX(id) FROM ticketorders;");
                query.next();
                orderId = query.value(0).toInt();
                item->setData(1, QRKGastro::ORDER_ID,
                    orderId); // so that another call to this method does not
                              // insert items again to DB
                QString line = QString("Bonierung\tÄnderung\t\t%1\t%2\tTisch %3\t%4\t%5")
                                   .arg(orderId)
                                   .arg(QDateTime::currentDateTime().toString(Qt::ISODate))
                                   .arg(getTableName(tableId))
                                   .arg(item->data(0, Qt::DisplayRole).toInt())
                                   .arg(item->data(1, Qt::DisplayRole).toString());

                journal.journalInsertLine("Textposition", line);
            }

            // make it simple: delete all orderextras for this order item and insert
            // all we want now
            query.prepare("DELETE FROM orderextras WHERE orderId=:orderId");
            query.bindValue(":orderId", orderId);
            bool ok = query.exec();
            if (!ok) {
                return false;
            }

            for (int j = 0; j < item->childCount(); j++) {
                QTreeWidgetItem *child = item->child(j);
                child->setData(1, QRKGastro::ORDER_ID, orderId);
                if (!child->isHidden()) {
                    query.prepare("INSERT INTO orderextras (orderId, ticketId, type, product) "
                                  "VALUES(:orderId, :ticketId, :type, :product)");
                    query.bindValue(":orderId", orderId);
                    query.bindValue(":ticketId", ticketId);
                    query.bindValue(":type", child->data(1, EXTRA_TYPE).toInt());
                    query.bindValue(":product", child->data(1, PRODUCT_ID).toInt());
                    bool ok = query.exec();
                    if (!ok) {
                        return false;
                    }
                }
            }
        }
    }
    query.prepare("SELECT id FROM ticketorders WHERE ticketId=:ticketId");
    query.bindValue(":ticketId", ticketId);
    query.exec();
    if (query.next()) return true;

    // we have no orders for this ticket
    query.prepare("DELETE FROM tickets WHERE id=:ticketId");
    query.bindValue(":ticketId", ticketId);
    query.exec();
    ticketId = 0;
    return true;
}

void QRKGastro::updateOrderDescription(int id, const QString &descripton)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery orderDescription(dbc, Q_FUNC_INFO);
    QString oldDescription = "";
    orderDescription.prepare("SELECT description FROM orderdescs WHERE orderId=:id AND type=1");
    orderDescription.bindValue(":id", id);
    orderDescription.exec();
    if (orderDescription.next()) oldDescription = orderDescription.value("description").toString();

    if (oldDescription == descripton) return;

    if (oldDescription.isEmpty()) {
        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.prepare("INSERT INTO orderdescs (type, orderId, description) "
                      "VALUES(:type, :id, :description)");
        query.bindValue(":type", 1);
        query.bindValue(":description", descripton);
        query.bindValue(":id", id);
        query.exec();
        return;
    }

    if (descripton.isEmpty()) {
        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.prepare("DELETE FROM orderdescs WHERE orderId=:id");
        query.bindValue(":id", id);
        query.exec();
        return;
    }

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("UPDATE orderdescs SET description=:description WHERE orderId=:id");
    query.bindValue(":id", id);
    query.bindValue(":description", descripton);
    query.exec();
}

bool QRKGastro::openTickets(bool includeHotel)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery ot(dbc, Q_FUNC_INFO);
    CSqlQuery hotel(dbc, Q_FUNC_INFO);

    ot.exec("SELECT tableId from tickets WHERE open = 1");
    hotel.prepare("SELECT isHotel from rooms WHERE id = :id");

    int count = 0;

    while (ot.next()) {
        int tableId = ot.value("tableid").toInt();
        hotel.bindValue(":id", getRoomIdFromTableId(tableId));
        hotel.exec();
        if (hotel.next()) {
            bool isHotel = hotel.value("isHotel").toBool();
            if (!isHotel || includeHotel) count++;
        }
    }

    return count > 0;
}

QStringList QRKGastro::openTicketsList()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery ot(dbc, Q_FUNC_INFO);

    QStringList list;

    ot.exec("SELECT tableId from tickets WHERE open = 1");

    while (ot.next()) {
        int tableId = ot.value("tableid").toInt();
        if (!QRKGastro::isHotelRoom(tableId))
            list.append(tr("Raum: %1 - Tisch: %2")
                    .arg(QRKGastro::getRoomNameFromTableId(tableId))
                    .arg(QRKGastro::getTableName(tableId)));
    }

    return list;
}

bool QRKGastro::isHotelRoom(int tableId)
{
    int roomid = getRoomIdFromTableId(tableId);
    if (roomid < 1) return false;

    QSqlDatabase dbc = Database::database();
    CSqlQuery hotel(dbc, Q_FUNC_INFO);

    hotel.prepare("SELECT isHotel from rooms WHERE id = :id");
    hotel.bindValue(":id", roomid);
    hotel.exec();
    if (hotel.next()) {
        bool isHotel = hotel.value("isHotel").toBool();
        return isHotel;
    }

    return false;
}

void QRKGastro::infoMessage(bool paynow)
{

    if (QRKGastro::openTickets() && !paynow) {
        QMessageBox info;
        QStringList list = QRKGastro::openTicketsList();
        info.setDetailedText(list.join("\n"));
        info.information(Q_NULLPTR, tr("Fehlender Abschluss"),
            tr("Leider ist zurzeit kein neues Bonieren möglich.\n"
               "Als erstes bitte offene Bonierungen abschließen, danach den "
               "fehlenden Abschluss erstellen.\n"
               "Wenn es keine offenen Bonierungen gibt, kommt die Aufforderung "
               "automatisch beim nächsten Versuch."),
            QMessageBox::Ok);
        return;
    } else if (QRKGastro::openTickets(true) && paynow) {
        QMessageBox info;
        QStringList list = QRKGastro::openTicketsList();
        info.setDetailedText(list.join("\n"));
        info.information(Q_NULLPTR, tr("Fehlender Abschluss"),
            tr("Derzeit sind keine weiteren Aktionen möglich.\n"
               "Es ist erforderlich, den aktuellen Abschluss durchzuführen."),
            QMessageBox::Ok);
    }

    Reports rep;
    rep.checkEOAny();
}

bool QRKGastro::isBlackListet()
{
    QStringList list;

    if (list.isEmpty()) return false;

    QVariant vValue;
    QString strValue;
    int id = AbstractDataBase::select_globals("QRK-GASTRO", vValue, strValue);
    if (id > 0) {
        QJsonObject sign = Verification().readJsonFromString(strValue);
        if (list.contains(sign.value("Signature").toString())) return true;
    }

    return false;
}
