/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2026 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/profeatures/profeatures.h"
#include "barcodessettings.h"
#include "database.h"
#include "defines.h"
#include "documentprinter.h"
#include "qrkmultimedia.h"
#include "qrksettings.h"
#include "receiptitemmodel.h"
#include "reports.h"

#include "barcodes.h"

#include <QJsonObject>
#include <QtWidgets>

Barcodes::Barcodes()
{
    initBarcodes();
    m_discount = false;

    m_root = new QDialog();
}

Barcodes::~Barcodes()
{
}

void Barcodes::initBarcodes()
{

    QrkSettings settings;
    settings.beginGroup("BarCodesPlugin");

    m_barcode_finishReceipt = settings.value("barcodeFinishReceipt", "100009000001").toString();
    m_barcodes << m_barcode_finishReceipt;

    m_barcode_removeLastPosition = settings.value("barcodeRemoveLastPosition", "100009000002").toString();
    m_barcodes << m_barcode_removeLastPosition;

    m_barcode_endOfDay = settings.value("barcodeEndOfDay", "100009000003").toString();
    m_barcodes << m_barcode_endOfDay;

    m_barcode_discount = settings.value("barcodeDiscount", "100009000007").toString();
    m_barcodes << m_barcode_discount;

    m_barcode_editPrice = settings.value("barcodeEditPrice", "100009000010").toString();
    m_barcodes << m_barcode_editPrice;

    m_barcode_printLastReceiptAgain = settings.value("barcodePrintLastReceiptAgain", "100009000005").toString();
    m_barcodes << m_barcode_printLastReceiptAgain;

    m_barcode_cancelLastReceipt = settings.value("barcodeCancelReceipt", "100009000006").toString();
    m_barcodes << m_barcode_cancelLastReceipt;

    m_barcode_amount_0 = settings.value("barcodeAmount_0", "100008000000").toString();
    m_barcodes << m_barcode_amount_0;

    m_barcode_amount_1 = settings.value("barcodeAmount_1", "100008000001").toString();
    m_barcodes << m_barcode_amount_1;

    m_barcode_amount_2 = settings.value("barcodeAmount_2", "100008000002").toString();
    m_barcodes << m_barcode_amount_2;

    m_barcode_amount_3 = settings.value("barcodeAmount_3", "100008000003").toString();
    m_barcodes << m_barcode_amount_3;

    m_barcode_amount_4 = settings.value("barcodeAmount_4", "100008000004").toString();
    m_barcodes << m_barcode_amount_4;

    m_barcode_amount_5 = settings.value("barcodeAmount_5", "100008000005").toString();
    m_barcodes << m_barcode_amount_5;

    m_barcode_amount_6 = settings.value("barcodeAmount_6", "100008000006").toString();
    m_barcodes << m_barcode_amount_6;

    m_barcode_amount_7 = settings.value("barcodeAmount_7", "100008000007").toString();
    m_barcodes << m_barcode_amount_7;

    m_barcode_amount_8 = settings.value("barcodeAmount_8", "100008000008").toString();
    m_barcodes << m_barcode_amount_8;

    m_barcode_amount_9 = settings.value("barcodeAmount_9", "100008000009").toString();
    m_barcodes << m_barcode_amount_9;

    m_barcode_amount_00 = settings.value("barcodeAmount_00", "100008000020").toString();
    m_barcodes << m_barcode_amount_00;

    m_barcode_amount_000 = settings.value("barcodeAmount_000", "100008000030").toString();
    m_barcodes << m_barcode_amount_000;

    m_barcode_amount_250 = settings.value("barcodeAmount_250", "100008000250").toString();
    m_barcodes << m_barcode_amount_250;

    m_barcode_amount_500 = settings.value("barcodeAmount_500", "100008000500").toString();
    m_barcodes << m_barcode_amount_500;

    settings.endGroup();
}

bool Barcodes::process(ReceiptItemModel *model, int currIndex, QString barcode)
{
    m_model = model;
    m_index = currIndex;
    if (m_index < 0) return false;

    bool ean13 = isEan13Valid(barcode);

    switch (m_barcodes.indexOf(barcode)) {
        case 0: { // m_barcode_finishReceipt
            if (m_model->item(m_index, REGISTER_COL_PRODUCT)->text() == "") break;
            if (m_model->rowCount() == 0) break;

            Reports rep;
            bool ret = rep.checkEOAny();

            if (ret) {
                if (int id = m_model->createReceipts()) {
                    m_model->setCurrentReceiptNum(id);
                    if (m_model->createOrder()) {
                        m_model->finishReceipts(PAYED_BY_CASH);
                        emit finishedReceipt();
                    }
                }
            }
            break;
        }
        case 1: { // m_barcode_removeLastPosition
            emit minusSlot();
            break;
        }
        case 2: { // m_barcode_endOfDay
            Reports rep;
            rep.endOfDay(false);
            break;
        }
        case 3: { // m_barcode_discount
            m_discount = true;
            initAmount();
            initAppendType();
            m_model->item(m_index, REGISTER_COL_COUNT_TYPE_STR)->setText("1");
            resetAmount();
            break;
        }
        case 4: { // m_barcode_editPrice
            m_discount = false;
            initAmount();
            initAppendType();
            m_model->item(m_index, REGISTER_COL_COUNT_TYPE_STR)->setText("1");
            resetAmount();
            break;
        }
        case 5: { // m_barcode_printLastReceiptAgain
            printReceipt();
            break;
        }
        case 6: { // m_barcode_cancelLastReceipt
            stornoReceipt();
            break;
        }
        case 7: { // m_barcode_amount_0
            handleAmount("0");
            break;
        }
        case 8: { // m_barcode_amount_1
            handleAmount("1");
            break;
        }
        case 9: { // m_barcode_amount_2
            handleAmount("2");
            break;
        }
        case 10: { // m_barcode_amount_3
            handleAmount("3");
            break;
        }
        case 11: { // m_barcode_amount_4
            handleAmount("4");
            break;
        }
        case 12: { // m_barcode_amount_5
            handleAmount("5");
            break;
        }
        case 13: { // m_barcode_amount_6
            handleAmount("6");
            break;
        }
        case 14: { // m_barcode_amount_7
            handleAmount("7");
            break;
        }
        case 15: { // m_barcode_amount_8
            handleAmount("8");
            break;
        }
        case 16: { // m_barcode_amount_9
            handleAmount("9");
            break;
        }
        case 17: { // m_barcode_amount_00
            handleAmount("00");
            break;
        }
        case 18: { // m_barcode_amount_000
            handleAmount("000");
            break;
        }
        case 19: { // m_barcode_amount_250
            handleAmount("250");
            break;
        }
        case 20: { // m_barcode_amount_500
            handleAmount("500");
            break;
        }
        default: {
            if (ean13) {
                bool ret = processEan13(barcode);
                if (ret) QrkMultimedia::play(QRKMULTIMEDIA::BARCODE_SUCCSESS);
                return ret;
            }

            return false;
        }
    }

    return true;
}

void Barcodes::printReceipt()
{
    int id = Database::getLastReceiptNum(true);

    ReceiptItemModel reg;
    reg.setCurrentReceiptNum(id);

    QrkSettings settings;
    QJsonObject data = reg.compileData();
    data["isCopy"] = true;
    int storno = Database::getStorno(id);
    if (storno == 2) {
        id = Database::getStornoId(id);
        data["comment"] = (id > 0) ? tr("Storno für Beleg Nr: %1").arg(id)
                                   : settings.value("receiptPrinterHeading", "KASSABON").toString();
    }

    data["headerText"] = Database::getCustomerText(id);

    DocumentPrinter p;
    p.printReceipt(data);
    QApplication::restoreOverrideCursor();
}

void Barcodes::stornoReceipt()
{
    int id = Database::getLastReceiptNum(true);
    int payedBy = Database::getPayedBy(id);

    int storno = Database::getStorno(id);
    if (storno)
        return; // kann nicht storniert werden. wurde schon storniert oder ist ein
                // storno Beleg

    // Hier wird gecheckt ob ein Tags/Monatsabschluss gemacht werden muss
    Reports rep;
    bool ret = rep.checkEOAny();
    if (!ret) {
        return; // Fehler ???
    }

    ReceiptItemModel reg;
    reg.newOrder();
    reg.storno(id);

    int currentReceipt = reg.createReceipts();
    if (currentReceipt) {
        reg.setCurrentReceiptNum(currentReceipt);
        if (reg.createOrder(true)) {
            if (reg.finishReceipts(payedBy, id)) {
                return;
            }
        }
    }
}

void Barcodes::handleAmount(QString amount)
{

    if (m_index >= 0) {

        initAmount();
        initAppendType();

        QStandardItem *item = m_model->item(m_index, REGISTER_COL_COUNT_TYPE_STR);
        if (QString::compare(item->text(), "0", Qt::CaseInsensitive) == 0) {
            appendToAmount(amount);
        } else {
            if (m_discount)
                appendToDiscount(amount);
            else
                appendToPrice(amount);
        }
    }
}

void Barcodes::appendToAmount(QString digits)
{

    QString val;
    if (digits.length() == 1 || digits.startsWith("0"))
        val = m_model->item(m_index, REGISTER_COL_COUNT_STR)->text() + digits;
    else
        val = digits;

    m_model->item(m_index, REGISTER_COL_COUNT_STR)->setText(val);

    QString valueAsString;

    if (m_useStoredDecimals) {
        QBCMath current(m_model->item(m_index, REGISTER_COL_COUNT)->text().toDouble());
        QrkSettings settings;
        if (settings.value("useDecimalQuantity", false).toBool()) {
            int decimalDigits = settings.value("decimalDigits", 2).toInt();
            if (decimalDigits > 2) {
                current += QString::number(val.toDouble() / 1000.0);
                current.round(3);
                valueAsString = current.toString();
            } else {
                current += QString::number(val.toDouble() / 100.0);
                current.round(2);
                valueAsString = current.toString();
            }
        } else {
            current += QString::number(val.toDouble());
            current.round(0);
            valueAsString = current.getIntPart();
        }
    } else {
        valueAsString = QString::number(val.toDouble() / 1000.0);
    }

    m_model->item(m_index, REGISTER_COL_COUNT)->setText(valueAsString);
}

void Barcodes::initAmount()
{
    init(REGISTER_COL_COUNT_STR, "");
}

void Barcodes::initAppendType()
{
    init(REGISTER_COL_COUNT_TYPE_STR, "0");
}

void Barcodes::resetAmount()
{
    m_model->item(m_index, REGISTER_COL_COUNT_STR)->setText("");
}

void Barcodes::appendToPrice(QString digits)
{

    QString val;
    if (digits.length() == 1 || digits.startsWith("0"))
        val = m_model->item(m_index, REGISTER_COL_COUNT_STR)->text() + digits;
    else
        val = digits;

    m_model->item(m_index, REGISTER_COL_COUNT_STR)->setText(val);

    QString valueAsString = QString::number(val.toDouble() / 100.0);
    m_model->item(m_index, REGISTER_COL_SINGLE)->setText(valueAsString);
}

void Barcodes::appendToDiscount(QString digits)
{

    QString val;
    if (digits.length() == 1 || digits.startsWith("0"))
        val = m_model->item(m_index, REGISTER_COL_COUNT_STR)->text() + digits;
    else
        val = digits;

    m_model->item(m_index, REGISTER_COL_COUNT_STR)->setText(val);

    QString valueAsString = QString::number(val.toDouble() / 100.0);
    m_model->item(m_index, REGISTER_COL_DISCOUNT)->setText(valueAsString);
}

void Barcodes::init(int col, QString val)
{

    QStandardItem *item = m_model->item(m_index, col);
    if (item == Q_NULLPTR) {
        item = new QStandardItem(val);
        m_model->setItem(m_index, col, item);
        emit setColumnHidden(col);
    }
}

void Barcodes::setupUi()
{
    BarcodesSettings *barcodesettings = new BarcodesSettings();

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(barcodesettings);
    mainLayout->addStretch(1);
    mainLayout->addSpacing(12);
    m_root->setLayout(mainLayout);

    m_root->setWindowTitle(tr("Barcodes"));
    m_root->setMinimumWidth(700);

    if (QApplication::desktop()->height() < 650) {
        m_root->setFixedHeight(550);
    }

    connect(barcodesettings, &BarcodesSettings::cancelClicked, m_root, &QDialog::accept);
}

QDialog *Barcodes::SettingsDialog()
{
    setupUi();
    return m_root;
}


bool Barcodes::isEan13Valid(const QString ean13)
{
    if (ean13.length() != 13) return false;

    if (ean13.right(1).toInt() == calculateChecksum(ean13)) return true;

    return false;
}

int Barcodes::calculateChecksum(const QString ean13)
{
    const std::string ean = ean13.toStdString();
    int evens = 0;
    int odds = 0;

    for (int i = 0; i < 12; i = i + 2) {
        evens += ean[i] - '0';
        odds += ean[i + 1] - '0';
    }

    int S = evens + 3 * odds;
    int checkSum = 10 - S % 10;
    if (checkSum == 10) checkSum = 0;

    return checkSum;
}
bool Barcodes::processEan13(QString &barcode)
{

    QrkSettings settings;

    if (barcode.startsWith("21") || barcode.startsWith("22")) {
        QString partnumber = barcode.left(7);
        int id = Database::getProductIdByBarcode(partnumber);
        if (id > 0) {
            m_discount = false;

            bool newitem = false;
            int gross = 0;
            int currentGross = barcode.mid(7, 5).toInt();

            QList<QStandardItem *> items
                = m_model->findItems(Database::getProductNameById(id), Qt::MatchExactly, REGISTER_COL_PRODUCT);
            foreach (QStandardItem *item, items) {
                gross = m_model->item(item->row(), REGISTER_COL_SINGLE)->text().toDouble() * 100;
                m_index = item->row();
                if (gross == currentGross) break;
            }

            m_model->setCurrentRow(m_index);

            bool group = settings.value("BarCodesPlugin/Ean13Type21_22_group", true).toBool();
            if (group)
                newitem = gross != currentGross;
            else
                newitem = true;

            // && m_model->item(m_index, REGISTER_COL_PRODUCT)->text() != ""
            emit addProduct(id, newitem && m_model->item(m_index, REGISTER_COL_PRODUCT)->text() != "");
            m_index = m_model->currentRow();
            initAmount();
            initAppendType();

            if (newitem) {
                // if (newitem) {
                m_model->item(m_index, REGISTER_COL_COUNT_TYPE_STR)->setText("1");
                resetAmount();
                handleAmount(barcode.mid(7, 5));
                return true;
            }
            //            emit addProduct(id, false);

            return true;
        }
    } else if (barcode.startsWith("28") || barcode.startsWith("29")) {
        QString partnumber = barcode.left(7);
        m_useStoredDecimals = true;
        int id = Database::getProductIdByBarcode(partnumber);
        if (id > 0) {
            m_discount = false;
            int gross = 0;
            QJsonObject obj = Database::getProductById(id);
            int currentGross = obj.value("gross").toDouble() * 100;
            bool newitem = true;

            QList<QStandardItem *> items
                = m_model->findItems(Database::getProductNameById(id), Qt::MatchExactly, REGISTER_COL_PRODUCT);
            foreach (QStandardItem *item, items) {
                newitem = false;
                gross = m_model->item(item->row(), REGISTER_COL_SINGLE)->text().toDouble() * 100;
                m_index = item->row();
                if (gross == currentGross) break;
            }

            //            int currentWeight = barcode.mid(7, 5).toInt();
            //            int weight = m_model->item(m_index, REGISTER_COL_COUNT)->text().toDouble() * 100;
            m_model->setCurrentRow(m_index);
            bool group = settings.value("BarCodesPlugin/Ean13Type28_29_group", true).toBool();

            // && m_model->item(m_index, REGISTER_COL_PRODUCT)->text() != ""
            if (m_model->item(m_index, REGISTER_COL_PRODUCT)->text() == "" || newitem) {
                emit addProduct(id, false);
                m_index = m_model->currentRow();
                m_model->item(m_index, REGISTER_COL_COUNT)->setText("0");
            } else if (!group) {
                emit addProduct(id, true);
                m_index = m_model->currentRow();
                m_model->item(m_index, REGISTER_COL_COUNT)->setText("0");
            }

            initAmount();
            initAppendType();

            //            if (m_model->item(m_index, REGISTER_COL_PRODUCT)->text() == "") {
            // if (newitem) {
            //            m_model->item(m_index, REGISTER_COL_COUNT_TYPE_STR)->setText("0");
            resetAmount();
            handleAmount(barcode.mid(7, 5));
            return true;
            //            }
            //            emit addProduct(id, false);

            //            return true;
        }
    }

    return false;
}

bool Barcodes::isActivated()
{
    QrkSettings settings;
    return ProFeatures::isValid() && settings.value("BarCodesPlugin/barcode_enabled", false).toBool();
}
