/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "cashbookdialog.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/acl.h"
#include "3rdparty/ckvsoft/rbac/crypto.h"
#include "3rdparty/profeatures/profeatures.h"
#include "cashbookdelegate.h"
#include "cashbookexportdialog.h"
#include "cashbookinoutdialog.h"
#include "cashbooksettingswidget.h"
#include "cashbookstornodialog.h"
#include "defines.h"
#include "qrkjournal.h"
#include "qrksettings.h"
#include "qrktimedmessagebox.h"

#include "ui_cashbookdialog.h"

#include "database.h"

#include <QLocale>
#include <QMap>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlTableModel>
#include <QTimer>

#include <QDebug>

CashBookDialog::CashBookDialog(QDialog *parent)
    : QDialog(parent)
    , ui(new Ui::CashBookDialog)
{
    ui->setupUi(this);

    ui->tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    ui->tableView->setSelectionMode(QAbstractItemView::SingleSelection);
    ui->tableView->setEditTriggers(QAbstractItemView::NoEditTriggers);

    QRegExp rx("^[0-9]*$");
    QValidator *intVal = new QRegExpValidator(rx, this);
    ui->searchLineEdit->setValidator(intVal);

    /* comming soon */
    ui->exportPushButton->setVisible(false);
    // ui->exportPushButton->setEnabled(true);
    connect(ui->exportPushButton, &QPushButton::clicked, this, [this]() {
        CashBookExportDialog dialog(m_model);
        dialog.exec();
    });
    /****************/

    connect(ui->closePushButton, &QrkPushButton::clicked, this, &CashBookDialog::close);
    connect(ui->inoutPushButton, &QrkPushButton::clicked, this, &CashBookDialog::insertNew);
    connect(ui->fromDateEdit, &QDateTimeEdit::dateTimeChanged, this, &CashBookDialog::setupView);
    connect(ui->toDateEdit, &QDateTimeEdit::dateTimeChanged, this, &CashBookDialog::setupView);
    connect(ui->searchLineEdit, &QLineEdit::textChanged, this, &CashBookDialog::setupView);
    connect(ui->stornoPushButton, &QPushButton::clicked, this, &CashBookDialog::onStornoButtonClicked);

    m_timer = new QTimer(this);
    connect(m_timer, &QTimer::timeout, this, &CashBookDialog::timerDone);
    m_timer->start(1000);

    if (RBAC::Instance()->hasPermission("plugin_cashbook_access_editbox", true)) ui->inoutPushButton->setEnabled(true);
    readSettings();
    init();
}

CashBookDialog::~CashBookDialog()
{
    m_timer->stop();
    writeSettings();
    delete ui;
}

void CashBookDialog::timerDone()
{

    if (QDate::currentDate().toString(Qt::DateFormat::SystemLocaleShortDate).compare(ui->datumsLabel->text()) != 0)
        setupView();
}

void CashBookDialog::onStornoButtonClicked()
{
    QModelIndexList indexList = ui->tableView->selectionModel()->selectedIndexes();
    qlonglong row = 0;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }
    qlonglong id = m_model->data(m_model->index(row, m_model->fieldIndex("id"), QModelIndex())).toLongLong();
    if (id > 0) {
        CashBookStornoDialog storno(id);
        if (storno.exec() == QDialog::Accepted) {
            setupView();
        }
    }
}

void CashBookDialog::onSelectionChanged(const QItemSelection &, const QItemSelection &)
{
    QModelIndexList indexList = ui->tableView->selectionModel()->selectedIndexes();
    qlonglong row = 0;
    foreach (QModelIndex index, indexList) {
        row = index.row();
    }

    QString storniert;
    qlonglong deleted = m_model->data(m_model->index(row, m_model->fieldIndex("deleted"), QModelIndex())).toLongLong();
    if (deleted > 0) {
#if (QT_VERSION >= QT_VERSION_CHECK(5, 8, 0))
        storniert = tr("Storniert: %1\n").arg(QLocale().toString(QDateTime::fromSecsSinceEpoch(deleted)));
#else
        storniert = tr("Storniert: %1\n").arg(QLocale().toString(QDateTime::fromMSecsSinceEpoch(deleted * 1000)));
#endif
    }

    QString description
        = tr("Beschreibung: %1\n")
              .arg(m_model->data(m_model->index(row, m_model->fieldIndex("description"), QModelIndex())).toString());
    QString reference
        = tr("Referenz: %1\n")
              .arg(m_model->data(m_model->index(row, m_model->fieldIndex("reference"), QModelIndex())).toString());

    ui->textBrowser->setText(QString("%1%2%3").arg(storniert).arg(reference).arg(description));

    if (deleted > 0) {
        ui->stornoPushButton->setEnabled(false);
        return;
    }

    QDateTime date = m_model->data(m_model->index(row, m_model->fieldIndex("timestamp"), QModelIndex())).toDateTime();
    bool today
        = date.date().toString(Qt::ISODate).compare(QDateTime::currentDateTime().date().toString(Qt::ISODate)) == 0;
    bool enabled = today
        && m_model->data(m_model->index(row, m_model->fieldIndex("type"), QModelIndex())).toLongLong()
            > TYPE_BUSINESS_AUTO;
    ui->stornoPushButton->setEnabled(enabled);
}

QString CashBookDialog::getLastEntryDate()
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    QVariant value = 0;
    QString strValue = "";
    AbstractDataBase::select_globals("cashbook_begin", value, strValue);
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT timestamp FROM cashbook ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) return query.value("timestamp").toDateTime().toString(Qt::ISODate);

    return strValue;
}

void CashBookDialog::insertNew()
{
    QString d = getLastEntryDate();
    CashBookInOutDialog inoutDialog;
    inoutDialog.setMinimumDateTime(d);
    if (inoutDialog.exec() == QDialog::Accepted) {
        ui->toDateEdit->setDateTime(QDateTime::currentDateTime());
        setUiData(); // <-- Refresh table
    }
}

void CashBookDialog::init()
{
    QVariant value = 0;
    QString strValue = "";

    AbstractDataBase::select_globals("currency", value, strValue);
    m_currency = strValue;
    value = 0;
    strValue = "";

    AbstractDataBase::select_globals("cashbook_begin", value, strValue);
    if (strValue.isEmpty()) {
        m_startEntry = QDateTime::currentDateTime().toString(Qt::ISODate);
    } else {
        m_startEntry = strValue;
        m_cash = QString::number(value.toLongLong());
    }

    if (!checkDatabase()) {
        if (dataBaseExists()) {
            qDebug() << "Function Name: " << Q_FUNC_INFO << "reject";
            QrkTimedMessageBox messageBox(20, QMessageBox::Critical, tr("Kassabuch Fehler"),
                tr("Fehler in der  Kassabuch Datenbank. Checksum Error."), QMessageBox::Yes | QMessageBox::Default);

            messageBox.setDetailedText(tr(
                "Mögliche Ursachen.\nDaten wurden direkt in der Datenbank gelöscht.\nDatensätze wurden direkt in der Datenbank geändert.\n\nSie können die Datenbank aus einen aktuellen Backup wiederherstellen."));

            messageBox.setDefaultButton(QMessageBox::Yes);
            messageBox.setButtonText(QMessageBox::Yes, tr("OK"));
            foreach (QAbstractButton *button, messageBox.buttons()) {
                if (messageBox.buttonRole(button) == QrkTimedMessageBox::ActionRole) {
                    button->click();
                    break;
                }
            }
            messageBox.exec();
        }
        QTimer::singleShot(10, this, &CashBookDialog::reject);
        return;
    }
    setupView();
    setUiData();
}

void CashBookDialog::setUiData()
{
    ui->toDateEdit->setDateTime(QDateTime::currentDateTime());

    QDateTime start = QDateTime::fromString(m_startEntry, "yyyy-MM-ddThh:mm:ss");
    QString text = tr("Anfangsbestand: %1 %2 - %3")
                       .arg(QLocale().toString(m_cash.toDouble() / 100, 'f', 2))
                       .arg(m_currency)
                       .arg(QLocale().toString(start));
    ui->cashStocBeginLabel->setText(text);
    ui->fromDateEdit->setMinimumDateTime(start);
    ui->toDateEdit->setMinimumDateTime(start);
    ui->fromDateEdit->setMaximumDateTime(ui->toDateEdit->dateTime());
}

void CashBookDialog::setupView()
{

    ui->fromDateEdit->setMaximumDateTime(ui->toDateEdit->dateTime());

    m_model = new QSqlTableModel(this, AbstractDataBase::database("CASHBOOK"));
    m_model->setTable("cashbook");
    //    m_model->setQuery("SELECT id, flow, CASE WHEN flow = 0 THEN gross ELSE '' END flowin, CASE WHEN flow = 1 THEN
    //    gross ELSE '' END flowout FROM cashbook;");
    if (ui->searchLineEdit->text().isEmpty()) {
        m_model->setFilter(QString("timestamp BETWEEN '%1' AND '%2'")
                .arg(ui->fromDateEdit->date().toString(Qt::ISODate))
                .arg(ui->toDateEdit->dateTime().toString(Qt::ISODate)));
    } else {
        m_model->setFilter(QString("id=%1 AND timestamp BETWEEN '%2' AND '%3'")
                .arg(ui->searchLineEdit->text().toLongLong())
                .arg(ui->fromDateEdit->date().toString(Qt::ISODate))
                .arg(ui->toDateEdit->dateTime().toString(Qt::ISODate)));
    }
    m_model->select();
    while (m_model->canFetchMore())
        m_model->fetchMore();

    ui->tableView->setModel(m_model);

    m_model->setHeaderData(m_model->fieldIndex("id"), Qt::Horizontal, tr("Nr."), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("timestamp"), Qt::Horizontal, tr("Datum"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("flow"), Qt::Horizontal, tr("Ein/Ausgang"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("gross"), Qt::Horizontal, tr("Betrag"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("type"), Qt::Horizontal, tr("Verwendung"), Qt::DisplayRole);
    m_model->setHeaderData(m_model->fieldIndex("storno"), Qt::Horizontal, "", Qt::DisplayRole);

    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("timestamp"), new CashBookDelegate(CashBookDelegate::CASHBOOK_COLUMN::DATETIME));
    ui->tableView->setItemDelegateForColumn(m_model->fieldIndex("gross"),
        new CashBookDelegate(CashBookDelegate::CASHBOOK_COLUMN::CURRENCY, m_model->fieldIndex("deleted")));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("type"), new CashBookDelegate(CashBookDelegate::CASHBOOK_COLUMN::TYPE));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("flow"), new CashBookDelegate(CashBookDelegate::CASHBOOK_COLUMN::FLOW));
    ui->tableView->setItemDelegateForColumn(
        m_model->fieldIndex("storno"), new CashBookDelegate(CashBookDelegate::CASHBOOK_COLUMN::STORNO));


    ui->tableView->setColumnHidden(m_model->fieldIndex("description"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("reference"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("deleted"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("checksum"), true);
    ui->tableView->setColumnHidden(m_model->fieldIndex("userid"), true);

    ui->currentStatusLabel->setText(ProFeatures::getCheckoutContents());
    ui->tableView->horizontalHeader()->resizeSections(QHeaderView::ResizeToContents);
    ui->tableView->sortByColumn(m_model->fieldIndex("id"), Qt::DescendingOrder);
    ui->tableView->horizontalHeader()->setStretchLastSection(true);

    connect(ui->tableView->selectionModel(), &QItemSelectionModel::selectionChanged, this,
        &CashBookDialog::onSelectionChanged);
    ui->tableView->selectRow(0);

    ui->datumsLabel->setText(QDate::currentDate().toString(Qt::DateFormat::SystemLocaleShortDate));
    ui->saldoVortragLabel->setText(
        QString::number(double(getBalanceCarriedForward(QDateTime::currentDateTime())) / 100, 'f', 2) + " "
        + Database::getShortCurrency());
}

bool CashBookDialog::checkDatabase()
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    if (!dataBaseExists()) {
        QDialog dialog;
        QVBoxLayout *layout = new QVBoxLayout;
        CashBookSettingsWidget *sw = new CashBookSettingsWidget;
        layout->addWidget(sw);
        dialog.setLayout(layout);
        connect(sw, &CashBookSettingsWidget::saved, &dialog, &QDialog::accept);
        connect(sw, &CashBookSettingsWidget::rejected, &dialog, &QDialog::reject);
        if (dialog.exec() != QDialog::Accepted) return false;
        QrkJournal journal;
        journal.journalInsertLine("Kassabuch", "Kassabuch Datenbanktabelle angelegt.");
        initCashbookStartDate();
    }

    return updateDatabase();
}

QString CashBookDialog::getLastCheckSum()
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    QString decrypted = "";
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT checksum FROM cashbook where deleted = 0 ORDER BY id DESC LIMIT 1");
    query.exec();
    if (query.next()) {
        SecureByteArray saName = query.value("checksum").toByteArray();
        decrypted = Crypto::decrypt(saName, SecureByteArray("CashBook"), false);
    }
    if (decrypted.isEmpty()) return ProFeatures::getCheckoutContents(false);
    QBCMath cash(decrypted.toLongLong());
    return cash.getIntPart();
}

void CashBookDialog::setSales(QDateTime date)
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    date.setTime(QTime(0, 0, 0));
    QDateTime toDate = date;
    toDate.setTime(QTime(23, 23, 59));
    query.prepare(
        "SELECT flow, gross FROM cashbook WHERE storno < 2 AND timestamp BETWEEN :timestampfrom AND :timestampto");
    query.bindValue(":timestampfrom", date.toString(Qt::ISODate));
    query.bindValue(":timestampto", toDate.toString(Qt::ISODate));
    query.exec();
    QBCMath in, out;
    while (query.next()) {
        if (query.value("flow").toLongLong() == FLOW_IN)
            in += query.value("gross").toLongLong();
        else
            out += query.value("gross").toLongLong();
    }
    in /= 100;
    out /= 100;
    in.round(2);
    out.round(2);
    ui->flowInLabel->setText(in.toLocale() + " " + Database::getShortCurrency());
    ui->flowOutLabel->setText(out.toLocale() + " " + Database::getShortCurrency());
    ui->saldoDateLabel->setText(
        tr("Saldovortrag %1").arg(toDate.addDays(-1).date().toString(Qt::DateFormat::SystemLocaleShortDate)));
}

qlonglong CashBookDialog::getBalanceCarriedForward(QDateTime date)
{
    setSales(date);
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    QString decrypted = "";
    CSqlQuery query(dbc, Q_FUNC_INFO);
    date.setTime(QTime(0, 0, 0));
    query.prepare("SELECT checksum FROM cashbook WHERE timestamp<:timestamp ORDER BY id DESC LIMIT 1");
    query.bindValue(":timestamp", date.toString(Qt::ISODate));
    query.exec();
    if (query.next()) {
        SecureByteArray saName = query.value("checksum").toByteArray();
        decrypted = Crypto::decrypt(saName, SecureByteArray("CashBook"), false);
    } else {
        decrypted = m_startEntry.toLongLong();
    }

    QBCMath cash(decrypted.toLongLong());
    return cash.getIntPart().toLongLong();
}

void CashBookDialog::testChecksum()
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    if (!dbc.tables(QSql::AllTables).contains(QLatin1String("cashbook"))) return;

    QBCMath g = 20000;

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id, gross, checksum FROM cashbook where deleted = 0");
    query.exec();
    while (query.next()) {
        SecureByteArray saName = query.value("checksum").toByteArray();
        QString decrypted = Crypto::decrypt(saName, SecureByteArray("CashBook"), false);
        qDebug() << "id: " << query.value("id").toInt() << "gross: " << query.value("gross").toLongLong()
                 << "checksum: " << query.value("checksum").toString() << "decrypted: " << decrypted;
        g += query.value("gross").toLongLong();
        QBCMath c = decrypted;
        if (g.getIntPart() != c.getIntPart()) return;
    }
}

bool CashBookDialog::updateDatabase()
{

    // testChecksum();
    QrkSettings settings;
    bool cash_repayment = settings.value("cashrepayment", false).toBool();
    QVariant value = 0;
    QString strValue = "";
    QString startEntry;

    AbstractDataBase::select_globals("cashbook_begin", value, strValue);
    if (strValue.isEmpty()) {
        startEntry = QDateTime::currentDateTime().toString(Qt::ISODate);
    } else {
        startEntry = strValue;
    }

    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    if (!dbc.tables(QSql::AllTables).contains(QLatin1String("cashbook"))) return false;

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT timestamp FROM cashbook ORDER BY id DESC LIMIT 1");
    query.exec();
    QString lastEntry = "";
    if (query.next()) lastEntry = query.value("timestamp").toDateTime().toString(Qt::ISODate);
    if (lastEntry.isEmpty()) lastEntry = startEntry;

    QString payedByValues;
    if (cash_repayment) {
        payedByValues = QString("%1, %2, %3").arg(PAYED_BY_CASH).arg(PAYED_BY_DEBITCARD).arg(PAYED_BY_CREDITCARD);
    } else {
        payedByValues = QString::number(PAYED_BY_CASH);
    }
    if (dbc.driverName().compare("QMYSQL") == 0) {
        query.prepare(QString(
            "SELECT timestamp, receiptNum, gross, payedBy, storno, stornoId, userId FROM receipts WHERE DATE_FORMAT(timestamp, '%Y-%m-%d %H:%M:%S') > "
            "DATE_FORMAT(:timestamp, '%Y-%m-%d %H:%M:%S') "
            "AND payedBy IN (%1);")
                .arg(payedByValues));
    } else {
        query.prepare(QString(
            "SELECT timestamp, receiptNum, gross, payedBy, storno, stornoId, userId FROM receipts WHERE datetime(timestamp) > "
            "datetime(:timestamp) "
            "AND payedBy IN (%1);")
                .arg(payedByValues));
    }

    query.bindValue(":timestamp", QDateTime::fromString(lastEntry, Qt::ISODate));

    if (query.exec()) {
        CSqlQuery updateQuery(dbc, Q_FUNC_INFO);
        updateQuery.prepare(
            "INSERT INTO cashbook (timestamp, flow, type, gross, storno, reference, checksum, description, userid) "
            "VALUES "
            "(:timestamp, :flow, :type, :gross, :storno, :reference, :checksum, :description, :userid);");
        bool success = true;
        qlonglong checksumvalue = getLastCheckSum().toLongLong();
        QString journalText;
        QStringList journalList;

        while (query.next() && success) {

            qlonglong payedBy = query.value("payedBy").toLongLong();
            bool flow_in = !(query.value("gross").toLongLong() < 0);
            qlonglong receiptNum = query.value("receiptNum").toLongLong();
            qlonglong original = receiptNum;
            qlonglong stornoId = query.value("stornoId").toLongLong();
            qlonglong storno = query.value("storno").toLongLong();
            checksumvalue = getLastCheckSum().toLongLong();
            if (payedBy != PAYED_BY_CASH && storno < 2) continue;

            QMap<int, double> givenMap = Database::getGiven(receiptNum);

            if (storno == 2) {
                receiptNum = query.value("stornoId").toLongLong();
                cash_repayment = true;
            }
            // if ((storno == 0 || !m_cash_repayment) && givenMap.size() > 0) {
            if (cash_repayment && givenMap.size() > 1) {
                QBCMath iGross(givenMap.value(PAYED_BY_CASH) * 100);
                iGross.round(0);
                if (!flow_in) iGross *= -1;
                updateQuery.bindValue(":gross", iGross.getIntPart().toLongLong());
                checksumvalue += iGross.toLongLong();

                if (iGross < 0) flow_in = false;
                double given = givenMap.value(PAYED_BY_CASH);
                double secondPay = givenMap.take(PAYED_BY_DEBITCARD);
                if (secondPay == 0.0) secondPay = givenMap.take(PAYED_BY_CREDITCARD);

                QString givenText
                    = tr("BAR: %1 %2").arg(QLocale().toString(given, 'f', 2)).arg(Database::getCurrency());
                QString payedByText, secondText;
                if (secondPay > 0.0) {
                    // if (!flow_in) secondPay *= -1;
                    secondText = tr("%1: %2 %3")
                                     .arg(givenMap.lastKey() == PAYED_BY_DEBITCARD ? tr("Bankomat") : tr("Kreditkarte"))
                                     .arg(QLocale().toString(secondPay, 'f', 2))
                                     .arg(Database::getCurrency());
                    payedByText = tr("Mischzahlung %1 %2 %3 %4")
                                      .arg(QBCMath::bcround((iGross / 100).toLocale(), 2))
                                      .arg(Database::getCurrency())
                                      .arg(givenText)
                                      .arg(secondText);
                }
                if (original != receiptNum) {
                    updateQuery.bindValue(":description",
                        tr("Ausgang - Kassa Stornobeleg (%1) für Kassa Eingangsbeleg (%2) / %3")
                            .arg(original)
                            .arg(receiptNum)
                            .arg(payedByText));
                } else {
                    if (storno == 2) {
                        updateQuery.bindValue(":description",
                            tr("Ausgang - Kassa Stornobeleg (%1) / %2 für Eingangsbleg (%2)")
                                .arg(receiptNum)
                                .arg(payedByText)
                                .arg(stornoId));
                        journalText = tr("Ausgang - Kassa Stornobeleg (%1) / %2 für Eingangsbeleg (%2)")
                                          .arg(receiptNum)
                                          .arg(payedByText)
                                          .arg(stornoId);

                    } else {
                        updateQuery.bindValue(":description",
                            tr("Eingang - Kassa Beleg Nummer %1 / %2").arg(receiptNum).arg(payedByText));
                        journalText = tr("Eingang - Kassa Beleg Nummer %1 / %2").arg(receiptNum).arg(payedByText);
                    }
                }

            } else {
                if (storno == 2) {
                    QBCMath iGross(query.value("gross").toDouble() * 100);
                    iGross.round(0);
                    checksumvalue += iGross.getIntPart().toLongLong();
                    updateQuery.bindValue(":gross", iGross.getIntPart().toLongLong());
                    updateQuery.bindValue(":description",
                        tr("Ausgang - Kassa Stornobeleg (%1) für Eingangsbeleg (%2)").arg(original).arg(stornoId));
                    journalText
                        = tr("Ausgang - Kassa Stornobeleg (%1) für Eingangsbeleg (%2)").arg(original).arg(stornoId);
                } else {
                    QBCMath iGross(query.value("gross").toDouble() * 100);
                    iGross.round(0);
                    checksumvalue += iGross.toLongLong();
                    updateQuery.bindValue(":gross", iGross.getIntPart().toLongLong());
                    updateQuery.bindValue(":description", tr("Eingang - Kassa Beleg Nummer %1").arg(receiptNum));
                    journalText = tr("Eingang - Kassa Beleg Nummer %1").arg(receiptNum);
                }
            }

            updateQuery.bindValue(":timestamp", query.value("timestamp").toDateTime().toString(Qt::ISODate));
            updateQuery.bindValue(":storno", (storno != 2) ? 0 : 1);
            updateQuery.bindValue(":flow", (flow_in) ? CB_FLOW::FLOW_IN : CB_FLOW::FLOW_OUT);
            updateQuery.bindValue(":type", CB_TYPE::TYPE_BUSINESS_AUTO);
            updateQuery.bindValue(":reference", tr("Kassa Beleg %1").arg(original));
            updateQuery.bindValue(":userid", query.value("userId").toLongLong());
            updateQuery.bindValue(":checksum",
                Crypto::encrypt(
                    SecureByteArray(QString::number(checksumvalue).toLatin1()), SecureByteArray("CashBook")));

            journalText += tr("Checksum - %1")
                               .arg(Crypto::encrypt(SecureByteArray(QString::number(checksumvalue).toUtf8()),
                                   SecureByteArray("CashBook")));

            success = updateQuery.exec();
            journalList.append(journalText);
        }

        qlonglong checkoutcontent = int(ProFeatures::getCheckoutContents(false).toLongLong());
        if (checksumvalue != checkoutcontent) {
            qDebug() << "Function Name: " << Q_FUNC_INFO << "checksumvalue: " << checksumvalue
                     << "checkoutcontent: " << checkoutcontent;
            return false;
        }

        QrkJournal journal;
        for (const auto &i : journalList) {
            journal.journalInsertLine("Kassabuch", i);
        }
        journal.journalInsertLine("Kassabuch", tr("Kassastand - %1").arg(ProFeatures::getCheckoutContents()));

        return success;
    }

    return false;
}

void CashBookDialog::initCashbookStartDate()
{
    QVariant value = "";
    QString strValue = "";
    AbstractDataBase::select_globals("cashbook_begin", value, strValue);
    m_startEntry = strValue;
    m_cash = QString::number(value.toLongLong());
}

void CashBookDialog::readSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    restoreGeometry(settings.value("WindowGeometry").toByteArray());
    settings.endGroup();

    //    m_cash_repayment = settings.value("cashrepayment", false).toBool();
}

void CashBookDialog::writeSettings()
{
    QrkSettings settings;
    settings.beginGroup("CashBook");
    settings.save2Settings("WindowGeometry", saveGeometry());
    settings.endGroup();
}

bool CashBookDialog::dataBaseExists()
{
    QSqlDatabase dbc = Database::database("CASHBOOK");
    return dbc.tables(QSql::AllTables).contains(QLatin1String("cashbook"));
}
