/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "aclmanager.h"
#include "acl.h"
#include "aclwizard.h"
#include "permissionsadmin.h"
#include "preferences/settings.h"
#include "rolesadmin.h"
#include "useradmin.h"
#include "westtabwidget.h"

#include <QApplication>
#include <QBoxLayout>
#include <QDesktopWidget>
#include <QTabWidget>

AclManager::AclManager(ACSInterface *acs, QWidget *parent)
    : QDialog(parent, Qt::CustomizeWindowHint | Qt::WindowTitleHint)
{
    m_tabWidget = new WestTabWidget(this);
    m_rolesadmin = new RolesAdmin(m_tabWidget);
    m_useradmin = new UserAdmin(acs, m_tabWidget);
    m_permissionsadmin = new PermissionsAdmin(m_tabWidget);

    m_tabWidget->tabBar()->setIconSize(QSize(32, 32));
    m_tabWidget->addTab(m_useradmin, QPixmap(":/ckvsoft/resources/icons/user.png"), tr("Benutzer"));
    m_tabWidget->addTab(m_rolesadmin, QPixmap(":/ckvsoft/resources/icons/role.png"), tr("Rollen"));
    m_tabWidget->addTab(m_permissionsadmin, QPixmap(":/ckvsoft/resources/icons/permission.png"), tr("Berechtigungen"));

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(m_tabWidget, 1);
    mainLayout->addSpacing(12);
    setLayout(mainLayout);

    setWindowTitle(tr("Benutzer Manager"));
    resize(820, 500);
    setMinimumWidth(750);
    setMinimumHeight(400);

    if (QApplication::desktop()->height() < 650) {
        setMinimumWidth(700);
        setFixedHeight(550);
    }

    connect(m_useradmin, &UserAdmin::exitButtonClicked, m_useradmin, &UserAdmin::maybeSaved);
    connect(m_useradmin, &UserAdmin::exitButtonClicked, this, &AclManager::closeAclManager);
    connect(m_useradmin, &UserAdmin::closeAclManager, this, &AclManager::closeAclManager);
    connect(m_rolesadmin, &RolesAdmin::exitButtonClicked, m_rolesadmin, &RolesAdmin::maybeSaved);
    connect(m_rolesadmin, &RolesAdmin::exitButtonClicked, this, &AclManager::closeAclManager);
    connect(
        m_permissionsadmin, &PermissionsAdmin::exitButtonClicked, m_permissionsadmin, &PermissionsAdmin::maybeSaved);
    connect(m_permissionsadmin, &PermissionsAdmin::exitButtonClicked, this, &AclManager::closeAclManager);

    connect(m_useradmin, &UserAdmin::createRoles, this, &AclManager::createRoles);
    connect(m_tabWidget, &QTabWidget::currentChanged, m_useradmin, &UserAdmin::maybeSaved);
    connect(m_tabWidget, &QTabWidget::currentChanged, m_rolesadmin, &RolesAdmin::maybeSaved);
    connect(m_tabWidget, &QTabWidget::currentChanged, m_permissionsadmin, &PermissionsAdmin::maybeSaved);

    readSettings();
}

AclManager::~AclManager()
{
    writeSettings();
}

void AclManager::closeAclManager()
{
    close();
}

void AclManager::createRoles()
{
    m_tabWidget->setCurrentWidget(m_rolesadmin);
}

void AclManager::setACS(ACSInterface *acs)
{
    if (m_useradmin) m_useradmin->setACS(acs);
}

void AclManager::readSettings()
{
    Settings settings;
    restoreGeometry(settings.value("usermanager/managerWindowGeometry").toByteArray());
}

void AclManager::writeSettings()
{
    Settings settings;
    settings.save2Settings("usermanager/managerWindowGeometry", saveGeometry());
}
