/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "qrkgastromanagerroomedit.h"
#include "3rdparty/ckvsoft/csqlquery.h"
#include "database.h"
#include "ui_qrkgastromanagerroomedit.h"
#include "utils/utils.h"

#include <QDebug>
#include <QMessageBox>
#include <QSqlError>

//--------------------------------------------------------------------------------

QRKGastroManagerRoomEdit::QRKGastroManagerRoomEdit(QWidget *parent, int id)
    : QDialog(parent)
    , ui(new Ui::QRKGastroManagerRoomEdit)
    , m_id(id)
{
    ui->setupUi(this);

    const QStringList colorNames = QColor::colorNames();
    int index = 0;
    ui->colorComboBox->addItem(tr("Standard Farbe"));
    const QModelIndex idx = ui->colorComboBox->model()->index(index++, 0);
    ui->colorComboBox->model()->setData(idx, "", Qt::BackgroundColorRole);
    ui->okButton->setDisabled(true);

    foreach (const QString &colorName, colorNames) {
        const QColor color(colorName);
        QString fg = Utils::color_best_contrast(color.name());
        const QColor fg_color(fg);
        ui->colorComboBox->addItem(colorName, color);
        const QModelIndex idx = ui->colorComboBox->model()->index(index++, 0);
        ui->colorComboBox->model()->setData(idx, color, Qt::BackgroundColorRole);
        ui->colorComboBox->model()->setData(idx, fg_color, Qt::ForegroundRole);
    }

    if (m_id != -1) {
        QSqlDatabase dbc = Database::database();

        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.exec(QString("SELECT name, color, isHotel FROM rooms WHERE id=%1").arg(id));
        query.next();

        ui->name->setText(query.value("name").toString());
        ui->okButton->setDisabled(ui->name->text().isEmpty());

        int i;
        for (i = 0; i <= ui->colorComboBox->count(); i++) {
            QString color = ui->colorComboBox->model()->index(i, 0).data(Qt::BackgroundColorRole).toString();
            if (query.value(2).toString() == color) break;
        }

        if (i > ui->colorComboBox->count()) i = 0;

        ui->colorComboBox->setCurrentIndex(i);
        QPalette palette(ui->colorComboBox->palette());
        QString colorValue = query.value("color").toString();
        if (!colorValue.isEmpty()) {
            QColor color(query.value("color").toString());
            palette.setColor(QPalette::Active, QPalette::Button, color);
            palette.setColor(QPalette::Highlight, color);
            //    palette.setColor(QPalette::ButtonText, Qt::white);
            ui->colorComboBox->setPalette(palette);
        }

        ui->isHotel->setChecked(query.value("isHotel").toBool());
    }

    connect(ui->colorComboBox, static_cast<void (QComboBox::*)(int)>(&QComboBox::currentIndexChanged), this,
        &QRKGastroManagerRoomEdit::colorComboChanged);
    connect(ui->okButton, &QPushButton::clicked, this, &QRKGastroManagerRoomEdit::accept);
    connect(ui->cancelButton, &QPushButton::clicked, this, &QRKGastroManagerRoomEdit::reject);
    connect(ui->name, &QLineEdit::textChanged, this, [this](const QString &newText) {
        ui->okButton->setDisabled(newText.isEmpty());
    });
}

QRKGastroManagerRoomEdit::~QRKGastroManagerRoomEdit()
{
    delete ui;
}

void QRKGastroManagerRoomEdit::colorComboChanged(int idx)
{

    QString colorValue = ui->colorComboBox->itemData(idx, Qt::BackgroundColorRole)
                             .toString(); // ->itemText(idx); //
                                          // ->model()->index(ui->colorComboBox->currentIndex(),
                                          // 0).data(Qt::BackgroundColorRole).toString();
    QPalette palette(ui->colorComboBox->palette());
    QColor color(colorValue);
    palette.setColor(QPalette::Active, QPalette::Button, color);
    palette.setColor(QPalette::Highlight, color);
    ui->colorComboBox->setPalette(palette);
}

//--------------------------------------------------------------------------------

void QRKGastroManagerRoomEdit::accept()
{

    if (m_id == -1 && existRoom(ui->name->text())) {
        QMessageBox::information(
            this, tr("Raumname"), tr("Der Name %1 ist schon in Verwendung.").arg(ui->name->text()), QMessageBox::Ok);
        return;
    }

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QString color = ui->colorComboBox->model()
                        ->index(ui->colorComboBox->currentIndex(), 0)
                        .data(Qt::BackgroundColorRole)
                        .toString();

    if (m_id == -1) { // new entry
        query.prepare(QString("INSERT INTO rooms (name, color, isHotel) "
                              "VALUES(:name, :color, :isHotel)"));
    } else {
        query.prepare(QString("UPDATE rooms SET name=:name, color=:color, "
                              "isHotel=:isHotel WHERE id=:id"));
        query.bindValue(":id", m_id);
    }

    query.bindValue(":name", ui->name->text());
    query.bindValue(":color", color);
    query.bindValue(":isHotel", ui->isHotel->isChecked());

    query.exec();
    QDialog::accept();
}

bool QRKGastroManagerRoomEdit::existRoom(const QString &roomname)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id FROM rooms WHERE name=:roomname");
    query.bindValue(":roomname", roomname);
    query.exec();
    if (query.next()) {
        return true;
    }
    return false;
}

//--------------------------------------------------------------------------------
