/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "cashbookstornodialog.h"
#include "ui_cashbookstornodialog.h"

#include "3rdparty/ckvsoft/csqlquery.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/crypto.h"
#include "database.h"

#include <QDateTime>
#include <QSqlError>

#include <QDebug>

CashBookStornoDialog::CashBookStornoDialog(int id, QWidget *parent)
    : QDialog(parent)
    , ui(new Ui::CashBookStornoDialog)
    , m_id(id)
{
    ui->setupUi(this);


    connect(ui->textEdit, &QTextEdit::textChanged, this, &CashBookStornoDialog::onTextChanged);
    connect(ui->okPushButton, &QPushButton::clicked, this, &CashBookStornoDialog::cancelation);
    connect(ui->cancelPushButton, &QPushButton::clicked, this, &CashBookStornoDialog::close);
}

CashBookStornoDialog::~CashBookStornoDialog()
{
    delete ui;
}

void CashBookStornoDialog::onTextChanged()
{
    ui->okPushButton->setDisabled(ui->textEdit->toPlainText().isEmpty());
}

void CashBookStornoDialog::cancelation()
{
    QSqlDatabase dbc = AbstractDataBase::database("CASHBOOK");
    QString decrypted = "";
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery updateQuery(dbc, Q_FUNC_INFO);
    if (dbc.driverName().compare("QSQLITE") == 0) {
        query.prepare(
            "UPDATE cashbook set storno=:storno, deleted=:timestamp, description=description || :description WHERE id=:id");
    } else {
        query.prepare(
            "UPDATE cashbook set storno=:storno, deleted=:timestamp, description=CONCAT(description,:description) WHERE id=:id");
    }
    query.bindValue(":id", m_id);
    query.bindValue(":storno", 2);
#if (QT_VERSION >= QT_VERSION_CHECK(5, 8, 0))
    query.bindValue(":timestamp", QDateTime::currentSecsSinceEpoch());
#else
    query.bindValue(":timestamp", QDateTime::currentMSecsSinceEpoch() / 1000);
#endif
    query.bindValue(":description", "\nStornogrund: " + ui->textEdit->toPlainText());
    updateQuery.prepare("UPDATE cashbook set checksum=:checksum WHERE id=:id");

    if (!query.exec()) {
        return;
    } else {
        query.prepare("SELECT id, gross, checksum FROM cashbook WHERE id>=:id");
        query.bindValue(":id", m_id);
        query.exec();
        QBCMath gross;
        while (query.next()) {
            int id = query.value("id").toInt();
            if (id == m_id) gross = query.value("gross").toInt();
            SecureByteArray crypted(query.value("checksum").toByteArray());
            QBCMath decrypted(Crypto::decrypt(crypted, SecureByteArray("CashBook")).toInt());
            decrypted -= gross;

            updateQuery.bindValue(":id", id);
            updateQuery.bindValue(
                ":checksum", Crypto::encrypt(decrypted.getIntPart().toLatin1(), SecureByteArray("CashBook")));
            if (!updateQuery.exec()) {
                return;
            }
        }
    }

    QDialog::accept();
}
