#include "pindialog.h"
#include "abstractdatabase.h"
#include "csqlquery.h"

#include <QDebug>
#include <QHBoxLayout>
#include <QSqlError>
#include <QSqlQuery>
#include <QVBoxLayout>

PinDialog::PinDialog(QWidget *parent)
    : QDialog(parent)
{
    setWindowTitle(tr("PIN Eingabe"));
    resize(300, 200);

    pinEdit1 = new QLineEdit(this);
    pinEdit2 = new QLineEdit(this);
    pinEdit1->setEchoMode(QLineEdit::Password);
    pinEdit2->setEchoMode(QLineEdit::Password);
    pinEdit1->setPlaceholderText(tr("PIN eingeben"));
    pinEdit2->setPlaceholderText(tr("PIN wiederholen"));

    infoLabel = new QLabel(tr("Bitte kontrollieren Sie die PIN sorgfältig!\n"
                              "Bei falscher Eingabe kann die Karte unwiderruflich gesperrt werden."),
        this);
    infoLabel->setWordWrap(true);

    errorLabel = new QLabel(this);
    errorLabel->setStyleSheet("color: red");
    errorLabel->hide();

    okButton = new QPushButton(tr("OK"), this);
    cancelButton = new QPushButton(tr("Abbrechen"), this);

    QHBoxLayout *buttonLayout = new QHBoxLayout;
    buttonLayout->addStretch();
    buttonLayout->addWidget(cancelButton);
    buttonLayout->addWidget(okButton);

    QVBoxLayout *mainLayout = new QVBoxLayout(this);
    mainLayout->addWidget(infoLabel);
    mainLayout->addWidget(pinEdit1);
    mainLayout->addWidget(pinEdit2);
    mainLayout->addWidget(errorLabel);
    mainLayout->addLayout(buttonLayout);
    setLayout(mainLayout);

    connect(okButton, &QPushButton::clicked, this, &PinDialog::onAcceptClicked);
    connect(cancelButton, &QPushButton::clicked, this, &PinDialog::reject);
}

void PinDialog::setCardSerial(const QString &serial)
{
    cardSerial = serial;
}

bool PinDialog::isSuccess() const
{
    return success;
}

void PinDialog::onAcceptClicked()
{
    QString pin1 = pinEdit1->text();
    QString pin2 = pinEdit2->text();

    if (pin1.isEmpty() || pin2.isEmpty()) {
        errorLabel->setText(tr("Bitte geben Sie die PIN zweimal ein."));
        errorLabel->show();
        return;
    }

    if (pin1 != pin2) {
        errorLabel->setText(tr("Die PINs stimmen nicht überein."));
        errorLabel->show();
        return;
    }

    if (!storePinInDatabase(pin1)) {
        errorLabel->setText(tr("Fehler beim Speichern der PIN."));
        errorLabel->show();
        return;
    }

    success = true;
    accept();
}

bool PinDialog::storePinInDatabase(const QString &pin)
{
    if (cardSerial.isEmpty()) {
        qWarning() << "Card serial is empty!";
        return false;
    }

    QSqlDatabase dbc = AbstractDataBase::database("PIN");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("UPDATE certs SET pin = :pin WHERE serial = :serial");
    query.bindValue(":pin", pin);
    query.bindValue(":serial", cardSerial);

    if (!query.exec()) {
        qWarning() << "Failed to update certs:" << query.lastError().text();
        return false;
    }

    if (query.numRowsAffected() == 0) {
        qWarning() << "No cert entry found for serial" << cardSerial;
        return false;
    }

    return true;
}
