#include "primesignonline.h"
#include <QJsonDocument>
#include <QJsonObject>
#include <QUrl>

PrimeSignOnline::PrimeSignOnline(QString connectionString)
{
    m_manager = new QNetworkAccessManager;
    QStringList parts = connectionString.split("@");
    if (parts.size() == 3) {
        m_userid = parts.at(0);
        m_sharedsecret = parts.at(1);
        m_url = parts.at(2);
    } else {
        qCritical() << "Invalid connection string, expected userid@sharedsecret@url";
    }
}

PrimeSignOnline::~PrimeSignOnline()
{
    // no session to close
}

QString PrimeSignOnline::getCardType()
{
    if (m_userid == QLatin1String("user123")) return QObject::tr("PrimeSign Online (Testzugang)");
    return QObject::tr("PrimeSign Online (Livezugang)");
}

bool PrimeSignOnline::selectApplication()
{
    return !(m_userid.isEmpty() || m_sharedsecret.isEmpty() || m_url.isEmpty());
}

QString PrimeSignOnline::getCertificate(bool /* base64 */)
{
    // if keyId not set, fetch it from config
    if (m_keyId.isEmpty()) {
        QUrl cfgUrl(m_url + "/rs/rk/config");
        QNetworkRequest cfgReq(cfgUrl);
        setAuthHeader(cfgReq);
        QJsonObject cfgObj;
        cfgObj.insert("request", "GET");
        if (!doRequest(cfgReq, cfgObj)) {
            qWarning() << "getCertificate failed (config):" << cfgObj.value("errorstring").toString();
            return QString();
        }
        QJsonObject user = cfgObj.value("user").toObject();
        m_keyId = user.value("defaultKey").toString();
        if (m_keyId.isEmpty()) {
            qWarning() << "getCertificate: defaultKey missing in config response";
            return QString();
        }
    }

    // now fetch the certificate JSON for that keyId
    QUrl certUrl(m_url + "/rs/keys/" + m_keyId + "/certificate");
    QNetworkRequest certReq(certUrl);
    setAuthHeader(certReq);
    QJsonObject certObj;
    certObj.insert("request", "GET");
    if (!doRequest(certReq, certObj)) {
        qWarning() << "getCertificate failed (certificate fetch):" << certObj.value("errorstring").toString();
        return QString();
    }

    // extract serial numbers directly from root object
    m_certificateSerial = certObj.value("serialNumber").toString();
    m_certificateSerialHex = certObj.value("serialNumberHex").toString();

    // put Certificate to Database
    if (!isCertificateInDB(m_certificateSerialHex)) {
        QByteArray pem = getRaw(QUrl(m_url + "/rs/keys/" + m_keyId + "/certificate.pem"));
        QString base64 = extractBase64FromPEM(pem);
        putCertificate(m_certificateSerialHex, base64);
    }

    // return hex serial by convention
    return m_certificateSerialHex;
}

QString PrimeSignOnline::getCertificateSerial(bool hex)
{
    if (m_certificateSerial.isEmpty()) getCertificate(); // Call getCertificate if serial is not yet fetched

    return hex ? m_certificateSerialHex : m_certificateSerial;
}

QString PrimeSignOnline::signReceipt(QString data)
{
    QString jwsDataToBeSigned = RKSignatureModule::getDataToBeSigned(data);

    QByteArray rawHash
        = RKSignatureModule::RawHashValue(data); // oder RawHash(jwsDataToBeSigned), je nach Spezifikation

    if (m_keyId.isEmpty()) {
        getCertificateSerial(true);
        if (m_keyId.isEmpty()) {
            qWarning() << "Kein keyId gefunden!";
            return QString();
        }
    }

    QString path = QString("/rs/rk/keys/%1/signatures/r1raw").arg(m_keyId);
    QUrl url(m_url + path);
    QNetworkRequest req(url);

    setAuthHeader(req);

    req.setHeader(QNetworkRequest::ContentTypeHeader, "text/plain;charset=UTF-8");
    req.setRawHeader("Accept", "text/plain");

    QNetworkReply *reply = m_manager->post(req, rawHash);

    QEventLoop loop;
    QObject::connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
    loop.exec();

    QByteArray response = reply->readAll();
    if (reply->error() != QNetworkReply::NoError) {
        qWarning() << "PrimeSign raw signature error:" << reply->errorString();
        reply->deleteLater();
        // Fallback-Signatur anhängen
        QString fallback = base64Url_encode("Sicherheitseinrichtung ausgefallen");
        return jwsDataToBeSigned + "." + fallback;
    }
    reply->deleteLater();

    QString jwsSignature = base64Url_encode(QString::fromUtf8(response));
    return jwsDataToBeSigned + "." + jwsSignature;
}

bool PrimeSignOnline::doRequest(QNetworkRequest &req, QJsonObject &obj)
{
    QString method = obj.value("request").toString("POST");
    QNetworkReply *reply = nullptr;
    if (method == QLatin1String("GET")) {
        reply = m_manager->get(req);
    } else {
        reply = m_manager->post(req, QJsonDocument(obj).toJson());
    }

    QEventLoop loop;
    QObject::connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
    loop.exec();

    if (reply->error() == QNetworkReply::NoError) {
        obj = QJsonDocument::fromJson(reply->readAll()).object();
        reply->deleteLater();
        return true;
    } else {
        obj.insert("errorstring", reply->errorString());
        reply->deleteLater();
        return false;
    }
}

QByteArray PrimeSignOnline::getRaw(const QUrl &url)
{
    QNetworkRequest req(url);
    setAuthHeader(req);
    QNetworkReply *reply = m_manager->get(req);

    QEventLoop loop;
    QObject::connect(reply, &QNetworkReply::finished, &loop, &QEventLoop::quit);
    loop.exec();

    QByteArray data;
    if (reply->error() == QNetworkReply::NoError) {
        data = reply->readAll();
    } else {
        qCritical() << "GET failed:" << reply->errorString();
    }

    reply->deleteLater();
    return data;
}

QString PrimeSignOnline::extractBase64FromPEM(const QByteArray &pem)
{
    QString pemStr = QString::fromUtf8(pem);
    QStringList lines = pemStr.split('\n');

    QStringList base64Lines;
    for (const QString &line : lines) {
        if (line.startsWith("-----")) continue; // BEGIN/END auslassen
        if (line.trimmed().isEmpty()) continue; // leere Zeilen auslassen
        base64Lines << line.trimmed();
    }

    return base64Lines.join("");
}

void PrimeSignOnline::setAuthHeader(QNetworkRequest &req)
{
    QByteArray cred = QString("%1:%2").arg(m_userid, m_sharedsecret).toUtf8().toBase64();
    req.setRawHeader("Authorization", "Basic " + cred);
}
