/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2025 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "database.h"
#include "3rdparty/ckvsoft/qbcmath/bcmath.h"
#include "3rdparty/ckvsoft/rbac/crypto.h"
#include "defines.h"
#ifdef QRK_PROFEATURES
#    include "3rdparty/profeatures/profeatures.h"
#endif
#include "backup.h"
#include "csqlquery.h"
#include "preferences/settings.h"
#include "qrkjournal.h"
#include "utils/demomode.h"
#include "utils/utils.h"

#include <QApplication>
#include <QDate>
#include <QDebug>
#include <QDir>
#include <QFile>
#include <QJsonArray>
#include <QJsonDocument>
#include <QJsonObject>
#include <QMessageBox>
#include <QRegularExpression>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlRecord>
#include <QStandardPaths>

Database::Database(QObject *parent)
    : AbstractDataBase(parent)
{
}

//--------------------------------------------------------------------------------

Database::~Database()
{
}

QDateTime Database::getLastJournalEntryDate()
{
    QDateTime lastDateTime;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT DateTime FROM journal WHERE id = (SELECT MAX(id) FROM journal)");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();
    lastDateTime = query.value("datetime").toDateTime();

    return lastDateTime;
}

QString Database::getDayCounter()
{
    QDateTime dateFrom(Database::getFromDateTime());
    QDateTime dateTo(QDateTime::currentDateTime());

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    /* Summe */
    query.prepare(QString("SELECT sum(gross) as total FROM receipts WHERE timestamp "
                          "BETWEEN :fromdate AND :todate AND payedBy < :payedby"));
    query.bindValue(":fromdate", dateFrom.toString(Qt::ISODate));
    query.bindValue(":todate", dateTo.toString(Qt::ISODate));
    query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();

    return QString::number(query.value("total").toDouble(), 'f', 2);
}

QString Database::getMonthCounter()
{
    QDateTime dateFrom;
    QDateTime dateTo(QDateTime::currentDateTime());

    dateFrom.setDate(QDate::fromString(
        QString("%1-%2-1").arg(QDate::currentDate().year()).arg(QDate::currentDate().month()), "yyyy-M-d"));
    dateFrom.setTime(Database::getCurfewTimeByDate(dateFrom.date()));

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    /* Summe */
    query.prepare(QString("SELECT sum(gross) as total FROM receipts WHERE timestamp "
                          "BETWEEN :fromdate AND :todate AND payedBy < :payedby"));
    query.bindValue(":fromdate", dateFrom.toString(Qt::ISODate));
    query.bindValue(":todate", dateTo.toString(Qt::ISODate));
    query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();

    return QString::number(query.value("total").toDouble(), 'f', 2);
}

QString Database::getYearCounter()
{
    QDateTime dateFrom;
    QDateTime dateTo(QDateTime::currentDateTime());

    dateFrom.setDate(QDate::fromString(QString("%1-1-1").arg(QDate::currentDate().year()), "yyyy-M-d"));
    dateFrom.setTime(Database::getCurfewTimeByDate(dateFrom.date()));

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    /* Summe */
    query.prepare(QString("SELECT sum(gross) as total FROM receipts WHERE timestamp "
                          "BETWEEN :fromdate AND :todate AND payedBy < :payedby"));
    query.bindValue(":fromdate", dateFrom.toString(Qt::ISODate));
    query.bindValue(":todate", dateTo.toString(Qt::ISODate));
    query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();

    return QString::number(query.value("total").toDouble(), 'f', 2);
}

QMap<QString, QMap<QString, double> > Database::getSalesPerUser(const QString &from, const QString &to, int &size)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    /* Umsätze Zahlungsmittel */
    query.prepare("SELECT actiontypes.actionText, receiptNum, gross, users.username, r2b, "
                  "actiontypes.actionId from receipts LEFT JOIN actiontypes on "
                  "receipts.payedBy=actiontypes.actionId LEFT JOIN users ON "
                  "receipts.userId=users.ID WHERE receipts.timestamp between :fromDate AND "
                  ":toDate AND receipts.payedBy < :payedby ORDER BY users.username, "
                  "receipts.payedBy");

    query.bindValue(":fromDate", from);
    query.bindValue(":toDate", to);
    query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }

    bool ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }

    /*
     * FIXME: We do this workaroud, while SUM and ROUND will
     * give use the false result. SQL ROUND/SUM give xx.98 from xx.985
     * should be xx.99
     */

    QMap<QString, double> zm, zmprivate;
    QMap<QString, QMap<QString, double> > user;
    QMap<QString, QBCMath> realTotalGross;
    while (query.next()) {
        int r2b = query.value("r2b").toInt();
        QString username = query.value("username").toString();
        QString realkey, key = query.value("actionText").toString();
        int keyId = query.value("actionId").toInt();
        if (r2b == 1) key += " R2B";

        int id = query.value("receiptNum").toInt();
        if (keyId < 0) {
            QMap<QString, QBCMath> realGross = Database::getProductRealGrossFromReceipt(id);
            QMap<QString, QBCMath>::iterator i;
            QString tmpKey;
            for (i = realGross.begin(); i != realGross.end(); ++i) {
                tmpKey = key + " (" + i.key() + "%";
                if (realkey.compare(tmpKey) != 0) realkey = tmpKey;

                if (zmprivate.contains(tmpKey)) {
                    zmprivate[tmpKey] += i.value().toDouble();
                } else {
                    zmprivate[tmpKey] = i.value().toDouble();
                }
                realTotalGross[i.key()] += i.value().toDouble();
            }
            continue;
        }

        QBCMath total(query.value("gross").toString());
        total.round(2);

        QMap<int, double> mixed = Database::getGiven(id);
        if (mixed.size() > 1) {
            mixed.take(Database::getActionTypeByName(key));

            QString type = Database::getActionType(mixed.lastKey());
            if (r2b == 1) type += " R2B";

            QBCMath secondPay(mixed.last());
            secondPay.round(2);
            total -= secondPay;
            if (zm.contains(type)) {
                zm[type] += secondPay.toDouble();
            } else {
                zm[type] = secondPay.toDouble();
            }
        }

        if (zm.contains(key)) {
            zm[key] += total.toDouble();
        } else {
            zm[key] = total.toDouble();
        }

        QMap<QString, double>::iterator i;
        for (i = zmprivate.begin(); i != zmprivate.end(); ++i) {
            QBCMath value(i.value());
            value.round(2);
            QString key = i.key() + " " + value.toLocale() + " " + Database::getShortCurrency() + ")";
            zm[key] = 0.0;
        }

        if (user.contains(username)) {
            QMap<QString, double> zm2 = user[username];
            QMap<QString, double>::iterator i;
            for (i = zm.begin(); i != zm.end(); ++i) {
                QString key = i.key();
                if (zm2.contains(key)) {
                    zm2[key] += i.value();
                } else {
                    zm2[key] = i.value();
                }
            }
            user[username] = zm2;
        } else {
            user[username] = zm;
        }
        zm.clear();
        zmprivate.clear();
    }
    size = zm.size();
    return user;
}

QString Database::getSalesPerPaymentSQLQueryString()
{
    return QString("SELECT actiontypes.actionText, orders.tax, (orders.count * "
                   "orders.gross) - ((orders.count * orders.gross / 100) * "
                   "orders.discount) as total from orders "
                   " LEFT JOIN receipts on orders.receiptId=receipts.receiptNum"
                   " LEFT JOIN actiontypes on receipts.payedBy=actiontypes.actionId"
                   " WHERE receipts.timestamp between :fromDate AND :toDate AND "
                   "receipts.payedBy < %1"
                   " ORDER BY receipts.payedBy, orders.tax")
        .arg(PAYED_BY_REPORT_EOD);
    /*
  return "SELECT actiontypes.actionText, orders.tax, SUM((orders.count *
  orders.gross) - ((orders.count * orders.gross / 100) * orders.discount)) as
  total from orders " " LEFT JOIN receipts on
  orders.receiptId=receipts.receiptNum" " LEFT JOIN actiontypes on
  receipts.payedBy=actiontypes.actionId" " WHERE receipts.timestamp between
  :fromDate AND :toDate AND receipts.payedBy < 3" " GROUP BY orders.tax,
  receipts.payedBy ORDER BY receipts.payedBy, orders.tax";
  */
    /*
  return "SELECT actiontypes.actionText, orders.tax, SUM((orders.count *
  orders.gross) - round(((orders.count * orders.gross / 100) *
  orders.discount),2)) as total from orders " " LEFT JOIN receipts on
  orders.receiptId=receipts.receiptNum" " LEFT JOIN actiontypes on
  receipts.payedBy=actiontypes.actionId" " WHERE receipts.timestamp between
  :fromDate AND :toDate AND receipts.payedBy < 3" " GROUP BY orders.tax,
  receipts.payedBy ORDER BY receipts.payedBy, orders.tax";
         */
}

//--------------------------------------------------------------------------------

void Database::updateSortorder(const QString &which, QList<int> indexList)
{
    if (which.isEmpty()) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    dbc.transaction();
    query.prepare(QString("UPDATE %1 SET sortorder=:sortorder WHERE id=:id").arg(which));

    bool ok = false;
    for (int i = 0; i < indexList.count(); i++) {
        query.bindValue(":sortorder", i);
        query.bindValue(":id", indexList.at(i));
        ok = query.exec();
        if (!ok) {
            break;
        }
    }
    if (ok)
        dbc.commit();
    else
        dbc.rollback();
}

void Database::updateProductPrinter(const int &printerId, const int &productId)
{
    //    if (product.startsWith("Zahlungsbeleg für Rechnung")) return;

    if (productId <= 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("UPDATE products SET printerid=:printerid WHERE id=:id"));

    if (printerId > 0)
        query.bindValue(":printerid", printerId);
    else
        query.bindValue(":printerid", QVariant(QVariant::Int));

    query.bindValue(":id", productId);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
}

void Database::updateProductSold(double count, const int &productId, bool updateStock)
{
    if (productId <= 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);


    query.prepare(QString("UPDATE products SET sold=sold+:sold, stock=stock-:stock WHERE id=:id"));
    query.bindValue(":sold", count);
    query.bindValue(":stock", updateStock ? count : 0.00);
    query.bindValue(":id", productId);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
}

void Database::updateProductPrice(const double &gross, const double &tax, const int &productId)
{
    //    if (product.startsWith("Zahlungsbeleg für Rechnung")) return;
    if (productId <= 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QBCMath net = gross / (1.0 + tax / 100.0);
    net.round(2);

    query.prepare(QString("UPDATE products SET gross=:gross, net=:net, lastchange=:lastchange WHERE id=:id"));
    query.bindValue(":gross", gross);
    query.bindValue(":net", net.toDouble());
    query.bindValue(":lastchange", QDateTime::currentDateTime());
    query.bindValue(":id", productId);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
}

void Database::updateProductTax(double tax, const int &productId)
{
    // if (product.startsWith("Zahlungsbeleg für Rechnung")) return;

    if (productId <= 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("UPDATE products SET tax=:tax, lastchange=:lastchange WHERE id=:id"));
    query.bindValue(":tax", tax);
    query.bindValue(":lastchange", QDateTime::currentDateTime());
    query.bindValue(":id", productId);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
}

void Database::insertProductItemnumToExistingProduct(const QString &itemnum, const int &productId)
{
    if (productId <= 0) return;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("UPDATE products SET itemnum=:itemnum WHERE id=:id"));
    query.bindValue(":itemnum", itemnum);
    query.bindValue(":id", productId);

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
}

bool Database::moveProductsToDefaultGroup(int oldgroupid)
{
    QSqlDatabase dbc = Database::database();
    dbc.transaction();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery movequery(dbc, Q_FUNC_INFO);
    movequery.prepare("UPDATE products SET groupid=2, visible=:visible WHERE groupid=:id");

    query.prepare(QString("SELECT id, name FROM products WHERE groupid=:id"));
    query.bindValue(":id", oldgroupid);
    query.exec();
    while (query.next()) {
        int id = Database::getProductIdByName(query.value("name").toString(), 2);
        if (id > 0) {
            movequery.bindValue(":visible", -1);
            movequery.bindValue(":id", id);
            movequery.exec();
            movequery.exec();
        }
    }

    query.prepare(QString("UPDATE products SET groupid=2 WHERE groupid=:id"));
    query.bindValue(":id", oldgroupid);
    query.exec();
    if (dbc.commit()) return true;

    dbc.rollback();
    return false;
}

bool Database::moveGroupsToDefaultCategory(int oldcategoryid)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("UPDATE groups SET categoryId=1 WHERE categoryId=:id"));
    query.bindValue(":id", oldcategoryid);

    return query.exec();
}

QJsonArray Database::findMissingProducts(int year)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("SELECT id from receipts where infodate like '%1%' ORDER by id ASC LIMIT 1").arg(year));
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();
    int idb = query.value("id").toInt();

    query.prepare(QString("SELECT id from receipts where infodate like '%1%' ORDER by id DESC LIMIT 1").arg(year));
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();
    int ide = query.value("id").toInt();

    if (idb < 1 || ide < 1) return {};

    query.prepare(QString(
        "SELECT product, count, net, gross, tax FROM orders WHERE receiptId BETWEEN :idb AND :ide GROUP BY product ORDER BY product"));
    query.bindValue(":idb", idb);
    query.bindValue(":ide", ide);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }

    QJsonArray products;
    while (query.next()) {
        if (Database::getProductById(query.value("product").toInt(), -1).isEmpty()) {
            QJsonObject data;
            data["name"] = tr("unbekannt mit id %1").arg(query.value("product").toInt());
            data["id"] = query.value("product").toInt();
            data["net"] = query.value("net").toDouble();
            data["gross"] = query.value("gross").toDouble();
            data["tax"] = query.value("tax").toDouble();
            products.append(data);
        }
    }

    qInfo() << "Function Name: " << Q_FUNC_INFO << " found missing products: " << products.size();
    return products;
}

QStringList Database::getStockInfoList()
{
    Settings settings;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("select name, stock, minstock from products inner join orders "
                  "on products.id=orders.product where orders.receiptId= (select "
                  "max(receipts.receiptNum) from receipts) and products.stock <= "
                  "products.minstock");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }

    int decimals = settings.value("decimalDigits", 2).toInt();
    QStringList list;
    QBCMath stock;
    QBCMath minstock;
    QString name;
    while (query.next()) {
        name = query.value("name").toString();
        if (name.startsWith("Zahlungsbeleg für Rechnung")) continue;

        stock = query.value("stock").toString();
        minstock = query.value("minstock").toString();

        list.append(QString("%1 (%2 / %3)")
                .arg(query.value("name").toString())
                .arg(QBCMath::bcround(stock.toString(), decimals))
                .arg(QBCMath::bcround(minstock.toString(), decimals)));
    }
    return list;
}

//--------------------------------------------------------------------------------

QString Database::getTaxLocation()
{
    if (globalStringValues.contains("taxlocation")) return globalStringValues.value("taxlocation");

    QVariant value;
    QString strValue;
    select_globals("taxlocation", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("taxlocation", strValue);
        return globalStringValues.value("taxlocation");
    }

    return Database::updateGlobals("taxlocation", Q_NULLPTR, "AT");
}

//--------------------------------------------------------------------------------

QString Database::getDefaultTax()
{
    if (globalStringValues.contains("defaulttax")) return globalStringValues.value("defaulttax");

    QVariant value;
    QString strValue;
    select_globals("defaulttax", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("defaulttax", strValue);
        return globalStringValues.value("defaulttax");
    }

    return Database::updateGlobals("defaulttax", Q_NULLPTR, "20");
}

QString Database::getAdvertisingText()
{
    // AdvertisingText
    if (globalStringValues.contains("printAdvertisingText")) return globalStringValues.value("printAdvertisingText");

    QVariant value;
    QString strValue;
    select_globals("printAdvertisingText", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("printAdvertisingText", strValue);
        return globalStringValues.value("printAdvertisingText");
    }

    return Database::updateGlobals("printAdvertisingText", Q_NULLPTR, "");
}

QString Database::getHeaderText()
{
    // Header
    if (globalStringValues.contains("printHeader")) return globalStringValues.value("printHeader");

    QVariant value;
    QString strValue;
    select_globals("printHeader", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("printHeader", strValue);
        return globalStringValues.value("printHeader");
    }

    return Database::updateGlobals("printHeader", Q_NULLPTR, "");
}

QString Database::getFooterText()
{
    // Footer
    if (globalStringValues.contains("printFooter")) return globalStringValues.value("printFooter");

    QVariant value;
    QString strValue;
    select_globals("printFooter", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("printFooter", strValue);
        return globalStringValues.value("printFooter");
    }

    return Database::updateGlobals("printFooter", Q_NULLPTR, "");
}

QJsonArray Database::getPrinters()
{
    QJsonArray printers;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id, name, printer, definition, mode FROM printers");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    while (query.next()) {
        QJsonObject printer;
        printer["name"] = query.value("name").toString();
        printer["id"] = query.value("id").toInt();
        printer["printer"] = query.value("printer").toString();
        printer["definition"] = query.value("definition").toJsonArray();
        printer["mode"] = query.value("mode").toJsonArray();
        printers.append(printer);
    }

    return printers;
}

QString Database::getPrinterName(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT name FROM printers WHERE id=:id");
    query.bindValue(":id", id);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        return query.value("name").toString();
    }

    return "QrkPDF";
}

int Database::getPrinterIdFromProduct(int id)
{
    int printerid = -1;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery query2(dbc, Q_FUNC_INFO);
    CSqlQuery query3(dbc, Q_FUNC_INFO);
    query.prepare(QString("SELECT printerid, groupid FROM products WHERE id=:id"));
    query.bindValue(":id", id);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        if (query.value("printerid").isNull()) {
            query2.prepare(QString("SELECT printerid, categoryId FROM groups WHERE id=:id"));
            query2.bindValue(":id", query.value("groupid").toInt());
            query2.exec();
            if (query2.next()) {
                if (query2.value("printerid").isNull()) {
                    query3.prepare(QString("SELECT printerid FROM categories WHERE id=:id"));
                    query3.bindValue(":id", query2.value("categoryId").toInt());
                    query3.exec();
                    if (query3.next()) {
                        if (query3.value("printerid").isNull()) {
                            return printerid;
                        } else {
                            return query3.value("printerid").toInt();
                        }
                    }
                } else {
                    return query2.value("printerid").toInt();
                }
            }
        } else {
            return query.value("printerid").toInt();
        }
    }
    return printerid;
}

QJsonArray Database::getDefinitions()
{
    QJsonArray definitions;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT id, name, mode FROM printerdefs");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    while (query.next()) {
        QJsonObject definition;
        definition["name"] = query.value("name").toString();
        definition["id"] = query.value("id").toString();
        definition["mode"] = query.value("mode").toString();
        definitions.append(definition);
    }

    return definitions;
}

int Database::getDefinitionId(const QString &name)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    int id = 0;
    query.prepare("SELECT id FROM printerdefs WHERE name=:name");
    query.bindValue(":name", name);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    while (query.next()) {
        id = query.value("id").toInt();
    }

    return id;
}

QString Database::getDefinitionName(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    QString name = "n/a";
    query.prepare("SELECT name FROM printerdefs WHERE id=:id");
    query.bindValue(":id", id);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    while (query.next()) {
        name = query.value("name").toString();
    }

    return name;
}

//--------------------------------------------------------------------------------

QString Database::getShortCurrency()
{
    if (globalStringValues.contains("shortcurrency")) return globalStringValues.value("shortcurrency");

    QString currency = getCurrency();
    QString currencySymbol;

    if (currency == "CHF")
        currencySymbol = "Fr";
    else
        currencySymbol = "€";

    Database::updateGlobals("shortcurrency", Q_NULLPTR, currencySymbol);
    return currencySymbol;
}
//--------------------------------------------------------------------------------

QString Database::getCurrency()
{
    if (globalStringValues.contains("currency")) return globalStringValues.value("currency");

    QVariant value;
    QString strValue;
    select_globals("currency", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("currency", strValue);
        return globalStringValues.value("currency");
    }

    return Database::updateGlobals("currency", Q_NULLPTR, QLocale().currencySymbol());
}

//--------------------------------------------------------------------------------

QString Database::getCashRegisterId()
{
    if (globalStringValues.contains("shopCashRegisterId")) {
        if (DemoMode::isDemoMode())
            return "DEMO-" + globalStringValues.value("shopCashRegisterId");
        else
            return globalStringValues.value("shopCashRegisterId");
    }

    QVariant value;
    QString strValue;
    select_globals("shopCashRegisterId", value, strValue);
    if (!strValue.isEmpty()) {
        globalStringValues.insert("shopCashRegisterId", strValue);
        if (DemoMode::isDemoMode()) return "DEMO-" + globalStringValues.value("shopCashRegisterId");

        return globalStringValues.value("shopCashRegisterId");
    }

    return "";
}

//--------------------------------------------------------------------------------

QStringList Database::getMaximumItemSold()
{
    QStringList list;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("select DISTINCT max (p2.sold), p2.name, p2.tax, p2.gross from (select max(version) as version, "
                  "origin from products group by origin) p1 inner join (select * from products) as p2 on "
                  "p1.version=p2.version and p1.origin=p2.origin where visible = 1");
    //    query.prepare("SELECT max(sold), name, tax, gross FROM products LIMIT 1;");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        list << query.value("name").toString() << query.value("tax").toString() << query.value("gross").toString();
        return list;
    }

    list << ""
         << "20"
         << "0,00";
    return list;
}

//--------------------------------------------------------------------------------

bool Database::addCustomerText(int id, const QString &text)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    QString q = QString("INSERT INTO customer (receiptNum, text) VALUES (:receiptNum, :text)");
    bool ok = query.prepare(q);

    query.bindValue(":receiptNum", id);
    query.bindValue(":text", text);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.exec())
        return true;
    else
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();

    return false;
}

//--------------------------------------------------------------------------------

QString Database::getCustomerText(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = query.prepare("SELECT text FROM customer WHERE receiptNum=:receiptNum");

    query.bindValue(":receiptNum", id);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.exec()) {
        if (query.next()) return query.value("text").toString();

        return "";
    } else {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
    }

    return "";
}

//--------------------------------------------------------------------------------

double Database::getTaxFromProduct(const int id)
{
#ifdef QRK_PROFEATURES
    return ProFeatures::getTaxFromProduct(id);
#else
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    double tax = Database::getDefaultTax().toDouble();

    query.prepare("SELECT tax, groupid FROM products WHERE id = :id");
    query.bindValue(":id", id);
    if (query.exec()) {
        if (query.next()) tax = query.value("tax").toDouble();
    }

    return tax;
#endif
}

int Database::getProductIdByName(const QString &name, const int groupid)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = false;
    /*
    if (groupid > 0)
        ok = query.prepare("select p2.id from (select max(version) as version, origin from "
                           "products group by origin) p1 inner join (select * from products) as  "
                           "p2 on p1.version=p2.version and p1.origin=p2.origin where name=:name "
                           "AND groupid=:groupid AND visible >= 0");
    else
        ok = query.prepare("select p2.id from (select max(version) as version, origin from "
                           "products group by origin) p1 inner join (select * from products) as  "
                           "p2 on p1.version=p2.version and p1.origin=p2.origin where name=:name "
                           "AND visible >= 0");
    */
    if (groupid > 0)
        ok = query.prepare("select p2.id from (select max(version) as version, origin from "
                           "products group by origin) p1 inner join (select * from products) as  "
                           "p2 on p1.version=p2.version and p1.origin=p2.origin where name=:name "
                           "AND groupid=:groupid AND (visible >= 0 OR itemnum = 'D')");
    else
        ok = query.prepare("select p2.id from (select max(version) as version, origin from "
                           "products group by origin) p1 inner join (select * from products) as  "
                           "p2 on p1.version=p2.version and p1.origin=p2.origin where name=:name "
                           "AND (visible >= 0 OR itemnum = 'D')");

    query.bindValue(":name", name);
    query.bindValue(":groupid", groupid);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.exec()) {
        if (query.next()) return query.value("id").toInt();
    }

    return -1;
}

int Database::getProductIdByNumber(const QString &number)
{
    if (number.isEmpty()) return -1;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = query.prepare("select p2.id from (select max(version) as version, origin from products "
                            "group by origin) p1 inner join (select * from products) as  p2 on "
                            "p1.version=p2.version and p1.origin=p2.origin where itemnum=:number AND "
                            "visible >= 0");
    query.bindValue(":number", number);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.exec()) {
        if (query.next()) return query.value("id").toInt();
    }

    return -1;
}

int Database::getProductIdByBarcode(const QString &code)
{
    if (code.isEmpty()) return -1;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = query.prepare("select p2.id from (select max(version) as version, origin from products "
                            "group by origin) p1 inner join (select * from products) as  p2 on "
                            "p1.version=p2.version and p1.origin=p2.origin where barcode=:barcode "
                            "AND visible >= 0");
    query.bindValue(":barcode", code);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    if (query.exec()) {
        if (query.next()) {
            int id = query.value("id").toInt();
            return id;
        }
    }

    return -1;
}

QString Database::getProductNameById(int id)
{
    if (id == 0) return QString();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    QString q = QString("SELECT name FROM products WHERE id=:id");
    bool ok = query.prepare(q);
    query.bindValue(":id", id);

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }
    if (query.next()) {
        return query.value("name").toString();
    }

    return QString();
}

bool Database::addProduct(const QJsonObject &data, int &productId)
{

    //    bool productExists = Database::exists(data["name"].toString()) && data["id"].isUndefined();
    bool productExists = productId > 0 && data["id"].isUndefined();

    int visible = data["visible"].toInt();
    int group = 2;
    QString itemNum = "";

    if (!data["itemnum"].toString().isEmpty()) {
        if (Utils::isNumber(data["itemnum"].toString())) {
            itemNum = (data["itemnum"].toString().toInt() == 0) ? Database::getNextProductNumber()
                                                                : data["itemnum"].toString();
        } else {
            itemNum = data["itemnum"].toString();
        }
    } else {
        itemNum = Database::getNextProductNumber();
    }

    if (data.contains("group") && !data["group"].isNull()) {
        group = data["group"].toInt();
    }
    if (data["name"].toString().startsWith("Zahlungsbeleg für Rechnung")
        || data["name"].toString().startsWith("Startbeleg")
        || QRegularExpression("^\\S+\\s+Gutschein|^Gutschein").match(data["name"].toString()).hasMatch()) {
        if (productExists) {
            if (data["name"].toString().startsWith("Startbeleg")) {
                qWarning() << "Function Name: " << Q_FUNC_INFO << "Ein Startbeleg ist schon vorhanden.";
                return true;
            }
            return false;
        }

        /* We do not need to display this in the Manager*/
        visible = 0;
        group = 1;
        itemNum = "";
    }

    if (data["name"].toString().startsWith("Monatsbeleg") || data["name"].toString().startsWith("Jahresbeleg")) {
        itemNum = "";
        group = 1;
    }

    if (productExists) {
        if ((data["itemnum"].toString().toInt() == 0) && !itemNum.isEmpty() && itemNum.compare("D") != 0) {
            Database::insertProductItemnumToExistingProduct(itemNum, productId);
        }
        return true;
    }

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    QString q = "";
    if (data["id"].isUndefined()) {
        q = QString("INSERT INTO products (name, itemnum, barcode, tax, net, gross, "
                    "visible, groupid, version, origin) VALUES (:name, :itemnum, '', "
                    ":tax, :net, :gross, :visible, :group, :version, :origin)");

    } else {
        q = QString("INSERT INTO products (id, name, itemnum, barcode, tax, net, gross, "
                    "visible, groupid, version, origin) VALUES (:id, :name, :itemnum, '', "
                    ":tax, :net, :gross, :visible, :group, :version, :origin)");
    }

    bool ok = query.prepare(q);

    query.bindValue(":name", data["name"].toString());
    query.bindValue(":itemnum", itemNum);
    query.bindValue(":tax", data["tax"].toDouble());
    query.bindValue(":net", data["net"].toDouble());
    query.bindValue(":gross", data["gross"].toDouble());
    query.bindValue(":group", group);
    if (data["id"].isUndefined()) {
        query.bindValue(":visible", visible);
        query.bindValue(":version", 0);
        query.bindValue(":origin", -1);
    } else {
        query.bindValue(":id", data["id"].toInt());
        query.bindValue(":visible", -1);
        query.bindValue(":version", -1);
        query.bindValue(":origin", data["id"].toInt());
    }

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
        return false;
    }

    if (query.exec()) {
        if (!data["id"].isUndefined() && !data["id"].isNull()) {
            productId = data["id"].toInt();
            return true;
        }
        int id = Database::getProductIdByName(data["name"].toString(), group);
        if (id > 0) {
            productId = id;
            query.prepare(QString("UPDATE products SET origin=:origin WHERE id=:id"));
            query.bindValue(":id", id);
            query.bindValue(":origin", id);
            ok = query.exec();
            if (!ok) {
                qWarning() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
                qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
            }
            return ok;
        }
        return true;
    } else {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
    }

    return false;
}

QMap<QString, QBCMath> Database::getProductRealGrossFromReceipt(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery grossQuery(dbc, Q_FUNC_INFO);

    QMap<QString, QBCMath> map;
    grossQuery.prepare("SELECT gross, tax FROM products WHERE id=:id");
    query.prepare("SELECT count, product FROM orders WHERE receiptId=:receiptId");
    query.bindValue(":receiptId", id);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    while (query.next()) {
        grossQuery.bindValue(":id", query.value("product").toInt());
        if (!grossQuery.exec()) {
            qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << grossQuery.lastError().text();
            qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(grossQuery);
        }

        if (grossQuery.next()) {
            QBCMath count(query.value("count").toDouble());
            QBCMath gross(grossQuery.value("gross").toDouble());
            QBCMath tax(grossQuery.value("tax").toDouble());
            count.round(2);
            gross.round(2);
            gross *= count;
            tax.round(2);
            if (map.contains(tax.toLocale()))
                map[tax.toLocale()] += gross;
            else
                map[tax.toLocale()] = gross;
        }
    }

    return map;
}

QJsonObject Database::getProductByName(const QString &name, int visible)
{
    QJsonObject data;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = query.prepare("select p2.id, p2.name, p2.itemnum, p2.barcode, p2.tax, p2.net, p2.gross, "
                            "p2.description, p2.version, p2.origin from (select max(version) as "
                            "version, origin from products group by origin) p1 inner join (select * "
                            "from products) as  p2 on p1.version=p2.version and p1.origin=p2.origin "
                            "WHERE name=:name AND visible >= :visible");

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
        return QJsonObject();
    }

    query.bindValue(":name", name);
    query.bindValue(":visible", visible);

    ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
        return QJsonObject();
    }

    if (query.next()) {
        data["id"] = query.value("id").toString().trimmed();
        data["name"] = query.value("name").toString().trimmed();
        data["itemnum"] = query.value("itemnum").toString().trimmed();
        data["barcode"] = query.value("barcode").toString().trimmed();
        data["tax"] = query.value("tax").toDouble();
        data["net"] = query.value("net").toDouble();
        data["gross"] = query.value("gross").toDouble();
        data["description"] = query.value("description").toString().trimmed();
        data["version"] = query.value("version").toInt();
        data["origin"] = query.value("gross").toInt();
        return data;
    }

    return QJsonObject();
}

QJsonObject Database::getProductById(int id, int visible)
{
    if (id < 1) return QJsonObject();

    QJsonObject data;
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    bool ok = query.prepare("select name, itemnum, barcode, tax, net, gross, description, version, "
                            "origin from products WHERE id=:id AND visible >= :visible");

    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
        return QJsonObject();
    }

    query.bindValue(":id", id);
    query.bindValue(":visible", visible);

    if (!query.exec()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << getLastExecutedQuery(query);
        return QJsonObject();
    }

    if (query.next()) {
        data["name"] = query.value("name").toString().trimmed();
        data["itemnum"] = query.value("itemnum").toString().trimmed();
        data["barcode"] = query.value("barcode").toString().trimmed();
        data["tax"] = query.value("tax").toDouble();
        data["net"] = query.value("net").toDouble();
        data["gross"] = query.value("gross").toDouble();
        data["description"] = query.value("description").toString().trimmed();
        data["version"] = query.value("version").toInt();
        data["origin"] = query.value("origin").toInt();
        return data;
    }

    return QJsonObject();
}

qulonglong Database::getFirstProductNumber()
{

    if (globalStringValues.contains("firstProductnumber"))
        return globalStringValues.value("firstProductnumber").toULongLong();

    Settings settings;
    qulonglong value = settings.value("firstProductnumber", 1).toULongLong();
    globalStringValues.insert("firstProductnumber", QString::number(value));
    return globalStringValues.value("firstProductnumber").toULongLong();
}

QString Database::getNextProductNumber(bool find_gaps)
{
    qulonglong value = getFirstProductNumber();

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    if (find_gaps) {
        query.prepare(QString("SELECT min(a.itemnum) + 1 as lastItemNum FROM products a LEFT OUTER JOIN products b ON "
                              "b.itemnum = a.itemnum + 1 WHERE b.itemnum is null AND a.itemnum + 1 >= %1")
                .arg(value));
    } else {
        query.prepare("SELECT MAX(CAST(itemnum as decimal)) AS lastItemNum FROM products");
    }

    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        QString itemnum = query.value("lastItemNum").toString();
        qulonglong maxItemnum = itemnum.toULongLong();
        maxItemnum = (value > maxItemnum) ? value : maxItemnum;
        return QString::number(maxItemnum);
    }

    return "";
}

//--------------------------------------------------------------------------------

//--------------------------------------------------------------------------------

int Database::getLastReceiptNum(bool realReceipt)
{

    if (realReceipt) {
        QSqlDatabase dbc = Database::database();
        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.prepare("SELECT receiptNum as value FROM receipts WHERE id=(SELECT "
                      "max(id) FROM receipts WHERE payedBy < :payedby);");
        query.bindValue(":payedby", PAYED_BY_REPORT_EOD);

        if (!query.exec()) {
            qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
            qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
        }

        if (query.next()) return query.value("value").toInt();

    } else {
        QVariant value;
        QString strValue;
        int id = select_globals("lastReceiptNum", value, strValue);
        if (id > 0) return value.toInt();
    }

    return 0;
}

QStringList Database::getLastReceipt()
{
    QStringList list;

    int lastReceipt = getLastReceiptNum();

    if (lastReceipt == 0) return list;

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT timestamp, receiptNum, payedBy, gross FROM receipts "
                  "WHERE receiptNum=:receiptNum");
    query.bindValue(":receiptNum", lastReceipt);
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    query.next();
    list << query.value("timestamp").toString() << query.value("receiptNum").toString()
         << query.value("payedBy").toString() << query.value("gross").toString();

    return list;
}

//--------------------------------------------------------------------------------

void Database::setCurfewTime(QTime time, bool temp)
{
    time = QTime(time.hour(), time.minute(), 0);
    QString timeString = time.toString("hh:mm:ss");
    timeString = timeString.mid(0, timeString.length() - 2) + "00";

    QDateTime dt(QDateTime::currentDateTime());
    dt.setTime(time);
    if (temp)
        insert2globals("curfewTemp", QVariant(), dt.toString(Qt::ISODate));
    else
        insert2globals("curFew", QVariant(), timeString);
}

QTime Database::getCurfewTime()
{
    QString curfew = "curFew";

    if (globalStringValues.contains("curfewTemp")) {
        QDateTime dt = QDateTime::fromString(globalStringValues.value("curfewTemp"), Qt::ISODate);
        QTime time = dt.time();
        dt.setTime(QTime(23, 59, 59));
        qint64 secsTo = QDateTime::currentDateTime().secsTo(dt);
        if (secsTo < 60) {
            Database::updateGlobals("curfewTemp", Q_NULLPTR, Q_NULLPTR);
            Database::delete_globals("curfewTemp");
        } else {
            return time;
        }
    }

    QVariant value;
    QString strValue;
    int id = select_globals("curfewTemp", value, strValue);
    if (id > 0) {
        strValue = strValue.mid(0, strValue.length() - 2) + "00";
        QDateTime dt = QDateTime::fromString(strValue, Qt::ISODate);
        QTime time = dt.time();
        dt.setTime(QTime(23, 59, 59));
        qint64 secsTo = QDateTime::currentDateTime().secsTo(dt);
        if (secsTo < 60) {
            Database::updateGlobals("curfewTemp", Q_NULLPTR, Q_NULLPTR);
            Database::delete_globals("curfewTemp");
        } else {
            return time;
        }
    }

    if (globalStringValues.contains(curfew)) {
        QTime t(QTime::fromString(globalStringValues.value(curfew), "hh:mm:ss"));
        return QTime(t.hour(), t.minute(), 0);
    }

    id = select_globals(curfew, value, strValue);
    if (id > 0) {
        QTime t = QTime::fromString(strValue, "hh:mm:ss");
        strValue = strValue.mid(0, strValue.length() - 2) + "00";
        if (t.second() > 0) Database::setCurfewTime(QTime(t.hour(), t.minute(), 0));
        Database::updateGlobals(curfew, Q_NULLPTR, strValue);
        return QTime::fromString(globalStringValues.value(curfew), "hh:mm:ss");
    }

    Database::setCurfewTime(QTime(0, 0, 0));
    return QTime(0, 0, 0);
}

QTime Database::getCurfewTimeByDate(QDate date)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(
        QString("SELECT curfew FROM reports WHERE id=(SELECT max(id) FROM reports WHERE timestampfrom like '%1%')")
            .arg(date.toString(Qt::ISODate)));
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        QString time = query.value("curfew").toString();
        return QTime::fromString(time, "hh:mm");
    }

    int check = QString("%1%2%3").arg(date.year()).arg(date.month() < 10 ? "0" : "").arg(date.month()).toInt();
    int checkNow = QString("%1%2%3")
                       .arg(QDate::currentDate().year())
                       .arg(QDate::currentDate().month() < 10 ? "0" : "")
                       .arg(QDate::currentDate().month())
                       .toInt();

    if (check >= checkNow) return Database::getCurfewTime();

    return QTime(0, 0, 0);
}

QTime Database::getLastEOACurfewTime()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT curfew FROM reports WHERE id=(SELECT max(id) FROM reports)");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) {
        QString time = query.value("curfew").toString();
        return QTime::fromString(time, "hh:mm");
    }

    // return QTime(0, 0, 0);
    return Database::getCurfewTime();
}

QDateTime Database::getFromDateTime(QDateTime datetime)
{

    QTime time(Database::getCurfewTimeByDate(datetime.date()));
    QDateTime dt;
    dt.setDate(datetime.date());
    dt.setTime(time);

    qint64 diff = datetime.time().secsTo(time);
    if (diff < 0) return dt;

    return dt.addDays(-1);
}

QDate Database::getFirstReceiptDate()
{
    return getFirstReceiptDateTime().date();
}

QDateTime Database::getFirstReceiptDateTime()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT infodate FROM receipts where receiptNum IN (SELECT "
                  "min(receiptNum) FROM receipts)");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    QDateTime dt(QDateTime::currentDateTime());
    if (query.next()) {
        dt = query.value(0).toDateTime();
        return dt;
    }

    return QDateTime();
}

QDate Database::getLastReceiptDate()
{
    return getLastReceiptDateTime().date();
}

//--------------------------------------------------------------------------------

QDateTime Database::getLastReceiptDateTime(bool realtime)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    if (realtime)
        query.prepare("SELECT timestamp FROM receipts where receiptNum IN (SELECT "
                      "max(receiptNum) FROM receipts)");
    else
        query.prepare("SELECT infodate FROM receipts where receiptNum IN (SELECT "
                      "max(receiptNum) FROM receipts)");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    QDateTime dt(QDateTime::currentDateTime());
    if (query.next()) {
        dt = query.value(0).toDateTime();
        return dt;
    }

    return QDateTime();
}

//--------------------------------------------------------------------------------

QStringList Database::getDatabaseTableHeaderNames(const QString &tablename)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("SELECT * FROM %1 LIMIT 1").arg(tablename));

    if (query.exec()) {
        QStringList list;
        if (query.next()) {
            const QSqlRecord record = query.record();
            int count = record.count();
            for (int i = 0; i < count; ++i) {
                list << tablename + "." + record.fieldName(i); // Headers
            }
            return list;
        }
    }

    return QStringList();
}

//--------------------------------------------------------------------------------

QString Database::getShopName()
{

    QVariant value;
    QString strValue;
    int id = select_globals("shopName", value, strValue);
    if (id > 0) return strValue.trimmed();

    return "";
}

QString Database::getShopMasterData()
{

    QString name;
    QVariant value;
    QString strValue;
    QString tmp = "";
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    if (globalStringValues.contains("shopOwner")) {
        tmp = globalStringValues.value("shopOwner");
    } else {
        int id = select_globals("shopOwner", value, strValue);
        if (id > 0) tmp = strValue;

        Database::updateGlobals("shopOwner", Q_NULLPTR, tmp);
    }
    name = (tmp.isEmpty()) ? "" : "\n" + tmp;
    tmp = "";

    if (globalStringValues.contains("shopAddress")) {
        tmp = globalStringValues.value("shopAddress");
    } else {
        int id = select_globals("shopAddress", value, strValue);
        if (id > 0) tmp = strValue;

        Database::updateGlobals("shopAddress", Q_NULLPTR, tmp);
    }
    name += (tmp.isEmpty()) ? "" : "\n" + tmp;
    tmp = "";

    if (globalStringValues.contains("shopUid")) {
        tmp = globalStringValues.value("shopUid");
    } else {
        int id = select_globals("shopUid", value, strValue);
        if (id > 0) tmp = strValue;

        Database::updateGlobals("shopUid", Q_NULLPTR, tmp);
    }
    name += (tmp.isEmpty()) ? "" : "\n" + tmp;

    return name;
}

QString Database::getLastVersionInfo()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT version FROM journal WHERE id = (SELECT MAX(id) FROM journal)");
    if (!query.exec()) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " error: " << query.lastError().text();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " query: " << Database::getLastExecutedQuery(query);
    }
    if (query.next()) return query.value("version").toString();

    return "";
}

//--------------------------------------------------------------------------------
void Database::reopen()
{
    QSqlDatabase dbc = QSqlDatabase::database("CN");
    dbc.close();
    open(false);
}

QJsonObject Database::getConnectionDefinition()
{
    QJsonObject jobj;
    jobj.insert("dbtype", getDatabaseType());
    jobj.insert("databasename", globalStringValues.value("databasename"));
    jobj.insert("databasehost", globalStringValues.value("databasehost"));
    jobj.insert("databaseusername", globalStringValues.value("databaseusername"));
    jobj.insert("databasepassword", globalStringValues.value("databasepassword"));
    jobj.insert("databaseoptions", globalStringValues.value("databaseoptions"));
    return jobj;
}

int Database::getCcurrentSchemaVersion()
{
    const int CURRENT_SCHEMA_VERSION = 30;
    return CURRENT_SCHEMA_VERSION;
}

bool Database::doDatabaseStuff(QSqlDatabase &connection, bool &database_exists)
{
    CSqlQuery query(connection, Q_FUNC_INFO);
    query.exec("SELECT 1 FROM globals");
    database_exists = query.next();
    QString dbType = connection.driverName();

    if (!database_exists) { // empty DB, create all tables
        QFile f;

        if (dbType == "QSQLITE") {
            f.setFileName(":src/sql/QRK-sqlite.sql");
        } else if (dbType == "QMYSQL") {
            f.setFileName(":src/sql/QRK-mysql.sql");
            connection.setConnectOptions("MYSQL_OPT_RECONNECT=1");
        }

        f.open(QIODevice::ReadOnly);
        qInfo() << "Function Name: " << Q_FUNC_INFO << " Database type " << Database::getDatabaseVersion();
        QString cmd = f.readAll();
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
        QStringList commands = cmd.split(';', Qt::SkipEmptyParts);
#else
        QStringList commands = cmd.split(';', QString::SkipEmptyParts);
#endif
        foreach (const QString &command, commands) {
            if (command.trimmed().isEmpty()) continue;
            bool ok = query.exec(command);
            if (!ok) {
                qWarning() << "Function Name: " << Q_FUNC_INFO << " " << query.lastError().text();
                qWarning() << "Function Name: " << Q_FUNC_INFO << " " << Database::getLastExecutedQuery(query);
            }
        }

        insert2globals("demomode", 1, QVariant());
        insert2globals("lastReceiptNum", 0, QVariant());
        insert2globals("shopName", QVariant(), "");
        insert2globals("backupTarget", QVariant(), "");
        insert2globals("schemaVersion", getCcurrentSchemaVersion(), QVariant());

        query.exec(QString("INSERT INTO "
                           "`journal`(id,version,cashregisterid,datetime,data,checksum) "
                           "VALUES (NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                           "'9f11c3693ee2f40c9c10d741bc13f5652586fa564a071eb231541abf64dc1"
                           "f5aa4c8845119ddc2734e836dfa394426d02609a90ad99d5ec1212988424e1"
                           "6c9f47f679b48253b55b0af91d0ee22dacc9f947201288d48b7f14a6fe1c89"
                           "5e1b4cf','DBD7ACB39B653D948DEDD342B912D66F14482DBB')"));
        query.exec(QString("INSERT INTO "
                           "`journal`(id,version,cashregisterid,datetime,data,checksum) "
                           "VALUES (NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                           "'9f11c3693ee2f40c9c10d741bc13f5652782f7d13ee7f6bd1725691470a0c"
                           "ed96ea4ef910accfa7f416797b7a73c17f239a1abe7f52887d582a719a320d"
                           "480b76e90b95edfe7eda059aca296e2916d6fa0cfee77d4db0dd01a25d3720"
                           "f89f633e314241be48b6078a3dc4a13fb11cea51a9c582dff0b7dae944945f"
                           "9d84eb72a','08EF99A8218FA5130A2B0C58F27D897195162744')"));
        query.exec(QString("INSERT INTO "
                           "`journal`(id,version,cashregisterid,datetime,data,checksum) VALUES "
                           "(NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                           "'9f11c3693ee2f40c9c10d741bc13f5656545e1d75d25e2b35c2b958d9c37bc733b080"
                           "292d1738c51366c0cc0e5317acdae667e3e1a166ba75be259466db75a5546ab362107a"
                           "37cb7a3bd2eaf5785c3d7baa9d5aa723cb1b8333a66af9e59dfced2cbc6f233614ac42"
                           "5b77794d0541ab86019388693a8d32064cfadddaff462412bfd20a876c8bbf503bf224"
                           "561fe52b2551258978ed8bd0d33382a5d3c5b9768f2828b512d3264dd6b8c1314b0a15"
                           "856b3ae5f8ebe5830fdd5e2629c18fc2b3e8511eaa6aaf59fc359a531fb6e5f8658','"
                           "5C29C8A36F5B46F46CE78FB4502F4DC8DFDCEBA6')"));
        query.exec(QString("INSERT INTO "
                           "`journal`(id,version,cashregisterid,datetime,data,checksum) VALUES "
                           "(NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                           "'9f11c3693ee2f40c9c10d741bc13f5656545e1d75d25e2b35c2b958d9c37bc73a79a1"
                           "c487e6ea70eb7e63c6e708d983879852e1a8d9ea66167824c5312f1d12ca86ae59bd44"
                           "98a5f6b4cecfd27e28218','6B4D651268E7436F43E668590BFC5D6C86F1AEE7')"));
        updatePrinters();
        query.exec(QString("UPDATE `categories` SET printerid=NULL"));
        query.exec(QString("UPDATE `groups` SET printerid=NULL"));
        query.exec(QString("UPDATE `products` SET printerid=NULL"));

    } else { // db already exists; check if we need to run an update
        Database::encryptData(connection);
        int schemaVersion = 1;
        QVariant value;
        QString strValue;
        int id = select_globals("schemaVersion", value, strValue);
        if (id > 0)
            schemaVersion = value.toInt();
        else // schemaVersion not set in globals, must be version 1
            insert2globals("schemaVersion", 1, QVariant());

        if (schemaVersion > getCcurrentSchemaVersion()) {
            connection.close();
            QMessageBox errorDialog;
            errorDialog.setIcon(QMessageBox::Critical);
            errorDialog.addButton(QMessageBox::Ok);
            errorDialog.setText(tr("Falsches QRK Registrier Kassen Datenbank Schema.\nVerwenden Sie "
                                   "bitte eine neuere Version von QRK.\nBei Fragen wenden Sie sich "
                                   "bitte an das QRK Forum (www.ckvsoft.at)"));
            errorDialog.setWindowTitle(QObject::tr("Falsche Version"));
            errorDialog.exec();
            return false;
        }

        if (schemaVersion < getCcurrentSchemaVersion()) Backup::create();

        // run all db update scripts from the db version + 1 to what the program
        // currently needs
        for (int i = schemaVersion + 1; i <= getCcurrentSchemaVersion(); i++) {
            QFile f;

            if (dbType == "QSQLITE")
                f.setFileName(QString(":src/sql/QRK-sqlite-update-%1.sql").arg(i));
            else if (dbType == "QMYSQL")
                f.setFileName(QString(":src/sql/QRK-mysql-update-%1.sql").arg(i));

            if (!f.open(QIODevice::ReadOnly)) {
                qCritical("could not load internal update file %s", qPrintable(f.fileName()));
                return false; // should never happen
            }

            QString cmd = f.readAll();
#if (QT_VERSION >= QT_VERSION_CHECK(5, 14, 0))
            QStringList commands = cmd.split(';', Qt::SkipEmptyParts);
#else
            QStringList commands = cmd.split(';', QString::SkipEmptyParts);
#endif
            bool ok;
            foreach (const QString &command, commands) {
                if (command.trimmed().isEmpty()) continue;
                ok = query.exec(command.trimmed());
                if (!ok) {
                    qCritical() << "Function Name: " << Q_FUNC_INFO << " " << query.lastError().text();
                    qCritical() << "Function Name: " << Q_FUNC_INFO << " " << Database::getLastExecutedQuery(query);
                    connection.rollback();
                    return false;
                }
            }

            // changes which are not possible in sql file needed for update-7
            if (i == 7) {
                ok = query.exec(QString("UPDATE products SET groupid=2 WHERE name NOT "
                                        "LIKE 'Zahlungsbeleg für Rechnung%'"));
                if (!ok) {
                    qWarning() << "Function Name: " << Q_FUNC_INFO << " " << query.lastError().text();
                    qWarning() << "Function Name: " << Q_FUNC_INFO << " " << Database::getLastExecutedQuery(query);
                }
                ok = query.exec(QString("UPDATE products SET groupid=1 WHERE name LIKE "
                                        "'Zahlungsbeleg für Rechnung%'"));
                if (!ok) {
                    qWarning() << "Function Name: " << Q_FUNC_INFO << " " << query.lastError().text();
                    qWarning() << "Function Name: " << Q_FUNC_INFO << " " << Database::getLastExecutedQuery(query);
                }
            }
            if (i == 14) {
                ok = query.exec(QString("UPDATE journal SET `text`='%1' WHERE id=1")
                        .arg("Id\tProgrammversion\tKassen-"
                             "Id\tKonfigurationsänderung\tBeschreibung\tErs"
                             "tellungsdatum"));
                if (!ok) {
                    qCritical() << "Function Name: " << Q_FUNC_INFO << " " << query.lastError().text();
                    qCritical() << "Function Name: " << Q_FUNC_INFO << " " << Database::getLastExecutedQuery(query);
                    connection.rollback();
                    return false;
                }
            }
            if (i == 19) {
                QrkJournal::encodeJournal(connection);
            }
            if (i == 21) {
                updatePrinters();
                fixDoubleProductNames();
                query.exec("UPDATE `categories` SET printerid=NULL");
                query.exec("UPDATE `groups` SET printerid=NULL");
                query.exec("UPDATE `products` SET printerid=NULL");
                query.exec("update products set itemnum='' where itemnum in (select itemnum "
                           "from ((select max(version) as version, origin from products group "
                           "by origin) p1 inner join (select * from products) as  p2 on "
                           "p1.version=p2.version and p1.origin=p2.origin) where itemnum !='' "
                           "group by itemnum having count(itemnum) > 1)");
            }

            if (i == 23) {
                Database::insert2globals("ckvsoftSchemaVersion", 1, QVariant());
                Database::insert2globals("gastroSchemaVersion", 1, QVariant());
            }
            if (i == 30) {
                QVariant vValue;
                QString strValue;
                int id = AbstractDataBase::select_globals("certificate", vValue, strValue);
                if (id > 0) {
                    QString serialHex = QString::number(vValue.toInt(), 16).toUpper();
                    query.prepare("INSERT INTO certs (serial, cert, created) "
                                  "VALUES (:serial, :cert, datetime('now'))");

                    query.bindValue(":serial", serialHex);
                    query.bindValue(":cert", strValue);

                    if (!query.exec()) {
                        qCritical() << "query:" << AbstractDataBase::getLastExecutedQuery(query);
                        qCritical() << "DB insert error:" << query.lastError().text();
                    } else {
                        AbstractDataBase::delete_globals("certificate");
                    }
                }
            }

            Database::insert2globals("schemaVersion", i, QString());
        }

        if (schemaVersion != getCcurrentSchemaVersion())
            Database::insert2globals("schemaVersion", getCcurrentSchemaVersion(), QString());
    }

    // check for group with id 1 and id 2
    // these are absolutely necessary
    query.exec("SELECT id FROM groups WHERE id=1");
    if (!query.next()) {
        query.exec("INSERT INTO `groups`(id,name,visible) VALUES (1,'auto',0);");
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Database manibulation: missing groupid -> 1";
    }
    query.exec("SELECT id FROM groups WHERE id=2");
    if (!query.next()) {
        query.exec("INSERT INTO `groups`(id,name,visible) VALUES (2,'Standard',0);");
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Database manibulation: missing groupid -> 2";
    }

    query.exec("SELECT id FROM categories WHERE id=1");
    if (!query.next()) {
        query.exec("INSERT INTO `categories`(id,name) VALUES (1,'Standard');");
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Database manibulation: missing categoryid -> 1";
    }

    if (!query.exec("SELECT COUNT(*) FROM `actiontypes` WHERE `actionId` = 3 AND `actionText` = 'Gutschein'")) {
        qDebug() << "Function Name: " << Q_FUNC_INFO << "SQL Error:" << query.lastError().text();
        return false; // Oder handle den Fehler entsprechend
    }

    int count = 1;
    if (query.next()) {
        count = query.value(0).toInt();
        qDebug() << "Function Name: " << Q_FUNC_INFO << " Count:" << count; // Debug-Ausgabe, um den Wert zu überprüfen
    } else {
        qDebug() << "Function Name: " << Q_FUNC_INFO
                 << " No results found."; // Dies bedeutet, dass die Abfrage nicht erfolgreich war
    }

    if (count == 0) {
        query.exec("UPDATE `receipts` SET `payedBy` = `payedBy` + 1 WHERE `payedBy` > 2");
        query.exec("UPDATE `actiontypes` SET `actionId` = `actionId` + 1 WHERE `actionId` > 2");

        query.exec(
            "INSERT INTO `actiontypes`(`id`, `actionId`, `actionText`, `comment`) VALUES (NULL, 3, 'Gutschein', 'PayedByText')");
        qInfo() << "Function Name: " << Q_FUNC_INFO << " Database manibulation fix: missing actionId 3";
    } else {
        qDebug() << "Function Name: " << Q_FUNC_INFO << " actionId 3 already there";
    }
    return true;
}

//--------------------------------------------------------------------------------

int Database::getPayedBy(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT payedBy FROM receipts WHERE receiptNum=:id");
    query.bindValue(":id", id);
    query.exec();
    query.next();

    return query.value(0).toInt();
}

QMap<int, double> Database::getGiven(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    QMap<int, double> given;
    query.prepare("SELECT payedBy, gross FROM receiptspay WHERE receiptNum=:id");
    query.bindValue(":id", id);
    query.exec();
    while (query.next()) {
        given.insert(query.value("payedBy").toInt(), query.value("gross").toDouble());
    }

    return given;
}

//--------------------------------------------------------------------------------

int Database::getActionTypeByName(const QString &name)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare(QString("SELECT actionId FROM actiontypes WHERE actionText=:actionText"));
    query.bindValue(":actionText", name);
    query.exec();
    query.next();

    return query.value(0).toInt();
}

//--------------------------------------------------------------------------------

QString Database::getActionType(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT actionText FROM actiontypes WHERE actionId=:id");
    query.bindValue(":id", id);
    query.exec();
    query.next();

    return query.value(0).toString();
}

//--------------------------------------------------------------------------------

QString Database::getTaxType(double id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT comment FROM taxtypes WHERE tax=:id");
    query.bindValue(":id", id);
    query.exec();
    query.next();

    QString value = query.value(0).toString();
    if (value.isEmpty() || value == "Satz-Erweitert") value = "Satz-Null";

    return value;
}

//--------------------------------------------------------------------------------

void Database::setStorno(int id, int value)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("UPDATE receipts SET storno=:value WHERE receiptNum=:receiptNum");
    query.bindValue(":value", value);
    query.bindValue(":receiptNum", id);
    bool ok = query.exec();
    if (!ok) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }
}

//--------------------------------------------------------------------------------

void Database::setStornoId(int sId, int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    // Beleg wurde von 'sId' storniert
    query.prepare("UPDATE receipts SET stornoId=:stornoId WHERE receiptNum=:receiptNum");
    query.bindValue(":stornoId", sId);
    query.bindValue(":receiptNum", id);
    bool ok = query.exec();
    if (!ok) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }

    // Beleg ist Stornobeleg von Beleg Nr: 'id'
    query.prepare("UPDATE receipts SET stornoId=:stornoId WHERE receiptNum=:receiptNum");
    query.bindValue(":stornoId", id);
    query.bindValue(":receiptNum", sId);
    ok = query.exec();
    if (!ok) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }

    setStorno(id);     // Beleg wurde storniert
    setStorno(sId, 2); // Beleg ist StornoBeleg
}

//--------------------------------------------------------------------------------

int Database::getStorno(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT storno FROM receipts WHERE receiptNum=:receiptNum");
    query.bindValue(":receiptNum", id);

    bool ok = query.exec();
    if (!ok) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }

    query.next();
    return query.value(0).toInt();
}

//--------------------------------------------------------------------------------

int Database::getStornoId(int id)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT stornoId FROM receipts WHERE receiptNum=:receiptNum");
    query.bindValue(":receiptNum", id);
    bool ok = query.exec();
    if (!ok) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
    }

    query.next();
    return query.value(0).toInt();
}

void Database::setTicketId(int id, int tktid)
{
    if (id < 1 || tktid < 1) {
        qInfo() << "Function Name: " << Q_FUNC_INFO << " id: " << id;
        qInfo() << "Function Name: " << Q_FUNC_INFO << " tktid: " << tktid;
        return;
    }

    QSqlDatabase dbc = Database::database();
    CSqlQuery updatequery(dbc, Q_FUNC_INFO);

    updatequery.prepare("UPDATE orderextras SET ticketId=:tktid WHERE orderid=:id");
    updatequery.bindValue(":tktid", id);
    updatequery.bindValue(":id", tktid);

    if (!updatequery.exec()) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << updatequery.lastError().text();
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(updatequery);
    }
}

void Database::setCashRegisterInAktive()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);

    QVariant value;
    QString strValue;
    int id = select_globals("CASHREGISTER INAKTIV", value, strValue);

    if (id > 0) {
        if (value.toInt() == 1) return;
    }

    insert2globals("CASHREGISTER INAKTIV", 1, QVariant());
}

bool Database::isCashRegisterInAktive()
{

    QVariant value;
    QString strValue;
    int id = select_globals("CASHREGISTER INAKTIV", value, strValue);

    if (id > 0)
        if (value.toInt() == 1) return true;

    return false;
}

void Database::resetAllData()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery q(dbc, Q_FUNC_INFO);

    q.prepare("DELETE FROM journal;");
    q.exec();

    q.prepare("DELETE FROM orders;");
    q.exec();

    q.prepare("DELETE FROM receipts;");
    q.exec();

    q.prepare("DELETE FROM reports;");
    q.exec();

    q.prepare("DELETE FROM dep;");
    q.exec();

    q.prepare("DELETE FROM history;");
    q.exec();

    q.prepare("DELETE FROM orderdescs;");
    q.exec();

    q.prepare("DELETE FROM orderextras;");
    q.exec();

    q.prepare("DELETE FROM receiptspay;");
    q.exec();

    q.prepare("DELETE FROM ticketorders;");
    q.exec();

    q.prepare("DELETE FROM tickets;");
    q.exec();

    q.prepare("DELETE FROM products WHERE groupid=1;");
    q.exec();

    if (dbc.tables(QSql::AllTables).contains(QLatin1String("cashbook"))) {
        q.prepare("DROP TABLE cashbook;");
        q.exec();
        Database::delete_globals("cashbook_begin");
        Database::delete_globals("cashbook_active");
    }

    Database::delete_globals("PrivateTurnoverKey");
    Database::insert2globals("lastReceiptNum", 0, QString());

    //    q.prepare("UPDATE globals SET value = 0 WHERE name =
    //    'lastReceiptNum';"); q.exec();

    Database::delete_globals("certificate");
    Database::delete_globals("DEP");

    Database::delete_globals("shopCashRegisterId");
    Database::delete_globals("signatureModuleIsDamaged");
    Database::delete_globals("CASHREGISTER INAKTIV");

    QString dbType = getDatabaseType();

    if (dbType == "QMYSQL") {
        /*MYSQL*/
        q.prepare("ALTER TABLE journal AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE orders AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE receipts AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE dep AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE orderdescs AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE ticketorders AUTO_INCREMENT = 1;");
        q.exec();

        q.prepare("ALTER TABLE tickets AUTO_INCREMENT = 1;");
        q.exec();

    } else {
        /*SQLITE*/
        q.prepare("delete from sqlite_sequence where name='journal';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='orders';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='receipts';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='dep';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='orderdescs';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='ticketorders';");
        q.exec();

        q.prepare("delete from sqlite_sequence where name='tickets';");
        q.exec();
    }

    q.exec(QString("INSERT INTO `journal`(id,version,cashregisterid,datetime,text) "
                   "VALUES (NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                   "'Id\tProgrammversion\tKassen-"
                   "Id\tKonfigurationsänderung\tBeschreibung\tErstellungsdatum')"));
    q.exec(QString("INSERT INTO `journal`(id,version,cashregisterid,datetime,text) "
                   "VALUES (NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                   "'Id\tProgrammversion\tKassen-"
                   "Id\tProduktposition\tBeschreibung\tMenge\tEinzelpreis\tGesamtpre"
                   "is\tUSt. Satz\tErstellungsdatum')"));
    q.exec(QString("INSERT INTO `journal`(id,version,cashregisterid,datetime,text) VALUES "
                   "(NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                   "'Id\tProgrammversion\tKassen-"
                   "Id\tBeleg\tBelegtyp\tBemerkung\tNachbonierung\tBelegnummer\tDatum\tUmsat"
                   "z Normal\tUmsatz Ermaessigt1\tUmsatz Ermaessigt2\tUmsatz Null\tUmsatz "
                   "Besonders\tJahresumsatz bisher\tErstellungsdatum')"));
    q.exec(QString("INSERT INTO `journal`(id,version,cashregisterid,datetime,text) "
                   "VALUES (NULL,'0.15.1222',0,CURRENT_TIMESTAMP, "
                   "'Id\tProgrammversion\tKassen-Id\tBeleg-"
                   "Textposition\tText\tErstellungsdatum')"));
}

void Database::cleanup()
{
    Database::updateGlobals("defaulttax", Q_NULLPTR, "20");
    Database::updateGlobals("CASHREGISTER INAKTIV", "0", Q_NULLPTR);
}

void Database::fixMissingProducts()
{

    QVariant value;
    QString strValue;

    Database::select_globals("fixMissingProducts", value, strValue);
    if (value.isValid() && value.toBool()) return;

    value = true;
    Database::insert2globals("fixMissingProducts", value, QDate::currentDate().toString(Qt::ISODate));

    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery checkQuery(dbc, Q_FUNC_INFO);

    checkQuery.prepare("select id, name, origin from products where origin = :id OR id = :id order by origin");
    query.exec("select product from orders group by product");

    while (query.next()) {
        int id = query.value("product").toInt();
        checkQuery.bindValue(":id", id);
        checkQuery.exec();
        if (checkQuery.next()) {
            int checkId = checkQuery.value("id").toInt();
            QJsonObject product = Database::getProductById(id, -1);
            if (product.isEmpty()) {
                qWarning() << "Function Name: " << Q_FUNC_INFO << " product info: " << checkQuery.value("name");
                qWarning() << "Function Name: " << Q_FUNC_INFO << " product not found: " << id;
                product = Database::getProductById(checkId, -1);
                product["name"] = QString("%1 ***").arg(product["name"].toString());
                product["id"] = id;
                Database::addProduct(product, id);
            }
        }
    }
    QJsonArray products = Database::findMissingProducts();
    for (auto i = products.begin(); i != products.end(); ++i) {
        QJsonObject product = (*i).toObject();
        qInfo() << "Function Name: " << Q_FUNC_INFO << " add missing product: " << product["name"].toString();
        product["visible"] = -1;
        int productId = -1;
        Database::addProduct(product, productId);
    }
}

void Database::fixDoubleProductNames()
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery updatequery(dbc, Q_FUNC_INFO);

    query.exec("update products set itemnum=TRIM(itemnum), "
               "barcode=TRIM(barcode), name=TRIM(name)");

    query.prepare("select * from products where name in (select name from "
                  "((select max(version) as version, origin from products group "
                  "by origin) p1 inner join (select * from products) as  p2 on "
                  "p1.version=p2.version and p1.origin=p2.origin) group by name "
                  "having count(name) > 1) order by name, id");
    query.exec();

    updatequery.prepare("update products set version=:version, origin=:origin where id=:id");
    int productid = 0;
    int version = 0;
    int origin = 0;
    QString name = "";
    while (query.next()) {
        if (name != query.value("name").toString()) {
            name = query.value("name").toString();
            origin = query.value("id").toInt();
            version = query.value("version").toInt();
        } else {
            productid = query.value("id").toInt();
            version++;
            updatequery.bindValue(":origin", origin);
            updatequery.bindValue(":version", version);
            updatequery.bindValue(":id", productid);
            updatequery.exec();
        }
    }
}

void Database::updatePrinters(int mode)
{
    QSqlDatabase dbc = Database::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    Settings settings;
    QJsonObject obj, printer;
    if (mode == 0) {
        obj["type"] = "custom";
        obj["pdf"] = false;
        obj["paperWidth"] = settings.value("paperWidth", 80).toInt();
        obj["paperHeight"] = settings.value("paperHeight", 210).toInt();
        obj["marginLeft"] = settings.value("marginLeft", 0).toDouble();
        obj["marginTop"] = settings.value("marginTop", 17).toDouble();
        obj["marginRight"] = settings.value("marginRight", 5).toDouble();
        obj["marginBottom"] = settings.value("marginBottom", 0).toDouble();
        QJsonDocument doc(obj);
        query.prepare("INSERT INTO printerdefs (name, definition, mode) VALUES (:name, :definition, 1)");
        query.bindValue(":name", "POS 80mm");
        query.bindValue(":definition", doc.toJson(QJsonDocument::Compact).toBase64());
        query.exec();
        query.exec("SELECT max(id) AS id FROM printerdefs");
        query.next();
        int id = query.value("id").toInt();
        printer["name"] = settings.value("receiptPrinter", "QrkPDF").toString();
        printer["definitionid"] = id;
        query.prepare("INSERT INTO printers (name, printer, definition, mode) VALUES "
                      "(:name, :printer, :definition, 1)");
        query.bindValue(":name", "Bondrucker");
        query.bindValue(":printer", QJsonDocument(QJsonArray() << printer).toJson(QJsonDocument::Compact).toBase64());
        query.bindValue(":definition", id);
        bool ok = query.exec();
        if (!ok) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
        }
        query.exec("SELECT max(id) AS id FROM printers");
        query.next();
        int printerid; // = query.value("id").toInt();

        settings.beginGroup("Printer");
        settings.save2Settings("receiptPrinter", id);
        settings.endGroup();

        printer["name"] = settings.value("collectionPrinter", "QrkPDF").toString();
        printer["definitionid"] = id;

        query.prepare("INSERT INTO printers (name, printer, definition, mode) VALUES "
                      "(:name, :printer, :definition, 1)");
        query.bindValue(":name", "Abholbondrucker");
        query.bindValue(":printer", QJsonDocument(QJsonArray() << printer).toJson(QJsonDocument::Compact).toBase64());
        query.bindValue(":definition", id);
        ok = query.exec();
        if (!ok) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
        }
        query.exec("SELECT max(id) AS id FROM printers");
        query.next();
        printerid = query.value("id").toInt();

        settings.beginGroup("Printer");
        settings.save2Settings("collectionPrinter", printerid);
        settings.endGroup();

        obj.remove("paperWidth");
        obj.remove("paperHeight");
        obj["type"] = settings.value("invoiceCompanyPaperFormat", "A4").toString();
        obj["marginLeft"] = settings.value("invoiceCompanyMarginLeft", 90).toDouble();
        obj["marginTop"] = settings.value("invoiceCompanyMarginTop", 50).toDouble();
        obj["marginRight"] = settings.value("invoiceCompanyMarginRight", 10).toDouble();
        obj["marginBottom"] = settings.value("invoiceCompanyMarginBottom", 0).toDouble();
        doc.setObject(obj);
        query.prepare("INSERT INTO printerdefs (name, definition, mode) VALUES (:name, :definition, 1)");
        query.bindValue(":name", "A4 invoice");
        query.bindValue(":definition", doc.toJson(QJsonDocument::Compact).toBase64());
        query.exec();
        query.exec("SELECT max(id) AS id FROM printerdefs");
        query.next();
        id = query.value("id").toInt();

        printer["name"] = settings.value("invoiceCompanyPrinter", "QrkPDF").toString();
        printer["definitionid"] = id;

        query.prepare("INSERT INTO printers (name, printer, definition, mode) VALUES "
                      "(:name, :printer, :definition, 1)");
        query.bindValue(":name", "Firmenrechnungsdrucker");
        query.bindValue(":printer", QJsonDocument(QJsonArray() << printer).toJson(QJsonDocument::Compact).toBase64());
        query.bindValue(":definition", id);
        ok = query.exec();
        if (!ok) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
        }
        query.exec("SELECT max(id) AS id FROM printers");
        query.next();
        printerid = query.value("id").toInt();

        settings.beginGroup("Printer");
        settings.save2Settings("invoiceCompanyPrinter", printerid);
        settings.endGroup();

        bool usePDF = settings.value("reportPrinterPDF", false).toBool();
        if (usePDF)
            obj["pdf"] = true;
        else
            obj["pdf"] = false;

        obj["type"] = settings.value("paperFormat", "A4").toString();

        obj["marginLeft"] = 5;
        obj["marginTop"] = 5;
        obj["marginRight"] = 5;
        obj["marginBottom"] = 5;
        doc.setObject(obj);
        query.prepare("INSERT INTO printerdefs (name, definition, mode) VALUES (:name, :definition, 1)");
        query.bindValue(":name", "A4");
        query.bindValue(":definition", doc.toJson(QJsonDocument::Compact).toBase64());
        query.exec();
        query.exec("SELECT max(id) AS id FROM printerdefs");
        query.next();
        id = query.value("id").toInt();

        printer["name"] = settings.value("reportPrinter", "QrkPDF").toString();
        printer["definitionid"] = id;

        query.prepare("INSERT INTO printers (name, printer, definition, mode) VALUES "
                      "(:name, :printer, :definition, :mode)");
        query.bindValue(":name", "Berichtdrucker");
        query.bindValue(":printer", QJsonDocument(QJsonArray() << printer).toJson(QJsonDocument::Compact).toBase64());
        query.bindValue(":definition", id);
        query.bindValue(":mode", 1);
        ok = query.exec();
        if (!ok) {
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << query.lastError().text();
            qWarning() << "Function Name: " << Q_FUNC_INFO << " Query: " << Database::getLastExecutedQuery(query);
        }
        query.exec("SELECT max(id) AS id FROM printers");
        query.next();
        printerid = query.value("id").toInt();

        bool noPrinter = settings.value("noPrinter").toBool();
        settings.beginGroup("Printer");
        settings.save2Settings("reportPrinter", printerid);
        settings.save2Settings("noPrinter", noPrinter);
        settings.endGroup();

        settings.removeSettings("receiptPrinter");

        settings.removeSettings("reportPrinter");
        settings.removeSettings("noPrinter");
        settings.removeSettings("numberCopies");

        settings.removeSettings("collectionPrinter");
        settings.removeSettings("collectionPrinterPaperFormat");
        settings.removeSettings("collectionReceiptCopies");

        settings.removeSettings("invoiceCompanyPrinter");
        settings.removeSettings("invoiceCompanyMarginBottom");
        settings.removeSettings("invoiceCompanyMarginLeft");
        settings.removeSettings("invoiceCompanyMarginRight");
        settings.removeSettings("invoiceCompanyMarginTop");
        settings.removeSettings("invoiceCompanyPaperFormat");

        settings.removeSettings("reportPrinterPDF");
        settings.removeSettings("paperFormat");
        settings.removeSettings("paperHeight");
        settings.removeSettings("paperWidth");
        settings.removeSettings("marginBottom");
        settings.removeSettings("marginLeft");
        settings.removeSettings("marginRight");
        settings.removeSettings("marginTop");
    }
}

bool Database::isAnyValueFunctionAvailable()
{
    QSqlDatabase dbc = Database::database();

    if (dbc.driverName() != "QMYSQL") {
        return false;
    }

    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.setShowError(false);

    if (!query.exec("SELECT ANY_VALUE(value) FROM globals")) {
        return false;
    }

    return true;
}

QString Database::getDatabaseVersion()
{

    if (globalStringValues.contains("databasetype")) return globalStringValues.value("databasetype");

    QString dbType = getDatabaseType();

    if (dbType == "QSQLITE") {
        QSqlDatabase dbc = Database::database();
        CSqlQuery query(dbc, Q_FUNC_INFO);

        query.exec("PRAGMA journal_mode;");
        query.next();
        QString mode = query.value(0).toString();

        query.exec("SELECT sqlite_version()");
        if (query.next()) dbType += " " + query.value(0).toString();

        dbType += " / " + QFileInfo(dbc.databaseName()).baseName() + " / journalmode = " + mode;
        globalStringValues.insert("databasetype", dbType);

    } else if (dbType == "QMYSQL") {
        QSqlDatabase dbc = Database::database();
        CSqlQuery query(dbc, Q_FUNC_INFO);
        query.exec("SHOW VARIABLES LIKE 'version'");
        if (query.next()) dbType += " " + query.value(1).toString();

        dbType += " / " + dbc.hostName() + " / " + dbc.databaseName();
        globalStringValues.insert("databasetype", dbType);
    }

    globalStringValues.insert("databasetype", dbType);
    return dbType;
}

void Database::encryptData(QSqlDatabase &dbc)
{
    CSqlQuery query(dbc, Q_FUNC_INFO);
    CSqlQuery updateQuery(dbc, Q_FUNC_INFO);

    dbc.transaction();
    updateQuery.prepare("UPDATE globals SET name=:name, value=:value, "
                        "strValue=:strValue WHERE id=:id");
    query.exec("SELECT * FROM globals");
    while (query.next()) {
        SecureByteArray saName = query.value("name").toByteArray();
        QString test = Crypto::decrypt(saName, SecureByteArray("Globals"), false);
        if (!saName.isEmpty() && test.isEmpty()) {
            SecureByteArray saStrValue = query.value("strValue").toByteArray();
            updateQuery.bindValue(":name", Crypto::encrypt(saName, SecureByteArray("Globals")));
            updateQuery.bindValue(":value", query.value("value"));
            if (saStrValue.isNull())
                updateQuery.bindValue(":strValue", query.value("strValue"));
            else
                updateQuery.bindValue(":strValue", Crypto::encrypt(saStrValue, SecureByteArray("Globals")));

            updateQuery.bindValue(":id", query.value("id").toInt());
            bool ok = updateQuery.exec();
            if (!ok) {
                qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << updateQuery.lastError().text();
                qWarning() << "Function Name: " << Q_FUNC_INFO
                           << " Query: " << Database::getLastExecutedQuery(updateQuery);
            }
        }
    }
    if (!dbc.commit()) {
        qWarning() << "Function Name: " << Q_FUNC_INFO << " Error: " << dbc.lastError().text();
        dbc.rollback();
    }
}
