#include "registrationtab.h"
#include "RK/rk_signaturemodulefactory.h"
#include "RK/rk_smartcardinfo.h"
#include "abstractdatabase.h"
#include "pushbutton.h"
#include "qnetworkaccessmanager.h"
#include "qnetworkreply.h"
#include "qnetworkrequest.h"
#include "qurlquery.h"
#include "uniquemachinefingerprint.h"
#include "verification.h"

#include <string>
using namespace std;
#include <cryptopp/base64.h>
#include <cryptopp/files.h>
#include <cryptopp/hex.h>
#include <cryptopp/osrng.h>
#include <cryptopp/rsa.h>

using namespace CryptoPP;

#include <QCheckBox>
#include <QDate>
#include <QDesktopServices>
#include <QFileDialog>
#include <QFormLayout>
#include <QGroupBox>
#include <QJsonDocument>
#include <QLabel>
#include <QMessageBox>
#include <QVBoxLayout>

#include <QDebug>

RegistrationTab::RegistrationTab(const QString &appname, bool showActivateMessageBox, QWidget *parent)
    : Widget(parent)
    , m_name(appname)
    , m_showActivateMessageBox(showActivateMessageBox)
{

    QGroupBox *registrationGroup = new QGroupBox(tr("Registration %1").arg(m_name));
    QHBoxLayout *registrationLayout = new QHBoxLayout;
    QFormLayout *regInfoLayout = new QFormLayout;
    m_serialnumber = new QLabel;
    m_nameLabel = new QLabel;
    m_expirydate = new QLabel;
    m_signature = new QLabel;
    m_validto = new QLabel;

    QStringList scardReaders;
    RKSmartCardInfo::getReaders(&scardReaders);

    if (scardReaders.isEmpty()) {
        m_serial = "n/a";
    } else {
        RKSignatureModule *module = RKSignatureModuleFactory::createInstance(scardReaders.at(0), false);
        m_serial = module->getCertificateSerial(true);
        delete module;
    }
    regInfoLayout->addRow(new QLabel(tr("Seriennummer")), m_serialnumber);
    regInfoLayout->addRow(new QLabel(tr("Name")), m_nameLabel);
    regInfoLayout->addRow(new QLabel(tr("Ablaufdatum")), m_expirydate);
    regInfoLayout->addRow(new QLabel(tr("Signatur")), m_signature);
    regInfoLayout->addRow(new QLabel(tr("Gültig bis")), m_validto);

    QVBoxLayout *regInputLayout = new QVBoxLayout;
    PushButton *loadLicensePushButton
        = new PushButton(QIcon(QPixmap(":/ckvsoft/resources/icons/transfer.png").scaled(32, 32, Qt::KeepAspectRatio)),
            tr("Lizenz Datei %1").arg(m_name));
    regInputLayout->addWidget(loadLicensePushButton);
    PushButton *downloadLicensePushButton
        = new PushButton(QIcon(QPixmap(":/ckvsoft/resources/icons/transfer.png").scaled(32, 32, Qt::KeepAspectRatio)),
            tr("Download Lizenz Datei"));
    regInputLayout->addWidget(downloadLicensePushButton);

    m_textlabel = new QLabel;
    regInputLayout->addWidget(m_textlabel);

    m_activateCheckBox = new QCheckBox(tr("60 Tage Version aktivieren"));
    regInputLayout->addWidget(m_activateCheckBox);
    bool active = isActive(m_name);
    m_activateCheckBox->setChecked(active);
    m_activateCheckBox->setHidden(active);
    connect(m_activateCheckBox, &QCheckBox::clicked, [this](bool checked) {
        if (checked) {
            setActive(m_name);
            m_activateCheckBox->setHidden(checked);
        }
    });

    m_webBuyPushButton = new PushButton(
        QIcon(QPixmap(":/ckvsoft/resources/icons/singleprice.png").scaled(32, 32, Qt::KeepAspectRatio)),
        tr("%1 Lizenz erwerben").arg(m_name));
    regInputLayout->addWidget(m_webBuyPushButton);

    m_webRenewPushButton = new PushButton(
        QIcon(QPixmap(":/ckvsoft/resources/icons/singleprice.png").scaled(32, 32, Qt::KeepAspectRatio)),
        tr("%1 Lizenz verlängern").arg(m_name));
    regInputLayout->addWidget(m_webRenewPushButton);
    m_webRenewPushButton->setVisible(false);

    QSpacerItem *spacer = new QSpacerItem(0, 0, QSizePolicy::Expanding, QSizePolicy::Expanding);
    regInputLayout->addItem(spacer);

    registrationLayout->addLayout(regInfoLayout);
    registrationLayout->addLayout(regInputLayout);
    registrationLayout->setStretch(0, 1);
    registrationGroup->setLayout(registrationLayout);

    QVBoxLayout *mainLayout = new QVBoxLayout;
    mainLayout->addWidget(registrationGroup);

    mainLayout->addStretch(1);
    setLayout(mainLayout);

    connect(loadLicensePushButton, &PushButton::clicked, this, &RegistrationTab::loadLicenseFromFile);
    connect(downloadLicensePushButton, &PushButton::clicked, this, &RegistrationTab::loadLicenseFromServer);
    connect(m_webBuyPushButton, &PushButton::clicked, this, &RegistrationTab::callWeb);
    connect(m_webRenewPushButton, &PushButton::clicked, this, &RegistrationTab::callWebRenew);
    //    connect(loadLicensePushButton, &PushButton::clicked, this,
    //    &RegistrationTab::loadLicense);

    m_verification = new Verification(this);
    m_license = loadLicense();
    m_demodays = getDemoDays(false);
    if (!m_license.isEmpty() || m_demodays >= 0) Verify();
}

QJsonObject RegistrationTab::loadLicense()
{
    QVariant vValue;
    QString strValue;
    int id = AbstractDataBase::select_globals(m_name, vValue, strValue);
    if (id > 0) {
        return m_verification->readJsonFromString(strValue);
    }
    return QJsonObject();
}

void RegistrationTab::saveLicense()
{
    if (m_license.isEmpty()) return;

    AbstractDataBase::delete_globals(m_name);
    QString value = "-----Begin-License-KeyFile-----\n";
    value.append(QJsonDocument(m_license).toJson());
    value.append("-----End-License-KeyFile-----\n");
    AbstractDataBase::insert2globals(m_name, QVariant(), value);
}

void RegistrationTab::loadLicenseFromFile()
{
    QString filename = QFileDialog::getOpenFileName(this, tr("Lizenz Datei öffnen"), ".",
        tr("Lizenz Dateien (%1)").arg("*.dat"), Q_NULLPTR, QFileDialog::DontUseNativeDialog);
    if (filename.isEmpty()) {
        return;
    }

    m_license = m_verification->readJsonFromFile(filename);
    Verify();
    if (m_isValid)
        saveLicense();
    else
        AbstractDataBase::delete_globals(m_name);
}

void RegistrationTab::loadLicenseFromServer()
{
    UniqueMachineFingerprint fp;

    QString baseUrl = QString("https://service.ckvsoft.at/registration/%1/getkey.php").arg(m_name.toLower());

    auto *manager = new QNetworkAccessManager(this);
    QUrlQuery postData;
    postData.addQueryItem("serial", fp.getSystemUniqueId());
    postData.addQueryItem("serialCard", m_serial);

    QByteArray postBytes = postData.toString(QUrl::FullyEncoded).toUtf8();

    // Deklariere std::function für rekursive Verwendung
    std::function<void(QUrl)> doPost;

    doPost = [=](QUrl url) mutable {
        QNetworkRequest req(url);
        req.setHeader(QNetworkRequest::ContentTypeHeader, "application/x-www-form-urlencoded");

        QNetworkReply *reply = manager->post(req, postBytes);

        connect(reply, &QNetworkReply::finished, this, [=]() mutable {
            QVariant redirectAttr = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);
            if (redirectAttr.isValid()) {
                QUrl newUrl = reply->url().resolved(redirectAttr.toUrl());
                reply->deleteLater();
                doPost(newUrl); // rekursiver Aufruf
                return;
            }

            if (reply->error() != QNetworkReply::NoError) {
                QMessageBox::warning(this, tr("Fehler"), reply->errorString());
                reply->deleteLater();
                return;
            }

            QByteArray response = reply->readAll();
            reply->deleteLater();

            QJsonParseError parseError;
            QJsonDocument jsonDoc = QJsonDocument::fromJson(response, &parseError);

            // Prüfe auf JSON-Fehler
            if (parseError.error != QJsonParseError::NoError) {
                QMessageBox::critical(this, tr("Fehler"), tr("Antwort konnte nicht gelesen werden."));
                return;
            }

            // Prüfe, ob eine "error"-Nachricht vom Server kam
            if (jsonDoc.isObject() && jsonDoc.object().contains("error")) {
                QString errorMsg = jsonDoc.object().value("error").toString();
                QMessageBox::critical(this, tr("Fehler"), errorMsg);
                return;
            }

            if (!response.contains("-----Begin-License-KeyFile-----")) {
                QMessageBox::critical(this, tr("Fehler"), tr("Keine gültige Lizenz empfangen."));
                return;
            }

            m_license = m_verification->readJsonFromString(response);
            Verify();
            if (m_isValid)
                saveLicense();
            else
                AbstractDataBase::delete_globals(m_name);

            QMessageBox::information(this, tr("Erfolg"), tr("Lizenz erfolgreich gespeichert."));
        });
    };

    // Initialer Aufruf
    doPost(QUrl(baseUrl));
}

void RegistrationTab::Verify()
{
    UniqueMachineFingerprint fp;

    // Step 1: Check the serial number
    if (fp.validate(m_license["SerialNumber"].toString())) {
        m_serialnumber->setPixmap(QPixmap(":/ckvsoft/resources/icons/ok.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_isValid = true;
    } else if (m_serial.compare(m_license["SerialNumberScard"].toString()) == 0) {
        m_serialnumber->setPixmap(QPixmap(":/ckvsoft/resources/icons/ok.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_isValid = true;
    } else {
        m_serialnumber->setPixmap(QPixmap(":/ckvsoft/resources/icons/cancel.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_license["SerialNumber"] = fp.getSystemUniqueId();
        m_isValid = false; // Serial number is invalid
    }

    // Step 2: Check the product name
    if (m_license["Product"].toString() == m_name) {
        m_nameLabel->setPixmap(QPixmap(":/ckvsoft/resources/icons/ok.png").scaled(24, 24, Qt::KeepAspectRatio));
    } else {
        m_nameLabel->setPixmap(QPixmap(":/ckvsoft/resources/icons/cancel.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_license["Product"] = m_name;
        m_isValid = false; // Product is not valid
    }

    // Step 3: Check the expiry date
    QString validTill = m_license["ValidTill"].toString();
    m_validto->setText(QLocale().toString(QDate::fromString(validTill, "yyyy-MM-dd"), QLocale::ShortFormat));

    bool isExpired = (validTill < QDate::currentDate().toString(Qt::ISODate));
    if (!isExpired) {
        m_expirydate->setPixmap(QPixmap(":/ckvsoft/resources/icons/ok.png").scaled(24, 24, Qt::KeepAspectRatio));
    } else {
        m_expirydate->setPixmap(QPixmap(":/ckvsoft/resources/icons/cancel.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_license["ValidTill"] = "0"; // Set to invalid
        m_isValid = false;            // License has expired
    }

    // Step 4: Check the signature
    string signedTxt = m_verification->getSignedText(m_license).toStdString();
    // Read the public key
    CryptoPP::ByteQueue bytes;
    StringSource s(
        m_verification->readKeyFromFile(":/ckvsoft/resources/keys/pubkey.txt").toStdString(), true, new Base64Decoder);
    s.TransferTo(bytes);
    bytes.MessageEnd();

    RSA::PublicKey pubKey;
    pubKey.Load(bytes);
    RSASSA_PKCS1v15_SHA_Verifier verifier(pubKey);

    // Read the signed message
    string sig;
    StringSource(m_license["Signature"].toString().toStdString(), true, new HexDecoder(new StringSink(sig)));

    // Verify the signature
    string combined(signedTxt);
    combined.append(sig);

    try {
        StringSource(combined, true,
            new SignatureVerificationFilter(verifier, Q_NULLPTR, SignatureVerificationFilter::THROW_EXCEPTION));
        m_signature->setPixmap(QPixmap(":/ckvsoft/resources/icons/ok.png").scaled(24, 24, Qt::KeepAspectRatio));
    } catch (SignatureVerificationFilter::SignatureVerificationFailed &err) {
        Q_UNUSED(err)
        m_signature->setPixmap(QPixmap(":/ckvsoft/resources/icons/cancel.png").scaled(24, 24, Qt::KeepAspectRatio));
        m_isValid = false; // Signature verification failed
    }

    // Step 5: Set button visibility based on checks
    if (m_isValid) {
        m_textlabel->setText("");
        m_webBuyPushButton->setVisible(false);  // Disable Buy button
        m_webRenewPushButton->setVisible(true); // Enable Renew button
    } else {
        // Check if ValidTill was previously set and is not empty
        if (!validTill.isEmpty() && validTill != "0") {
            m_webRenewPushButton->setVisible(true); // Enable Renew button
            m_webBuyPushButton->setVisible(false);  // Disable Buy button
        } else {
            m_webRenewPushButton->setVisible(false); // Disable Renew button
            m_webBuyPushButton->setVisible(true);    // Enable Buy button
        }

        // In demo mode, show remaining days
        if (m_demodays >= 0) {
            m_validto->setText(QLocale().toString(QDate::currentDate().addDays(m_demodays), QLocale::ShortFormat));
            m_textlabel->setText(tr("Trial version active\n(Still %1 days valid)").arg(m_demodays));
            m_webBuyPushButton->setVisible(true);    // Enable Buy button
            m_webRenewPushButton->setVisible(false); // Disable Renew button
        } else {
            m_textlabel->setText("");
            m_webBuyPushButton->setVisible(true);    // Enable Buy button
            m_webRenewPushButton->setVisible(false); // Disable Renew button
        }
    }
}

int RegistrationTab::getDemoDays(int valid)
{
    if (valid) return -1;

    QString strValue;
    bool active = isActive(m_name, strValue);
    if (!active) {
        activateMessageBox();
        isActive(m_name, strValue);
    }

    if (strValue.isEmpty()) return -1;

    QDate date = QDate::fromString(strValue);
    if (date.isValid()) {
        qint64 days = 60 - date.daysTo(QDate::currentDate());
        if (days > 0) return int(days);
    }

    return -1;
}

bool RegistrationTab::isValid(int &days)
{
    days = getDemoDays(m_isValid);
    return m_isValid || days >= 0;
}

bool RegistrationTab::isActive(const QString &name)
{
    QString strValue;
    return isActive(name, strValue);
}

bool RegistrationTab::isActive(const QString &name, QString &strValue)
{
    QVariant value;
    int id = AbstractDataBase::select_globals(name, value, strValue);
    if (id < 1) id = AbstractDataBase::select_globals(name + "DEMO", value, strValue);
    return id > 0;
}

void RegistrationTab::setActive(const QString &name)
{
    QString strValue;
    setActive(name, strValue);
}

void RegistrationTab::setActive(const QString &name, QString &strValue)
{
    strValue = QDate::currentDate().toString();
    AbstractDataBase::insert2globals(name + "DEMO", QVariant(), strValue);
}

void RegistrationTab::setAlternateWebJsonData(QJsonObject jsonObjectData)
{
    UniqueMachineFingerprint fp;
    jsonObjectData["name"] = m_name;
    jsonObjectData["serial"] = fp.getSystemUniqueId();
    jsonObjectData["serialCard"] = m_serial;
    m_alternatewebjsondata = QJsonDocument(jsonObjectData).toJson(QJsonDocument::Compact);
}

void RegistrationTab::callWebRenew()
{
    callWeb(true);
}

void RegistrationTab::callWeb(bool renew)
{
    UniqueMachineFingerprint fp;
    QString json;

    // Check if `m_alternatewebjsondata` is empty
    if (m_alternatewebjsondata.isEmpty()) {
        // Create a new JSON object
        QJsonObject jsonObject;
        jsonObject["name"] = m_name;
        jsonObject["serial"] = fp.getSystemUniqueId();
        jsonObject["renew"] = renew;

        // Convert the JSON object to a compact JSON string
        json = QJsonDocument(jsonObject)
                   .toJson(QJsonDocument::Compact)
                   .toBase64(QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
    } else {
        // Parse the existing JSON document from `m_alternatewebjsondata`
        QJsonDocument jsonDoc = QJsonDocument::fromJson(m_alternatewebjsondata.toUtf8());

        // Extract the JSON object from the document
        QJsonObject jsonObject = jsonDoc.object();

        // Add or overwrite the "renew" value
        jsonObject["renew"] = renew;

        // Create a new JSON document and convert it to a compact JSON string
        json = QJsonDocument(jsonObject)
                   .toJson(QJsonDocument::Compact)
                   .toBase64(QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
    }

    // Create the full URL link
    QString link = "https://service.ckvsoft.at/registration/qrk/";
    QUrl url(link);
    url.setQuery(QString("id=%1").arg(json));

    // Open the URL in the default browser
    QDesktopServices::openUrl(url);
}

void RegistrationTab::activateMessageBox()
{
    if (m_showActivateMessageBox) {
        QMessageBox msgBox;
        msgBox.setWindowTitle(tr("60 Tage Testversion"));
        msgBox.setText(tr("Möchten Sie \"%1\" für 60 Tage testen?").arg(m_name));
        msgBox.setStandardButtons(QMessageBox::Yes);
        msgBox.addButton(QMessageBox::No);
        msgBox.setDefaultButton(QMessageBox::No);
        if (msgBox.exec() == QMessageBox::Yes) {
            setActive(m_name);
        }
    }
}

void RegistrationTab::setShowActivateMessageBox(bool testing)
{
    m_showActivateMessageBox = testing;
}
