/*
 * This file is part of QRK - Qt Registrier Kasse
 *
 * Copyright (C) 2015-2024 Christian Kvasny <chris@ckvsoft.at>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, see <http://www.gnu.org/licenses/>.
 *
 * Button Design, and Idea for the Layout are lean out from LillePOS, Copyright
 * 2010, Martin Koller, kollix@aon.at
 *
 */

#include "rk_signaturemodule.h"
#include "abstractdatabase.h"
#include "base32decode.h"
#include "base32encode.h"
#include "csqlquery.h"

#include <stdio.h>

#include <cryptopp/aes.h>
#include <cryptopp/hex.h>
#include <cryptopp/modes.h>
#include <cryptopp/osrng.h>
#include <cryptopp/rsa.h>

#include <QDateTime>
#include <QDebug>
#include <QSqlError>
#include <QtEndian>

using namespace std;
using namespace CryptoPP;

bool SignatureModuleSetDamaged = false;

/**
 * @brief RKSignatureModule::RKSignatureModule
 */
RKSignatureModule::RKSignatureModule()
{
    m_certificateserial = "";
}

/**
 * @brief RKSignatureModule::~RKSignatureModule
 */
RKSignatureModule::~RKSignatureModule()
{
}

QString RKSignatureModule::parseExpiryDate(const QString &expiryDate, const QString &cardType)
{
    if (expiryDate.isEmpty()) {
        return "";
    }

    QDate currentDate = QDate::currentDate();
    QDate expiry;

    QStringList parts = expiryDate.split(' ');
    if (parts.size() == 3) { // Format: "Tag Monat Jahr"
        expiry = QDate(parts[2].toInt(), QDate::fromString(parts[1], "MMMM").month(), parts[0].toInt());
    } else if (parts.size() == 2) { // Format: "Monat Jahr"
        expiry = QDate(parts[1].toInt(), QDate::fromString(parts[0], "MMMM").month(), 1);
    } else {
        return "";
    }

    if (!expiry.isValid()) {
        return "";
    }

    if (expiry < currentDate) {
        return QObject::tr(
            "%1 ist bereits abgelaufen und kann nicht mehr für eine RKSV-konforme Signierung verwendet werden.")
            .arg(cardType);
    }

    if (expiry <= currentDate.addMonths(3)) {
        return QObject::tr("%1 läuft bald ab (%2) und sollte rechtzeitig ersetzt werden.")
            .arg(cardType)
            .arg(expiryDate);
    }

    return "";
}

/**
 * @brief RKSignatureModule::getDataToBeSigned
 * @param data
 * @return
 */
QString RKSignatureModule::getDataToBeSigned(QString data)
{
    QString JWS_Protected_Header = base64Url_encode("{\"alg\":\"ES256\"}");
    QString JWS_Payload = base64Url_encode(data);

    return JWS_Protected_Header + "." + JWS_Payload;
}

/**
 * @brief RKSignatureModule::HashValue
 * @param value
 * @return
 */
QByteArray RKSignatureModule::HashValue(QString value)
{
    SHA256 hashtest;
    string message = value.toStdString();
    string digest;

    StringSource s(message, true, new HashFilter(hashtest, new HexEncoder(new StringSink(digest))));
    return QByteArray::fromStdString(digest);
}

/**
 * @brief RKSignatureModule::RawHashValue
 * @param value
 * @return
 */
QByteArray RKSignatureModule::RawHashValue(QString value)
{
    CryptoPP::SHA256 hasher;
    std::string message = value.toStdString(); // Achtung bei Umlauten!
    CryptoPP::SecByteBlock digest(CryptoPP::SHA256::DIGESTSIZE);

    hasher.CalculateDigest(digest, reinterpret_cast<const byte *>(message.data()), message.size());

    return QByteArray(reinterpret_cast<const char *>(digest.data()), digest.size()); // 32 Bytes binär
}

/**
 * @brief RKSignatureModule::encryptTurnoverCounter
 * @param concatenated
 * @param turnoverCounter
 * @param symmetricKey
 * @return
 */
QString RKSignatureModule::encryptTurnoverCounter(QString concatenated, qlonglong turnoverCounter, QString symmetricKey)
{
    QString hashValue = HashValue(concatenated);
    return encryptCTR(hashValue.toStdString(), turnoverCounter, symmetricKey.toStdString());
}

/**
 * @brief RKSignatureModule::decryptTurnoverCounter
 * @param concatenated
 * @param encodedTurnoverCounter
 * @param symmetricKey
 * @return
 */
QString RKSignatureModule::decryptTurnoverCounter(
    QString concatenated, QString encodedTurnoverCounter, QString symmetricKey)
{
    QString hashValue = HashValue(concatenated);
    return decryptCTR(hashValue.toStdString(), encodedTurnoverCounter, symmetricKey.toStdString());
}

/**
 * @brief RKSignatureModule::getLastSignatureValue
 * @param sig
 * @return
 */
QString RKSignatureModule::getLastSignatureValue(QString sig)
{
    QString hashValue = HashValue(sig);
    QByteArray ls;
    ls.append(hashValue.toUtf8());
    ls = QByteArray::fromHex(ls);
    ls.resize(8);

    return ls.toBase64();
}

/**
 * @brief RKSignatureModule::base64_encode
 * @param str
 * @return
 */
QByteArray RKSignatureModule::base64_encode(QString str)
{
    QByteArray ba = 0;
    ba.append(str.toUtf8());

    return ba.toBase64();
}

/**
 * @brief RKSignatureModule::base64Url_encode
 * @param str
 * @return
 */
QByteArray RKSignatureModule::base64Url_encode(QString str)
{
    QByteArray ba = 0;
    ba.append(str.toUtf8());

    return ba.toBase64(QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
}

/**
 * @brief RKSignatureModule::base64_decode
 * @param str
 * @param hex
 * @return
 */
QByteArray RKSignatureModule::base64_decode(QString str, bool hex)
{
    QByteArray ba = 0;
    ba.clear();
    ba.append(str.toUtf8());
    if (hex) return QByteArray::fromBase64(ba).toHex();

    return QByteArray::fromBase64(ba);
}

/**
 * @brief RKSignatureModule::base64Url_decode
 * @param QString base64URL_str to decode
 * @return
 */
QByteArray RKSignatureModule::base64Url_decode(QString str)
{
    return QByteArray::fromBase64(str.toUtf8(), QByteArray::Base64UrlEncoding | QByteArray::OmitTrailingEquals);
}

/**
 * @brief RKSignatureModule::base32_encode
 * @param QString str to encode
 * @return
 */
QByteArray RKSignatureModule::base32_encode(QByteArray str)
{
    string decoded = str.toStdString();

    size_t req = Base32Encode::GetLength(decoded.size());
    char encoded[req];

    memset(encoded, 0x00, req);
    size_t size = Base32Encode::Encode(encoded, decoded.data(), decoded.size());

    return QByteArray::fromStdString(std::string(encoded, size)).simplified();
}

/**
 * @brief RKSignatureModule::base32_decode
 * @param QString base64_str to decode
 * @return
 */
QByteArray RKSignatureModule::base32_decode(QByteArray str)
{

    string encoded = str.toStdString();

    size_t req = Base32Decode::GetLength(encoded.size());
    char decoded[req];

    memset(decoded, 0x00, req);
    size_t size = Base32Decode::Decode(decoded, encoded.data(), encoded.size());

    return QByteArray::fromStdString(std::string(decoded, size));
}

/**
 * @brief RKSignatureModule::encryptCTR
 * @param string concatenatedHashValue
 * @param qlonglong turnoverCounter
 * @param string symmetricKey
 * @return
 */
QString RKSignatureModule::encryptCTR(
    std::string concatenatedHashValue, qlonglong turnoverCounter, std::string symmetricKey)
{

    const int RKSuitSize = 8;
    union {
        qlonglong source;
        byte data[RKSuitSize];
    } Union;

    memset(Union.data, 0x00, RKSuitSize);
    Union.source = qToBigEndian(turnoverCounter);

    byte key[AES::MAX_KEYLENGTH], iv[AES::BLOCKSIZE];
    StringSource ssk(symmetricKey, true, new HexDecoder(new ArraySink(key, AES::MAX_KEYLENGTH)));
    StringSource ssv(concatenatedHashValue, true, new HexDecoder(new ArraySink(iv, AES::BLOCKSIZE)));

    byte out[RKSuitSize];
    memset(out, 0x00, RKSuitSize);

    try {
        CTR_Mode<AES>::Encryption encryption(key, sizeof(key), iv, sizeof(iv));
        encryption.ProcessData(out, Union.data, sizeof(Union.data));
    } catch (Exception &e) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << e.what();
    }

    QByteArray ba = 0;
    ba.append(reinterpret_cast<const char *>(out), sizeof(out));

    return ba.toBase64();
}

/**
 * @brief RKSignatureModule::decryptCTR
 * @param string concatenatedHashValue
 * @param QString encryptedTurnoverCounter
 * @param string symmetricKey
 * @return
 */
QString RKSignatureModule::decryptCTR(
    std::string concatenatedHashValue, QString encryptedTurnoverCounter, std::string symmetricKey)
{
    const int RKSuitSize = 8;
    QByteArray baBase64 = 0;
    baBase64.append(encryptedTurnoverCounter.toUtf8());
    string encryptedTurnoverCounterHex = QByteArray::fromBase64(baBase64).toHex().toStdString();

    byte data[AES::BLOCKSIZE];
    memset(data, 0x00, AES::BLOCKSIZE);

    byte recovered[RKSuitSize];
    memset(recovered, 0x00, RKSuitSize);

    byte key[AES::MAX_KEYLENGTH], iv[AES::BLOCKSIZE];
    StringSource ssk(symmetricKey, true, new HexDecoder(new ArraySink(key, AES::MAX_KEYLENGTH)));
    StringSource ssv(concatenatedHashValue, true, new HexDecoder(new ArraySink(iv, AES::BLOCKSIZE)));
    StringSource ssd(encryptedTurnoverCounterHex, true, new HexDecoder(new ArraySink(data, AES::BLOCKSIZE)));

    try {
        CTR_Mode<AES>::Decryption decryption(key, sizeof(key), iv, sizeof(iv));
        decryption.ProcessData(recovered, data, AES::BLOCKSIZE);
    } catch (Exception &e) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << " Error: " << e.what();
    }

    /* we need only 8 Bytes for Turnover minimum is 5 Byte */
    union {
        qlonglong source;
        byte data[RKSuitSize];
    } Union;
    memset(Union.data, 0x00, RKSuitSize);
    /* reconstruct reverse */
    for (int i = 0; i < 8; i++)
        Union.data[i] = recovered[7 - i];

    return QString::number(Union.source);
}

/**
 * @brief RKSignatureModule::getPrivateTurnoverKey
 * @return QString PrivateTurnoverKey (AES Key)
 */
QString RKSignatureModule::getPrivateTurnoverKey()
{

    QVariant vValue;
    QString strValue;

    int id = AbstractDataBase::select_globals("PrivateTurnoverKey", vValue, strValue);
    if (id > 0) {
        int val = vValue.toInt();
        /* increment manual for keyversions check */
        if (val == 1) return strValue;
    }
    QString key = RKSignatureModule::generatePrivateTurnoverHexKey();
    AbstractDataBase::insert2globals("PrivateTurnoverKey", 1, key);

    return key;
}

/**
 * @brief RKSignatureModule::getPrivateTurnoverKeyBase64
 * @return QString AES Key Base64 encoded
 */
QString RKSignatureModule::getPrivateTurnoverKeyBase64()
{
    QString symmetricKey = RKSignatureModule::getPrivateTurnoverKey().toLower();
    QByteArray ba(QByteArray::fromHex(symmetricKey.toUtf8()));
    return ba.toBase64();
}

QString RKSignatureModule::getPrivateTurnoverKeyCheckValueBase64Trimmed()
{
    QString symmetricKeyBase64 = getPrivateTurnoverKeyBase64();
    QByteArray data = QByteArray::fromHex(HashValue(symmetricKeyBase64));
    data.resize(3);
    return data.toBase64().replace("=", "");
}

QJsonObject RKSignatureModule::getCertificateMap()
{
    QJsonObject certificateMap;
    QSqlDatabase dbc = AbstractDataBase::database();

    CSqlQuery query(dbc, Q_FUNC_INFO);
    if (query.exec("SELECT serial, cert FROM certs")) {
        while (query.next()) {
            QString serialHex = query.value("serial").toString().toUpper();
            QString certB64 = query.value("cert").toString();

            QJsonObject certificate;
            certificate["id"] = serialHex;
            certificate["signatureDeviceType"] = "CERTIFICATE";
            certificate["signatureCertificateOrPublicKey"] = certB64;

            certificateMap[serialHex] = certificate;
        }
    } else {
        qWarning() << "Function Name: " << Q_FUNC_INFO << "query:" << AbstractDataBase::getLastExecutedQuery(query);
        qWarning() << "Function Name: " << Q_FUNC_INFO << "DB error:" << query.lastError().text();
    }

    return certificateMap;
}

/**
 * @brief RKSignatureModule::isCertificateInDB
 * @param serialHex
 * @return
 */
bool RKSignatureModule::isCertificateInDB(const QString &serialHex)
{
    if (serialHex.isEmpty()) return false;

    QSqlDatabase dbc = AbstractDataBase::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("SELECT id FROM certs WHERE serial = :serial");
    query.bindValue(":serial", serialHex.toUpper());

    if (query.exec() && query.next()) {
        return true;
    }

    return false;
}

void RKSignatureModule::putCertificate(const QString &serialHex, const QString &certificateB64)
{
    if (serialHex.isEmpty()) return;

    if (isCertificateInDB(serialHex)) {
        return;
    }

    QSqlDatabase dbc = AbstractDataBase::database();
    CSqlQuery query(dbc, Q_FUNC_INFO);

    query.prepare("INSERT INTO certs (serial, cert, created) "
                  "VALUES (:serial, :cert, datetime('now'))");

    query.bindValue(":serial", serialHex.toUpper());
    query.bindValue(":cert", certificateB64);

    if (!query.exec()) {
        qCritical() << "Function Name: " << Q_FUNC_INFO << "query:" << AbstractDataBase::getLastExecutedQuery(query);
        qCritical() << "Function Name: " << Q_FUNC_INFO << "DB insert error:" << query.lastError().text();
    }
}

void RKSignatureModule::setSignatureModuleDamaged()
{
    AbstractDataBase::insert2globals(
        "signatureModuleIsDamaged", QVariant(), QDateTime::currentDateTime().toString(Qt::ISODate));
    SignatureModuleSetDamaged = true;
}

bool RKSignatureModule::isSignatureModuleSetDamaged()
{
    QVariant vValue;
    QString strValue;
    int id = AbstractDataBase::select_globals("signatureModuleIsDamaged", vValue, strValue);
    if (id > 0) return true;

    return SignatureModuleSetDamaged;
}

QString RKSignatureModule::resetSignatureModuleDamaged()
{
    QString ISODate = "";

    QVariant vValue;
    QString strValue;
    int id = AbstractDataBase::select_globals("signatureModuleIsDamaged", vValue, strValue);
    if (id > 0) ISODate = strValue;

    AbstractDataBase::delete_globals("signatureModuleIsDamaged");

    SignatureModuleSetDamaged = false;
    return ISODate;
}

/**
 * @brief RKSignatureModule::generatePrivateTurnoverHexKey
 * @return AES Key in Hex
 */
QString RKSignatureModule::generatePrivateTurnoverHexKey()
{

    AutoSeededRandomPool prng;

    /* generate key for encrypt */
    unsigned char key[AES::MAX_KEYLENGTH];
    prng.GenerateBlock(key, sizeof(key));

    string encoded;
    encoded.clear();

    StringSource ssk(key, sizeof(key), true, new HexEncoder(new StringSink(encoded)));

    return QString::fromStdString(encoded).toLower();
}

/**
 * @brief RKSignatureModule::isDEPactive
 * @return bool
 */
bool RKSignatureModule::isDEPactive()
{
    QVariant vValue;
    QString strValue;
    int id = AbstractDataBase::select_globals("DEP", vValue, strValue);
    if (id > 0) {
        bool active = vValue.toBool();
        return active;
    }

    return false;
}

/**
 * @brief RKSignatureModule::setDEPactive
 * @param bool active
 */
void RKSignatureModule::setDEPactive(bool active)
{
    AbstractDataBase::insert2globals("DEP", active, QString());
}

/**
 * @brief RKSignatureModule::getPinForSerial
 * @param serial
 * @return
 */
QByteArray RKSignatureModule::getPinForSerial(const QString &serial)
{
    QSqlDatabase dbc = AbstractDataBase::database("RK");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("SELECT pin FROM certs WHERE serial = :serial");
    query.bindValue(":serial", serial);

    if (query.exec() && query.next()) {
        return query.value(0).toByteArray();
    }

    return {};
}

bool RKSignatureModule::storePinForSerial(const QString &serial, const QByteArray &pin)
{
    QSqlDatabase dbc = AbstractDataBase::database("PIN");
    CSqlQuery query(dbc, Q_FUNC_INFO);
    query.prepare("UPDATE certs SET pin = :pin WHERE serial = :serial");
    query.bindValue(":pin", pin);
    query.bindValue(":serial", serial);

    if (!query.exec()) {
        qWarning() << "Failed to update certs:" << query.lastError().text();
        return false;
    }

    if (query.numRowsAffected() == 0) {
        qWarning() << "No cert entry found for serial" << serial;
        return false;
    }

    return true;
}
