#include "loguploader.h"
#include "quazip.h"
#include "quazipfile.h"

#include <QFile>
#include <QFileInfo>
#include <QHttpMultiPart>
#include <QHttpPart>
#include <QStandardPaths>
#include <QTextStream>

LogUploader::LogUploader(QObject *parent)
    : QObject(parent)
{
    connect(&networkManager, &QNetworkAccessManager::finished, this, &LogUploader::onUploadFinished);
}

QString LogUploader::highlightLogFileAsString(const QString &logFilePath)
{
    QFile file(logFilePath);
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) return QString();

    QTextStream in(&file);
    QString html;
    html += "<html><body style='font-family: monospace; white-space: pre-wrap;'>\n";

    while (!in.atEnd()) {
        QString line = in.readLine();
        QString color = "black";

        if (line.contains("ERROR", Qt::CaseInsensitive))
            color = "red";
        else if (line.contains("WARNING", Qt::CaseInsensitive))
            color = "orange";
        else if (line.contains("CRITICAL", Qt::CaseInsensitive))
            color = "purple";

        html += QString("<span style='color:%1;'>%2</span><br>\n").arg(color, line.toHtmlEscaped());
    }

    html += "</body></html>";
    return html;
}

void LogUploader::uploadLog(const QString &logFilePath, const QString &deviceId, const QUrl &serverUrl)
{
    QString htmlContent = highlightLogFileAsString(logFilePath);

    // Speichere HTML lokal als Datei für Anhang
    QString htmlPath = QStandardPaths::writableLocation(QStandardPaths::TempLocation) + "/log.html";
    QFile outFile(htmlPath);
    if (outFile.open(QIODevice::WriteOnly | QIODevice::Text)) {
        QTextStream out(&outFile);
        out.setCodec("UTF-8");
        out << htmlContent;
        outFile.close();
    }

    QHttpMultiPart *multiPart = new QHttpMultiPart(QHttpMultiPart::FormDataType);

    // Gerät-ID
    QHttpPart idPart;
    idPart.setHeader(QNetworkRequest::ContentDispositionHeader, QVariant("form-data; name=\"device_id\""));
    idPart.setBody(deviceId.toUtf8());
    multiPart->append(idPart);

    // HTML-Inhalt im Body
    QHttpPart bodyPart;
    bodyPart.setHeader(QNetworkRequest::ContentDispositionHeader, QVariant("form-data; name=\"body\""));
    bodyPart.setHeader(QNetworkRequest::ContentTypeHeader, QVariant("text/html; charset=utf-8"));
    bodyPart.setBody(htmlContent.toUtf8());
    multiPart->append(bodyPart);

    // HTML-Datei als Anhang
    QFile *file = new QFile(htmlPath);
    if (file->open(QIODevice::ReadOnly)) {
        QHttpPart filePart;
        filePart.setHeader(
            QNetworkRequest::ContentDispositionHeader, QVariant("form-data; name=\"logfile\"; filename=\"log.html\""));
        filePart.setHeader(QNetworkRequest::ContentTypeHeader, QVariant("text/html"));
        filePart.setBodyDevice(file);
        file->setParent(multiPart);
        multiPart->append(filePart);
    }

    QNetworkRequest request(serverUrl);
    QNetworkReply *reply = networkManager.post(request, multiPart);

    connect(reply, &QNetworkReply::uploadProgress, this, &LogUploader::uploadProgress);
}

void LogUploader::onUploadFinished(QNetworkReply *reply)
{
    bool success = (reply->error() == QNetworkReply::NoError);
    QString msg = success ? "Upload erfolgreich" : "Fehler beim Senden: " + reply->errorString();
    emit uploadFinished(success, msg);
    reply->deleteLater();
}

QString LogUploader::zipLogFile(const QString &logPath)
{
    QString zipPath = QStandardPaths::writableLocation(QStandardPaths::TempLocation) + "/log.zip";

    QuaZip zip(zipPath);
    if (!zip.open(QuaZip::mdCreate)) {
        qWarning() << Q_FUNC_INFO << "Kann Zip nicht erstellen";
        return QString();
    }

    QFile logFile(logPath);
    if (!logFile.open(QIODevice::ReadOnly)) {
        qWarning() << Q_FUNC_INFO << "Kann Logdatei nicht öffnen:" << logPath;
        return QString();
    }

    QFileInfo fi(logPath);
    QString filenameInZip = fi.fileName(); // z.B. "app.log"

    QuaZipFile zipFile(&zip);
    if (!zipFile.open(QIODevice::WriteOnly, QuaZipNewInfo(filenameInZip))) {
        qWarning() << Q_FUNC_INFO << "Kann" << filenameInZip << "in Zip nicht schreiben";
        return QString();
    }

    zipFile.write(logFile.readAll());
    zipFile.close();
    logFile.close();

    zip.close();

    qDebug() << Q_FUNC_INFO << "Zip erstellt:" << zipPath;
    return zipPath;
}
